package jp.groupsession.v2.cht.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import jp.co.sjts.util.CloneableUtil;
import jp.co.sjts.util.NullDefault;
import jp.co.sjts.util.PageUtil;
import jp.co.sjts.util.StringUtilHtml;
import jp.co.sjts.util.dao.AbstractDao;
import jp.co.sjts.util.date.UDate;
import jp.co.sjts.util.date.UDateUtil;
import jp.co.sjts.util.jdbc.JDBCUtil;
import jp.co.sjts.util.jdbc.SqlBuffer;
import jp.groupsession.v2.cht.GSConstChat;
import jp.groupsession.v2.cht.biz.ChtMentionBiz;
import jp.groupsession.v2.cht.biz.ChtReactionBiz;
import jp.groupsession.v2.cht.model.ChatDeleteModel;
import jp.groupsession.v2.cht.model.ChatGroupInfModel;
import jp.groupsession.v2.cht.model.ChatInformationModel;
import jp.groupsession.v2.cht.model.ChatMessageModel;
import jp.groupsession.v2.cht.model.ChatMidokuModel;
import jp.groupsession.v2.cht.model.ChatSearchModel;
import jp.groupsession.v2.cht.model.ChatUserInfModel;
import jp.groupsession.v2.cht.model.ChtGroupDataPinModel;
import jp.groupsession.v2.cht.model.ChtGroupUconfModel;
import jp.groupsession.v2.cht.model.ChtGroupUserModel;
import jp.groupsession.v2.cht.model.ChtReactionModel;
import jp.groupsession.v2.cht.model.ChtUserDataPinModel;
import jp.groupsession.v2.cht.model.ChtUserUconfModel;
import jp.groupsession.v2.cht.search.ChatMessageSearchFilter;
import jp.groupsession.v2.cht.search.ChatMessageSearchInfo;
import jp.groupsession.v2.cht.search.ChatMessageSearchRequest;
import jp.groupsession.v2.cht.search.position.AbstractMessagePosition;
import jp.groupsession.v2.cht.search.position.AfterMessage;
import jp.groupsession.v2.cht.search.position.BeforeMessage;
import jp.groupsession.v2.cht.search.position.MidokuPosition;
import jp.groupsession.v2.cht.search.position.NewestPosition;
import jp.groupsession.v2.cht.search.position.OffsetPosition;
import jp.groupsession.v2.cmn.GSConst;
import jp.groupsession.v2.cmn.dao.base.CmnBelongmDao;
import jp.groupsession.v2.cmn.dao.base.CmnUsrmDao;
import jp.groupsession.v2.cmn.model.CmnUserModel;
import jp.groupsession.v2.cmn.model.base.CmnBinfModel;
import jp.groupsession.v2.cmn.model.base.CmnGroupmModel;
import jp.groupsession.v2.cmn.model.base.CmnUsrmInfModel;
import jp.groupsession.v2.usr.GSConstUser;
import jp.groupsession.v2.usr.UserUtil;

/**
 *
 * <br>[機  能]チャットDAOクラス
 * <br>[解  説]
 * <br>[備  考]
 *
 * @author JTS
 */
public class ChatDao extends AbstractDao {

    /** Logging インスタンス */
    private static Log log__ = LogFactory.getLog(ChatDao.class);

    /**
     * <p>Default Constructor
     */
    public ChatDao() {
    }

    /**
     * <p>Set Connection
     * @param con Connection
     */
    public ChatDao(Connection con) {
        super(con);
    }
    /**
     * お気に入り登録されているユーザ件数
     * @param usrSid ユーザSID
     * @return List in CHT_FAVORITEModel
     * @throws SQLException SQL実行例外
     */
    public int favoriteUserCount(int usrSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql("  select ");
            sql.addSql("   count(CUP_SID) as CNT");
            sql.addSql("  from ");
            sql.addSql("   CHT_USER_UCONF");
            sql.addSql("  where ");
            sql.addSql("        USR_SID = ?");
            sql.addIntValue(usrSid);
            sql.addSql("      and");
            sql.addSql("        CUUC_FAVORITE_KBN = ?");
            sql.addIntValue(GSConstChat.CHAT_FAVORITE);

            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                return rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * お気に入り登録されているユーザSIDを取得する
     * @param usrSid ユーザSID
     * @return List in ChatUserInfModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<Integer> favoriteUserSelect(int usrSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<Integer> ret = new ArrayList<Integer>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql("  select ");
            sql.addSql("    CMN_USRM_INF.USR_SID");
            sql.addSql("  from ");
            sql.addSql("    CMN_USRM_INF,");
            sql.addSql("    CHT_USER_UCONF,");
            sql.addSql("    CHT_USER_PAIR");
            sql.addSql("  where");
            sql.addSql("    CHT_USER_UCONF.USR_SID = ?");
            sql.addIntValue(usrSid);
            sql.addSql("  and");
            sql.addSql("    CHT_USER_UCONF.CUUC_FAVORITE_KBN = ?");
            sql.addIntValue(GSConstChat.CHAT_FAVORITE);
            sql.addSql("  and");
            sql.addSql("    CHT_USER_UCONF.CUP_SID = CHT_USER_PAIR.CUP_SID");
            sql.addSql("  and");
            sql.addSql("    case");
            sql.addSql("      when CHT_USER_PAIR.CUP_UID_F = ?");
            sql.addIntValue(usrSid);
            sql.addSql("        then CHT_USER_PAIR.CUP_UID_S = CMN_USRM_INF.USR_SID");
            sql.addSql("      else CHT_USER_PAIR.CUP_UID_F = CMN_USRM_INF.USR_SID");
            sql.addSql("    end");

            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret.add(rs.getInt("USR_SID"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;

    }

    /**
     * <p>お気に入り登録されているグループ情報の取得
     * @param usrSid ユーザSID
     * @return List in ChatGroupInfModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<ChatGroupInfModel> favoriteGroupSelect(int usrSid)
            throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<ChatGroupInfModel> ret = new ArrayList<ChatGroupInfModel>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   CHT_GROUP_INF.CGI_SID,");
            sql.addSql("   CHT_GROUP_INF.CGI_NAME,");
            sql.addSql("   CHT_GROUP_INF.CGI_COMP_FLG,");
            sql.addSql("   COUNT_DATA.LAST_DATE,");
            sql.addSql("   COUNT_DATA.CNT,");
            sql.addSql("   COUNT_DATA.CGUC_MUTE_KBN, ");
            sql.addSql("   COUNT_DATA.CGUC_FAVORITE_KBN ");
                sql.addSql(" from ");
            sql.addSql("   CHT_GROUP_INF");
            sql.addSql(" left join (");
            writeQueryGroupChatExtra(sql, usrSid, null, false);
            sql.addSql(" )COUNT_DATA");
            sql.addSql(" on");
            sql.addSql("   CHT_GROUP_INF.CGI_SID = COUNT_DATA.GROUP_SID");

            //検索条件
            sql = __setFavoriteGroupWhereSql(sql, usrSid);

            //並び順
            sql.addSql(" order by ");
            sql.addSql("   CGI_SID ");

            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ChatGroupInfModel bean = new ChatGroupInfModel();
                bean.setCgiSid(rs.getInt("CGI_SID"));
                bean.setCgiName(rs.getString("CGI_NAME"));
                bean.setCgiCompFlg(rs.getInt("CGI_COMP_FLG"));
                bean.setChtGroupCount(rs.getInt("CNT"));
                bean.setChtLastDate(
                    UDate.getInstanceTimestamp(rs.getTimestamp("LAST_DATE"))
                    );
                ChtGroupUconfModel uconf = new ChtGroupUconfModel();
                uconf.setUsrSid(usrSid);
                uconf.setCgiSid(bean.getCgiSid());
                uconf.setCgucFavoriteKbn(rs.getInt("CGUC_FAVORITE_KBN"));
                uconf.setCgucMuteKbn(rs.getInt("CGUC_MUTE_KBN"));
                bean.setUconfModel(uconf);
                ret.add(bean);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }
    /**
     * <p>お気に入り登録されているグループ情報件数
     * @param usrSid ユーザSID
     * @return List in CHT_FAVORITEModel
     * @throws SQLException SQL実行例外
     */
    public int favoriteGroupCount(int usrSid)
            throws SQLException {
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   count(CGI_SID) as CNT");
            sql.addSql(" from ");
            sql.addSql("   CHT_GROUP_INF");

            //検索条件
            sql = __setFavoriteGroupWhereSql(sql, usrSid);

            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                return rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * 削除済みユーザを取得する
     * @return List Integer
     * @throws SQLException SQL実行例外
     */
    public ArrayList<Integer> getDelUser() throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<Integer> ret = new ArrayList<Integer>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   CMN_USRM.USR_SID");
            sql.addSql(" from ");
            sql.addSql("   CMN_USRM");
            sql.addSql(" where ");
            sql.addSql("   CMN_USRM.USR_JKBN = ?");

            sql.addIntValue(GSConst.JTKBN_DELETE);

            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret.add(rs.getInt("USR_SID"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <br>[機  能] 削除対象のグループチャットSID一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param delMdl 手動削除設定モデル
     * @return チャットSID一覧
     * @throws SQLException SQL実行時例外
     */
    public List<Integer> getDelGroupChatSidList(ChatDeleteModel delMdl) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        List<Integer> chtGrpSidList = new ArrayList<Integer>();
        SqlBuffer sql = null;

        try {
            UDate delDate = new UDate();
            delDate.setMaxHhMmSs();
            delDate.addYear((delMdl.getDelYear() * -1));
            delDate.addMonth((delMdl.getDelMonth() * -1));

            sql = new SqlBuffer();
            sql.addSql(" select CGI_SID from CHT_GROUP_INF");

            sql.addSql(" where");
            sql.addSql(" and CGI_ADATE <= ? ");
            sql.addDateValue(delDate);


            log__.info(sql.toLogString());

            pstmt = getCon().prepareStatement(sql.toSqlString());

            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();

            while (rs.next()) {
                chtGrpSidList.add(rs.getInt("CGI_SID"));
            }
        } catch (SQLException e) {
            if (sql != null) {
                log__.error("Error SQL: " + sql.toLogString());
            }
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closePreparedStatement(pstmt);
        }

        return chtGrpSidList;
    }

    /**
     * <br>[機  能] 削除対象のユーザチャットSID一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param delMdl 手動削除設定モデル
     * @return チャットSID一覧
     * @throws SQLException SQL実行時例外
     */
    public List<Integer> getDelUserChatSidList(ChatDeleteModel delMdl) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        List<Integer> chtUsrSidList = new ArrayList<Integer>();
        SqlBuffer sql = null;

        try {
            UDate delDate = new UDate();
            delDate.setMaxHhMmSs();
            delDate.addYear((delMdl.getDelYear() * -1));
            delDate.addMonth((delMdl.getDelMonth() * -1));

            sql = new SqlBuffer();
            sql.addSql(" select CGI_SID from CHT_GROUP_INF");

            sql.addSql(" where");
            sql.addSql(" and CGI_ADATE <= ? ");
            sql.addDateValue(delDate);


            log__.info(sql.toLogString());

            pstmt = getCon().prepareStatement(sql.toSqlString());

            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();

            while (rs.next()) {
                chtUsrSidList.add(rs.getInt("CGI_SID"));
            }
        } catch (SQLException e) {
            if (sql != null) {
                log__.error("Error SQL: " + sql.toLogString());
            }
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closePreparedStatement(pstmt);
        }

        return chtUsrSidList;
    }


    /**
     * <br>[機  能]メッセージ情報select(ユーザ)
     * <br>[解  説]
     * <br>[備  考]
     * @param sql SqlBuffer
     * @param pinKbn ピンどめ投稿取得区分
     * @return sql SqlBuffer
     */
    public SqlBuffer getUserMessage(SqlBuffer sql, int pinKbn) {
        sql.addSql(" select");
        sql.addSql("   CMN_USRM_INF.USI_SEI,");
        sql.addSql("   CMN_USRM_INF.USI_MEI,");
        sql.addSql("   CMN_USRM_INF.BIN_SID as USR_BIN,");
        sql.addSql("   CMN_USRM_INF.USI_PICT_KF,");
        sql.addSql("   CMN_USRM.USR_JKBN,");
        sql.addSql("   CMN_USRM.USR_UKO_FLG,");
        sql.addSql("   CMN_USRM.USR_LGID,");
        sql.addSql("   CHT_USER_DATA.CUD_SID,");
        sql.addSql("   CHT_USER_DATA.CUP_SID,");
        sql.addSql("   CHT_USER_DATA.CUD_TEXT,");
        sql.addSql("   CHT_USER_DATA.CUD_SEND_UID,");
        sql.addSql("   CHT_USER_DATA.CUD_STATE_KBN,");
        sql.addSql("   CHT_USER_DATA.CUD_AUID,");
        sql.addSql("   CHT_USER_DATA.CUD_ADATE,");
        sql.addSql("   CHT_USER_DATA.CUD_EUID,");
        sql.addSql("   CHT_USER_DATA.CUD_EDATE,");
        sql.addSql("   CHT_USER_DATA.CUD_REPLY_SID,");
        sql.addSql("   CHT_USER_DATA.CST_SID,");
        sql.addSql("   CHT_STAMP.BIN_SID as STAMP_BIN,");
        sql.addSql("   CHT_STAMP.CST_DEFSTAMP_ID");
        sql.addSql(" from");
        sql.addSql("   CHT_USER_DATA");
        sql.addSql(" left join");
        sql.addSql("   CMN_USRM_INF");
        sql.addSql(" on");
        sql.addSql("   CHT_USER_DATA.CUD_SEND_UID = CMN_USRM_INF.USR_SID");
        sql.addSql(" left join");
        sql.addSql("   CMN_USRM");
        sql.addSql(" on");
        sql.addSql("   CMN_USRM_INF.USR_SID = CMN_USRM.USR_SID");
        sql.addSql(" left join");
        sql.addSql("   CHT_STAMP");
        sql.addSql(" on");
        sql.addSql("   CHT_STAMP.CST_SID = CHT_USER_DATA.CST_SID");
        //ピンどめ一覧取得時用
        if (pinKbn == GSConstChat.CHAT_MESSAGE_GET_PIN) {
            sql.addSql("    inner join");
            sql.addSql("      CHT_USER_DATA_PIN");
            sql.addSql("    on");
            sql.addSql("      CHT_USER_DATA_PIN.CUP_SID = CHT_USER_DATA.CUP_SID");
            sql.addSql("      and CHT_USER_DATA_PIN.CUD_SID = CHT_USER_DATA.CUD_SID");
        }
        return sql;
    }

    /**
     * <br>[機  能] グループメッセージ検索 前提情報取得
     * <br>[解  説] 検索ヒット件数、閲覧情報の取得
     * <br>[備  考]
     * @param searchReq
     * @return ChatMessageModelList
     * @throws SQLException SQLException
     */
    public ChatMessageSearchInfo getSearchInfoUserMessage(ChatMessageSearchRequest searchReq)
            throws SQLException {

        //自身の投稿SIDを取得する
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ChatMessageSearchInfo ret = new ChatMessageSearchInfo();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   COALESCE(VIEW.VIEW_SID, 0) as VIEW_SID,");
            sql.addSql("   COALESCE(VIEW.VIEW_CNT, 0) as VIEW_CNT,");
            sql.addSql("   DATA.DATA_CNT,");
            sql.addSql("   DATA.TARGET_OFFSET,");
            sql.addSql("   DATA.TARGET_HAVE");
            sql.addSql(" from");
            sql.addSql("   (");
            sql.addSql("    select");
            sql.addSql("      CHT_USER_DATA.CUP_SID,");
            sql.addSql("      COUNT(CHT_USER_DATA.CUD_SID) as DATA_CNT,");
            switch (searchReq.getPosition().getType()) {
                case BEFORE_MESSAGE:
                case ARROUND_MESSAGE:
                case AFTER_MESSAGE:
                    sql.addSql("      COUNT(CHT_USER_DATA.CUD_SID < ? or NULL) as TARGET_OFFSET,");
                    sql.addLongValue(
                        (
                            (AbstractMessagePosition)
                                searchReq.getPosition()
                            ).getMessageSid()
                    );
                    sql.addSql("      COUNT(CHT_USER_DATA.CUD_SID = ? or NULL) as TARGET_HAVE");
                    sql.addLongValue(
                        (
                            (AbstractMessagePosition)
                                searchReq.getPosition()
                            ).getMessageSid()
                    );
                break;
                default:
                    sql.addSql("      0 as TARGET_OFFSET,");
                    sql.addSql("      0 as TARGET_HAVE");
            }
            sql.addSql("    from");
            sql.addSql("      CHT_USER_DATA");
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    inner join");
                sql.addSql("      CHT_USER_DATA_PIN");
                sql.addSql("    on");
                sql.addSql("      CHT_USER_DATA_PIN.CUP_SID = CHT_USER_DATA.CUP_SID");
                sql.addSql("      and CHT_USER_DATA_PIN.CUD_SID = CHT_USER_DATA.CUD_SID");
            }
            sql.addSql("    where");
            sql.addSql("      CHT_USER_DATA.CUP_SID = ?");
            sql.addIntValue(searchReq.getChatPairSid());
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    and");
                sql.addSql("      CHT_USER_DATA_PIN.USR_SID = ?");
                sql.addIntValue(searchReq.getSessionUserSid());
            }

            __writeSearchUserChatWhere(
                sql,
                searchReq.getFilterList(),
                searchReq.getChatPairSid());

            sql.addSql("    group by");
            sql.addSql("      CHT_USER_DATA.CUP_SID");
            sql.addSql("   )DATA");
            sql.addSql("   left join");
            sql.addSql("   (");
            sql.addSql("    select");
            sql.addSql("      CHT_USER_VIEW.CUP_SID,");
            sql.addSql("      CHT_USER_VIEW.CUD_SID as VIEW_SID,");
            sql.addSql("      COUNT(CHT_USER_DATA.CUD_SID) as VIEW_CNT");
            sql.addSql("    from");
            sql.addSql("      CHT_USER_VIEW");
            sql.addSql("    left join");
            sql.addSql("      CHT_USER_DATA");
            sql.addSql("    on");
            sql.addSql("      CHT_USER_VIEW.CUP_SID = CHT_USER_DATA.CUP_SID");
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    inner join");
                sql.addSql("      CHT_USER_DATA_PIN");
                sql.addSql("    on");
                sql.addSql("      CHT_USER_DATA_PIN.CUP_SID = CHT_USER_DATA.CUP_SID");
                sql.addSql("      and CHT_USER_DATA_PIN.CUD_SID = CHT_USER_DATA.CUD_SID");
            }
            sql.addSql("    where");
            sql.addSql("      CHT_USER_VIEW.CUD_SID >= CHT_USER_DATA.CUD_SID");
            sql.addSql("    and");
            sql.addSql("      CHT_USER_VIEW.CUP_SID = ?");
            sql.addIntValue(searchReq.getChatPairSid());
            sql.addSql("    and");
            sql.addSql("      CHT_USER_VIEW.CUV_UID = ?");
            sql.addIntValue(searchReq.getSessionUserSid());
            if (searchReq.getPosition()
                    instanceof MidokuPosition) {
                __writeSearchUserChatWhere(
                    sql,
                    searchReq.getFilterList(),
                    searchReq.getChatPairSid());
            }

            sql.addSql("    group by");
            sql.addSql("      CHT_USER_VIEW.CUD_SID,");
            sql.addSql("      CHT_USER_VIEW.CUP_SID");
            sql.addSql("   )VIEW");
            sql.addSql("   on");
            sql.addSql("     DATA.CUP_SID = VIEW.CUP_SID");

            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret.setViewSid(rs.getLong("VIEW_SID"));
                ret.setViewCnt(rs.getInt("VIEW_CNT"));
                ret.setHitCnt(rs.getInt("DATA_CNT"));
                ret.setTargetMessageOffset(rs.getInt("TARGET_OFFSET"));
                ret.setTargetMessageHave(rs.getInt("TARGET_HAVE"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }


    /**
     * <p>ユーザ間メッセージ情報の取得
     * @param viewSid 閲覧SID
     * @param partnerViewSid 閲覧SID
     * @param kidokuFlg 既読表示フラグ
     * @param searchReq 検索リクエスト
     * @return CHT_USER_DATAModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<ChatMessageModel> searchUserMessage(
            Long viewSid, int partnerViewSid,
             boolean kidokuFlg, ChatMessageSearchRequest searchReq) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<ChatMessageModel> ret = new ArrayList<ChatMessageModel>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            getUserMessage(sql, searchReq.getTargetPinKbn());
            sql.addSql(" where ");
            sql.addSql("   CHT_USER_DATA.CUP_SID=?");
            sql.addLongValue(searchReq.getChatPairSid());
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    and");
                sql.addSql("      CHT_USER_DATA_PIN.USR_SID = ?");
                sql.addIntValue(searchReq.getSessionUserSid());
            }
            if (searchReq.getPosition()
                    instanceof BeforeMessage) {
                    BeforeMessage pos
                         = (BeforeMessage) searchReq.getPosition();
                    sql.addSql(" and  CHT_USER_DATA.CUD_SID < ?");
                    sql.addLongValue(pos.getMessageSid());

            }
            if (searchReq.getPosition()
                    instanceof AfterMessage) {
                    AfterMessage pos
                        = (AfterMessage) searchReq.getPosition();
                    sql.addSql(" and  CHT_USER_DATA.CUD_SID > ?");
                    sql.addLongValue(pos.getMessageSid());
            }

            __writeSearchUserChatWhere(sql, searchReq.getFilterList(), searchReq.getChatPairSid());

            int order = searchReq.getOrder();
            if (searchReq.getPosition()
                    instanceof BeforeMessage) {
                order = GSConst.ORDER_KEY_DESC;
            }
            if (searchReq.getPosition()
                instanceof AfterMessage) {
                order = GSConst.ORDER_KEY_ASC;
            }
            if (searchReq.getPosition()
                instanceof NewestPosition) {
                order = GSConst.ORDER_KEY_DESC;
            }

            sql.addSql(" order by ");
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_ALL) {
                if (order == GSConst.ORDER_KEY_ASC) {
                    sql.addSql("   CHT_USER_DATA.CUD_SID ");
                } else {
                    sql.addSql("   CHT_USER_DATA.CUD_SID desc ");
                }
            } else {
                //ピンどめ並び順
                sql.addSql("   CHT_USER_DATA_PIN.CUDP_SORT desc ");
            }
            int offset = 0;
            if (searchReq.getPosition()
                    instanceof OffsetPosition) {
                        OffsetPosition pos
                        = (OffsetPosition) searchReq.getPosition();
                offset = pos.getOffset();
            }

            sql.setPagingValue(offset, searchReq.getLimit());

            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret.add(__getChatUsrMessageModel(rs, viewSid, partnerViewSid, kidokuFlg));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        ret = __setTempInfo(ret, GSConstChat.CHAT_KBN_USER);
        __setUserReactionInfo(ret);
        __setReplyMessage(ret, GSConstChat.CHAT_KBN_USER);
        __setMentionInfo(ret, GSConstChat.CHAT_KBN_USER);
        __setMessagePinInfo(ret, GSConstChat.CHAT_KBN_USER,
            searchReq.getChatPairSid(), searchReq.getSessionUserSid());
        __setOutMemberFlg(
            ret, searchReq.getChatPairSid(),
            GSConstChat.CHAT_KBN_USER);


        return ret;
    }

    /***
     * 検索条件を書き込み
     * @param sql
     * @param filterList
     * @param cupSid
     */
    private void __writeSearchUserChatWhere(SqlBuffer sql,
            List<ChatMessageSearchFilter> filterList, int cupSid) {

        if (filterList == null
            || filterList.size() <= 0) {
            return;
        }
        //検索条件振り分け
        List<ChatMessageSearchFilter> keywordFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> attachmentFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> urlFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> senderFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> messageFilters = new ArrayList<>();
        boolean outMember = false;
        boolean deletedMessageLess = false;
        if (filterList.size() > 0) {
            for (ChatMessageSearchFilter filter : filterList) {
                switch (filter.getMode()) {
                    case keyword:
                        keywordFilters.add(filter);
                        break;
                    case attachment:
                    case attachmentAll:
                    case attachmentLess:
                        attachmentFilters.add(filter);
                        break;
                    case url:
                    case urlAll:
                    case urlLess:
                        urlFilters.add(filter);
                        break;
                    case user:
                        senderFilters.add(filter);
                        break;
                    case outMember:
                        outMember = true;
                        break;
                    case deletedLess:
                        deletedMessageLess = true;
                        break;
                    case message:
                        messageFilters.add(filter);
                        break;
                    default:
                }
            }
        }
        //キーワード検索
        for (ChatMessageSearchFilter filter : keywordFilters) {
            sql.addSql(" and (");
            sql.addSql("   CHT_USER_DATA.CUD_TEXT like '%"
            + JDBCUtil.escapeForLikeSearch(filter.getValue())
            + "%' ESCAPE '"
            + JDBCUtil.def_esc
            + "'");
            sql.addSql(" or ");
            sql.addSql("   CHT_USER_DATA.CUD_SID in (");

            sql.addSql(" select");
            sql.addSql("   ATTACHED.CUD_SID");
            sql.addSql(" from");
            sql.addSql("  CHT_USER_DATA ATTACHED");
            sql.addSql("  inner join CHT_USER_DATA_TEMP ATTACHBELONG");
            sql.addSql("    on ATTACHED.CUD_SID = ATTACHBELONG.CUD_SID");
            sql.addSql("  inner join CMN_BINF ATTACH");
            sql.addSql("    on ATTACHBELONG.BIN_SID = ATTACH.BIN_SID");
            sql.addSql(" where ");
            sql.addSql("   ATTACHED.CUP_SID = ?");
            sql.addSql("     and ATTACH.BIN_FILE_NAME like '%");
            sql.addSql(JDBCUtil.escapeForLikeSearch(filter.getValue()));
            sql.addSql("%' ESCAPE '");
            sql.addSql(JDBCUtil.def_esc);
            sql.addSql("'");
            sql.addSql(" ) ");
            sql.addIntValue(cupSid);

            sql.addSql(" )");
        }
        //添付ファイル検索
        for (ChatMessageSearchFilter filter : attachmentFilters) {
            sql.addSql(" and ");
            if (filter.getMode() == ChatMessageSearchFilter.EnumType.attachmentLess) {
                sql.addSql("not exists (");
                sql.addSql(" select");
                sql.addSql("   1");
                sql.addSql(" from");
                sql.addSql("  CHT_USER_DATA_TEMP ATTACHBELONG");
                sql.addSql(" where ");
                sql.addSql("   ATTACHBELONG.CUD_SID = CHT_USER_DATA.CUD_SID");
                sql.addSql(" ) ");
            } else {

                sql.addSql("   CHT_USER_DATA.CUD_SID in (");

                sql.addSql(" select");
                sql.addSql("   ATTACHED.CUD_SID");
                sql.addSql(" from");
                sql.addSql("  CHT_USER_DATA ATTACHED");
                sql.addSql("  inner join CHT_USER_DATA_TEMP ATTACHBELONG");
                sql.addSql("    on ATTACHED.CUD_SID = ATTACHBELONG.CUD_SID");
                sql.addSql("  inner join CMN_BINF ATTACH");
                sql.addSql("    on ATTACHBELONG.BIN_SID = ATTACH.BIN_SID");
                sql.addSql(" where ");
                sql.addSql("   ATTACHED.CUP_SID = ?");
                sql.addIntValue(cupSid);
                if (filter.getMode() != ChatMessageSearchFilter.EnumType.attachmentAll) {
                    sql.addSql("     and ATTACH.BIN_FILE_NAME like '%");
                    sql.addSql(JDBCUtil.escapeForLikeSearch(filter.getValue()));
                    sql.addSql("%' ESCAPE '");
                    sql.addSql(JDBCUtil.def_esc);
                    sql.addSql("'");
                }
                sql.addSql(" ) ");
            }
        }

        //URL検索
        for (ChatMessageSearchFilter filter : urlFilters) {
            if (filter.getMode() == ChatMessageSearchFilter.EnumType.urlAll) {
                sql.addSql(" and ");
                sql.addSql("   CHT_USER_DATA.CUD_URL_FLG = 1");
            } else if (filter.getMode() == ChatMessageSearchFilter.EnumType.urlLess) {
                sql.addSql(" and ");
                sql.addSql("   CHT_USER_DATA.CUD_URL_FLG = 0");
            } else {
                sql.addSql(" and (");
                sql.addSql("   CHT_USER_DATA.CUD_TEXT like '%http:%' ESCAPE '"
                + JDBCUtil.def_esc
                + "'");
                sql.addSql("   OR CHT_USER_DATA.CUD_TEXT like '%https:%' ESCAPE '"
                + JDBCUtil.def_esc
                + "'");
                sql.addSql(" ) ");
            }
        }
        //ユーザ検索
        if (senderFilters.size() > 0  || outMember) {
            sql.addSql(" and (");
            boolean first = true;
            for (ChatMessageSearchFilter filter : senderFilters) {
                if (!first) {
                    sql.addSql(" or ");
                }
                first = false;
                sql.addSql("   CHT_USER_DATA.CUD_SEND_UID = ?");
                sql.addIntValue(NullDefault.getInt(filter.getValue(), -1));
            }
            if (outMember) {
                if (senderFilters.size() > 0) {
                    sql.addSql(" or ");
                }
                sql.addSql(" exists (");
                sql.addSql(" select");
                sql.addSql("  1 ");
                sql.addSql(" from");
                sql.addSql("  CMN_USRM");
                sql.addSql(" where ");
                sql.addSql("  CMN_USRM.USR_SID = CHT_USER_DATA.CUD_SEND_UID");
                sql.addSql("     and CMN_USRM.USR_JKBN = ?");
                sql.addIntValue(GSConstUser.USER_JTKBN_DELETE);
                sql.addSql(" ) ");

            }
            sql.addSql(" )");

        }
        //削除済みメッセージを除外
        if (deletedMessageLess) {
            sql.addSql(" and ");
            sql.addSql("   CHT_USER_DATA.CUD_STATE_KBN <> ?");
            sql.addIntValue(GSConst.JTKBN_DELETE);
        }
        //メッセージSID検索
        if (messageFilters.size() > 0) {
            sql.addSql(" and (");
            boolean first = true;

            for (ChatMessageSearchFilter filter : messageFilters) {
                if (!first) {
                    sql.addSql(" or ");
                }
                first = false;
                sql.addSql("   CHT_USER_DATA.CUD_SID = ?");
                sql.addIntValue(NullDefault.getInt(filter.getValue(), -1));
            }

            sql.addSql(" )");

        }


    }

    /**
     * <p>指定したメッセージSIDのユーザ間チャット送信者と送信日付を取得する
     * <p>返信元の投稿情報を取得する際に使用する
     * @param messageSidList 返信元投稿SID
     * @return メッセージSID:メッセージ情報を格納したマップ
     * @throws SQLException SQL実行例外
     */
    public Map<Long, ChatMessageModel> getUserReplyMap(
        List<Long> messageSidList) throws SQLException {

        Map<Long, ChatMessageModel> ret = new HashMap<Long, ChatMessageModel>();
        if (messageSidList == null || messageSidList.isEmpty()) {
            return ret;
        }

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        con = getCon();
        ArrayList<ChatMessageModel> cmList = new ArrayList<ChatMessageModel>();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            getUserMessage(sql, GSConstChat.CHAT_MESSAGE_GET_ALL);
            sql.addSql(" where ");
            sql.addSql("   CHT_USER_DATA.CUD_SID in ( ");
            for (int idx = 0; idx < messageSidList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql(", ");
                }
                sql.addSql("?");
                sql.addLongValue(messageSidList.get(idx));
            }
            sql.addSql("   ) ");

            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();

            while (rs.next()) {
                cmList.add(__getChatUsrMessageReplyModel(rs));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        __setTempInfo(cmList, GSConstChat.CHAT_KBN_USER);
        for (ChatMessageModel cmMdl : cmList) {
            ret.put(cmMdl.getMessageSid(), cmMdl);
        }

        return ret;
    }


    /**
     * <p>Create CHT_USER_DATA Data Bindding JavaBean From ResultSet
     * @param rs ResultSet
     * @param viewSid viewSid
     * @param partnerViewSid partnerViewSid
     * @param kidokuFlg kidokuFlg
     * @return created ChtGroupDataModel
     * @throws SQLException SQL実行例外
     */
    private ChatMessageModel __getChatUsrMessageModel(ResultSet rs,
            long viewSid, long partnerViewSid, boolean kidokuFlg) throws SQLException {

        ChatMessageModel cmMdl = new ChatMessageModel();
        cmMdl.setMessageSid(rs.getLong("CUD_SID"));
        if (viewSid >= rs.getInt("CUD_SID")) {
            //自身の既読情報登録
            cmMdl.setOwnKidoku(GSConstChat.KIDOKU_DISP);
        }
        cmMdl.setMessageKbn(rs.getInt("CUD_STATE_KBN"));
        UDate date = UDate.getInstanceTimestamp(rs.getTimestamp("CUD_ADATE"));
        cmMdl.setEntryDay(date.getStrYear() + "/" + date.getStrMonth() + "/" + date.getStrDay());
        cmMdl.setEntryTime(date.getStrHour() + ":" + date.getStrMinute());

        UDate uDate = UDate.getInstanceTimestamp(rs.getTimestamp("CUD_EDATE"));
        cmMdl.setUpdateDay(
            uDate.getStrYear()
            + "/" + uDate.getStrMonth()
            + "/" + uDate.getStrDay());
        cmMdl.setUpdateTime(uDate.getStrHour() + ":" + uDate.getStrMinute());

        //削除された投稿の場合は、画面描画に必要ない情報を設定しない
        if (cmMdl.getMessageKbn() == GSConstChat.TOUKOU_STATUS_DELETE) {
            return cmMdl;
        }

        cmMdl.setSelectSid(rs.getInt("CUP_SID"));
        cmMdl.setMessageText(rs.getString("CUD_TEXT"));
        cmMdl.setStampSid(rs.getInt("CST_SID"));
        cmMdl.setStampBinSid(rs.getInt("STAMP_BIN"));
        cmMdl.setStampDefaultId(rs.getInt("CST_DEFSTAMP_ID"));
        cmMdl.setInsertUid(rs.getInt("CUD_AUID"));
        cmMdl.setUpdateUid(rs.getInt("CUD_EUID"));
        if (partnerViewSid >= rs.getInt("CUD_SID") && kidokuFlg) {
            //相手の既読情報登録
            cmMdl.setPartnerKidoku(GSConstChat.KIDOKU_DISP);
        }
        cmMdl.setUsrSid(rs.getInt("CUD_SEND_UID"));
        cmMdl.setUsrName(UserUtil.makeName(rs.getString("USI_SEI"), rs.getString("USI_MEI")));
        cmMdl.setUsrId(rs.getString("USR_LGID"));
        cmMdl.setUsrJkbn(rs.getInt("USR_JKBN"));
        cmMdl.setUsrUkoFlg(rs.getInt("USR_UKO_FLG"));
        cmMdl.setUsrBinSid(rs.getInt("USR_BIN"));
        cmMdl.setUsrPictKf(rs.getInt("USI_PICT_KF"));
        cmMdl.setReplyMessageSid(rs.getLong("CUD_REPLY_SID"));
        return cmMdl;
    }

    /**
     * <p>SQL実行結果から、返信元投稿データを作成する
     * <p>返信元投稿は表示する内容のみを格納する
     * @param rs ResultSet
     * @return created ChtGroupDataModel
     * @throws SQLException SQL実行例外
     */
    private ChatMessageModel __getChatUsrMessageReplyModel(ResultSet rs) throws SQLException {

        ChatMessageModel cmMdl = new ChatMessageModel();
        cmMdl.setMessageSid(rs.getLong("CUD_SID"));
        cmMdl.setMessageKbn(rs.getInt("CUD_STATE_KBN"));
        cmMdl.setMessageText(rs.getString("CUD_TEXT"));
        cmMdl.setUsrSid(rs.getInt("CUD_SEND_UID"));
        cmMdl.setUsrName(UserUtil.makeName(rs.getString("USI_SEI"), rs.getString("USI_MEI")));
        cmMdl.setUsrId(rs.getString("USR_LGID"));
        cmMdl.setUsrJkbn(rs.getInt("USR_JKBN"));
        cmMdl.setUsrUkoFlg(rs.getInt("USR_UKO_FLG"));
        cmMdl.setUsrBinSid(rs.getInt("USR_BIN"));
        cmMdl.setUsrPictKf(rs.getInt("USI_PICT_KF"));
        cmMdl.setStampSid(rs.getInt("CST_SID"));
        cmMdl.setStampBinSid(rs.getInt("STAMP_BIN"));
        cmMdl.setStampDefaultId(rs.getInt("CST_DEFSTAMP_ID"));
        return cmMdl;
    }

    /**
     * <br>[機  能]メッセージ情報select(グループ)
     * <br>[解  説]
     * <br>[備  考]
     * @param sql SqlBuffer
     * @param pinKbn ピンどめ投稿取得区分
     * @return sql SqlBuffer
     */
    public SqlBuffer getGroupMessage(SqlBuffer sql, int pinKbn) {
        sql.addSql(" select");
        sql.addSql("   CMN_USRM_INF.USI_SEI,");
        sql.addSql("   CMN_USRM_INF.USI_MEI,");
        sql.addSql("   CMN_USRM_INF.BIN_SID as USR_BIN,");
        sql.addSql("   CMN_USRM_INF.USI_PICT_KF,");
        sql.addSql("   CMN_USRM.USR_JKBN,");
        sql.addSql("   CMN_USRM.USR_UKO_FLG,");
        sql.addSql("   CMN_USRM.USR_LGID,");
        sql.addSql("   CHT_GROUP_DATA.CGD_SID,");
        sql.addSql("   CHT_GROUP_DATA.CGI_SID,");
        sql.addSql("   CHT_GROUP_DATA.CGD_TEXT,");
        sql.addSql("   CHT_GROUP_DATA.CGD_SEND_UID,");
        sql.addSql("   CHT_GROUP_DATA.CGD_STATE_KBN,");
        sql.addSql("   CHT_GROUP_DATA.CGD_AUID,");
        sql.addSql("   CHT_GROUP_DATA.CGD_ADATE,");
        sql.addSql("   CHT_GROUP_DATA.CGD_EUID,");
        sql.addSql("   CHT_GROUP_DATA.CGD_EDATE,");
        sql.addSql("   CHT_GROUP_DATA.CGD_REPLY_SID,");
        sql.addSql("   CHT_GROUP_DATA.CST_SID,");
        sql.addSql("   CHT_STAMP.BIN_SID as STAMP_BIN,");
        sql.addSql("   CHT_STAMP.CST_DEFSTAMP_ID");
        sql.addSql(" from");
        sql.addSql("   CHT_GROUP_DATA");
        sql.addSql(" left join");
        sql.addSql("   CMN_USRM_INF");
        sql.addSql(" on");
        sql.addSql("   CHT_GROUP_DATA.CGD_SEND_UID = CMN_USRM_INF.USR_SID");
        sql.addSql(" left join");
        sql.addSql("   CMN_USRM");
        sql.addSql(" on");
        sql.addSql("   CMN_USRM_INF.USR_SID = CMN_USRM.USR_SID");
        sql.addSql(" left join");
        sql.addSql("   CHT_STAMP");
        sql.addSql(" on");
        sql.addSql("   CHT_STAMP.CST_SID = CHT_GROUP_DATA.CST_SID");

        //ピンどめ一覧取得時用
        if (pinKbn == GSConstChat.CHAT_MESSAGE_GET_PIN) {
            sql.addSql("    inner join");
            sql.addSql("      CHT_GROUP_DATA_PIN");
            sql.addSql("    on");
            sql.addSql("      CHT_GROUP_DATA_PIN.CGI_SID = CHT_GROUP_DATA.CGI_SID");
            sql.addSql("      and CHT_GROUP_DATA_PIN.CGD_SID = CHT_GROUP_DATA.CGD_SID");
        }

        return sql;
    }

    /**
     * <br>[機  能] グループメッセージ検索 前提情報取得
     * <br>[解  説] 検索ヒット件数、閲覧情報の取得
     * <br>[備  考]
     * @param searchReq
     * @return ChatMessageModelList
     * @throws SQLException SQLException
     */
    public ChatMessageSearchInfo getSearchInfoGroupMessage(ChatMessageSearchRequest searchReq)
            throws SQLException {

        //自身の投稿SIDを取得する
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ChatMessageSearchInfo ret = new ChatMessageSearchInfo();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   COALESCE(VIEW.VIEW_SID, 0) as VIEW_SID,");
            sql.addSql("   COALESCE(VIEW.VIEW_CNT, 0) as VIEW_CNT,");
            sql.addSql("   DATA.DATA_CNT,");
            sql.addSql("   DATA.TARGET_OFFSET,");
            sql.addSql("   DATA.TARGET_HAVE");
            sql.addSql(" from");
            sql.addSql("   (");
            sql.addSql("    select");
            sql.addSql("      CHT_GROUP_DATA.CGI_SID,");
            sql.addSql("      COUNT(CHT_GROUP_DATA.CGD_SID) as DATA_CNT,");
            switch (searchReq.getPosition().getType()) {
                case BEFORE_MESSAGE:
                case ARROUND_MESSAGE:
                case AFTER_MESSAGE:
                    sql.addSql("      COUNT(CHT_GROUP_DATA.CGD_SID < ? or NULL) as TARGET_OFFSET,");
                    sql.addLongValue(
                        (
                            (AbstractMessagePosition)
                                searchReq.getPosition()
                            ).getMessageSid()
                    );
                    sql.addSql("      COUNT(CHT_GROUP_DATA.CGD_SID = ? or NULL) as TARGET_HAVE");
                    sql.addLongValue(
                        (
                            (AbstractMessagePosition)
                                searchReq.getPosition()
                            ).getMessageSid()
                    );
                break;
                default:
                    sql.addSql("      0 as TARGET_OFFSET,");
                    sql.addSql("      0 as TARGET_HAVE");
            }
            sql.addSql("    from");
            sql.addSql("      CHT_GROUP_DATA");
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    inner join");
                sql.addSql("      CHT_GROUP_DATA_PIN");
                sql.addSql("    on");
                sql.addSql("      CHT_GROUP_DATA_PIN.CGI_SID = CHT_GROUP_DATA.CGI_SID");
                sql.addSql("      and CHT_GROUP_DATA_PIN.CGD_SID = CHT_GROUP_DATA.CGD_SID");
            }
            sql.addSql("    where");
            sql.addSql("      CHT_GROUP_DATA.CGI_SID = ?");
            sql.addIntValue(searchReq.getChatGroupSid());
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    and");
                sql.addSql("      CHT_GROUP_DATA_PIN.USR_SID = ?");
                sql.addIntValue(searchReq.getSessionUserSid());
            }

            __writeSearchGroupChatWhere(sql,
                searchReq.getFilterList(),
                searchReq.getChatGroupSid());

            sql.addSql("    group by");
            sql.addSql("      CHT_GROUP_DATA.CGI_SID");
            sql.addSql("   )DATA");
            sql.addSql("   left join");
            sql.addSql("   (");
            sql.addSql("    select");
            sql.addSql("      CHT_GROUP_VIEW.CGI_SID,");
            sql.addSql("      CHT_GROUP_VIEW.CGD_SID as VIEW_SID,");
            sql.addSql("      COUNT(CHT_GROUP_DATA.CGD_SID) as VIEW_CNT");
            sql.addSql("    from");
            sql.addSql("      CHT_GROUP_VIEW");
            sql.addSql("    left join");
            sql.addSql("      CHT_GROUP_DATA");
            sql.addSql("    on");
            sql.addSql("      CHT_GROUP_VIEW.CGI_SID = CHT_GROUP_DATA.CGI_SID");
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    inner join");
                sql.addSql("      CHT_GROUP_DATA_PIN");
                sql.addSql("    on");
                sql.addSql("      CHT_GROUP_DATA_PIN.CGI_SID = CHT_GROUP_DATA.CGI_SID");
                sql.addSql("      and CHT_GROUP_DATA_PIN.CGD_SID = CHT_GROUP_DATA.CGD_SID");
            }
            sql.addSql("    where");
            sql.addSql("      CHT_GROUP_VIEW.CGD_SID >= CHT_GROUP_DATA.CGD_SID");
            sql.addSql("    and");
            sql.addSql("      CHT_GROUP_VIEW.CGI_SID = ?");
            sql.addIntValue(searchReq.getChatGroupSid());
            sql.addSql("    and");
            sql.addSql("      CHT_GROUP_VIEW.CGV_UID = ?");
            sql.addIntValue(searchReq.getSessionUserSid());

            if (searchReq.getPosition()
            instanceof MidokuPosition) {
                __writeSearchGroupChatWhere(sql,
                searchReq.getFilterList(),
                searchReq.getChatGroupSid());

            }
            sql.addSql("    group by");
            sql.addSql("      CHT_GROUP_VIEW.CGD_SID,");
            sql.addSql("      CHT_GROUP_VIEW.CGI_SID");
            sql.addSql("   )VIEW");
            sql.addSql("   on");
            sql.addSql("     DATA.CGI_SID = VIEW.CGI_SID");



            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret.setViewSid(rs.getLong("VIEW_SID"));
                ret.setViewCnt(rs.getInt("VIEW_CNT"));
                ret.setHitCnt(rs.getInt("DATA_CNT"));
                ret.setTargetMessageOffset(rs.getInt("TARGET_OFFSET"));
                ret.setTargetMessageHave(rs.getInt("TARGET_HAVE"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <p>グループチャットメッセージ情報の取得
     * @param viewSid 閲覧SID
     * @param searchReq 検索設定
     * @return CHT_USER_DATAModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<ChatMessageModel> searchGroupMessage(
            Long viewSid, ChatMessageSearchRequest searchReq) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<ChatMessageModel> ret = new ArrayList<ChatMessageModel>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            getGroupMessage(sql, searchReq.getTargetPinKbn());

            sql.addSql(" where ");
            sql.addSql("   CHT_GROUP_DATA.CGI_SID=?");
            sql.addLongValue(searchReq.getChatGroupSid());
            //ピンどめ一覧取得時用
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_PIN) {
                sql.addSql("    and");
                sql.addSql("      CHT_GROUP_DATA_PIN.USR_SID = ?");
                sql.addIntValue(searchReq.getSessionUserSid());
            }
            if (searchReq.getPosition()
                    instanceof BeforeMessage) {
                    BeforeMessage pos
                         = (BeforeMessage) searchReq.getPosition();
                    sql.addSql(" and  CHT_GROUP_DATA.CGD_SID < ?");
                    sql.addLongValue(pos.getMessageSid());

            }
            if (searchReq.getPosition()
                    instanceof AfterMessage) {
                    AfterMessage pos
                        = (AfterMessage) searchReq.getPosition();
                    sql.addSql(" and  CHT_GROUP_DATA.CGD_SID > ?");
                    sql.addLongValue(pos.getMessageSid());
            }

            __writeSearchGroupChatWhere(sql,
                searchReq.getFilterList(),
                searchReq.getChatGroupSid());


            int order = searchReq.getOrder();
            if (searchReq.getPosition()
                    instanceof BeforeMessage) {
                order = GSConst.ORDER_KEY_DESC;
            }
            if (searchReq.getPosition()
                instanceof AfterMessage) {
                order = GSConst.ORDER_KEY_ASC;
            }
            if (searchReq.getPosition()
                instanceof NewestPosition) {
                order = GSConst.ORDER_KEY_DESC;
            }

            sql.addSql(" order by ");
            if (searchReq.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_ALL) {
                if (order == GSConst.ORDER_KEY_ASC) {
                    sql.addSql("   CHT_GROUP_DATA.CGD_SID ");

                } else {
                    sql.addSql("   CHT_GROUP_DATA.CGD_SID desc ");
                }
            } else {
                //ピンどめ並び順
                sql.addSql("   CHT_GROUP_DATA_PIN.CGDP_SORT desc ");
            }
            int offset = 0;
            if (searchReq.getPosition()
                    instanceof OffsetPosition) {
                        OffsetPosition pos
                        = (OffsetPosition) searchReq.getPosition();
                offset = pos.getOffset();
            }

            sql.setPagingValue(offset, searchReq.getLimit());


            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret.add(__getChatGrpMessageModel(rs, viewSid));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        ret = __setTempInfo(ret, GSConstChat.CHAT_KBN_GROUP);
        __setGroupReactionInfo(ret);
        __setReplyMessage(ret, GSConstChat.CHAT_KBN_GROUP);
        __setMentionInfo(ret, GSConstChat.CHAT_KBN_GROUP);
        __setMessagePinInfo(ret, GSConstChat.CHAT_KBN_GROUP,
        searchReq.getChatGroupSid(), searchReq.getSessionUserSid());
        __setOutMemberFlg(ret, searchReq.getChatGroupSid(), GSConstChat.CHAT_KBN_GROUP);


        return ret;
    }

    /***
     * 検索条件を書き込み
     * @param sql
     * @param filterList
     * @param chatGroupSid
     */
    private void __writeSearchGroupChatWhere(SqlBuffer sql,
            List<ChatMessageSearchFilter> filterList, int chatGroupSid) {
        if (filterList == null
            || filterList.size() <= 0) {
            return;
        }
        //検索条件振り分け
        List<ChatMessageSearchFilter> keywordFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> attachmentFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> urlFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> senderFilters = new ArrayList<>();
        List<ChatMessageSearchFilter> messageFilters = new ArrayList<>();
        boolean deleteUser = false;
        boolean deletedMessageLess = false;
        if (filterList.size() > 0) {
            for (ChatMessageSearchFilter filter : filterList) {
                switch (filter.getMode()) {
                    case keyword:
                        keywordFilters.add(filter);
                        break;
                    case attachment:
                    case attachmentAll:
                    case attachmentLess:
                        attachmentFilters.add(filter);
                        break;
                    case url:
                    case urlAll:
                    case urlLess:
                        urlFilters.add(filter);
                        break;
                    case user:
                        senderFilters.add(filter);
                        break;
                    case outMember:
                        deleteUser = true;
                        break;
                    case deletedLess:
                        deletedMessageLess = true;
                        break;
                    case message:
                        messageFilters.add(filter);
                        break;
                    default:
                }
            }
        }
        //キーワード検索
        for (ChatMessageSearchFilter filter : keywordFilters) {
            sql.addSql(" and (");
            sql.addSql("   CHT_GROUP_DATA.CGD_TEXT like '%"
            + JDBCUtil.escapeForLikeSearch(filter.getValue())
            + "%' ESCAPE '"
            + JDBCUtil.def_esc
            + "'");
            sql.addSql(" or ");
            sql.addSql("   CHT_GROUP_DATA.CGD_SID in (");

            sql.addSql(" select");
            sql.addSql("   ATTACHED.CGD_SID");
            sql.addSql(" from");
            sql.addSql("  CHT_GROUP_DATA ATTACHED");
            sql.addSql("  inner join CHT_GROUP_DATA_TEMP ATTACHBELONG");
            sql.addSql("    on ATTACHED.CGD_SID = ATTACHBELONG.CGD_SID");
            sql.addSql("  inner join CMN_BINF ATTACH");
            sql.addSql("    on ATTACHBELONG.BIN_SID = ATTACH.BIN_SID");
            sql.addSql(" where ");
            sql.addSql("   ATTACHED.CGI_SID = ?");
            sql.addSql("     and ATTACH.BIN_FILE_NAME like '%");
            sql.addSql(JDBCUtil.escapeForLikeSearch(filter.getValue()));
            sql.addSql("%' ESCAPE '");
            sql.addSql(JDBCUtil.def_esc);
            sql.addSql("'");
            sql.addSql(" ) ");
            sql.addIntValue(chatGroupSid);

            sql.addSql(" )");
        }

        //添付ファイル検索
        for (ChatMessageSearchFilter filter : attachmentFilters) {
            sql.addSql(" and ");
            if (filter.getMode() == ChatMessageSearchFilter.EnumType.attachmentLess) {
                sql.addSql("not exists (");
                sql.addSql(" select");
                sql.addSql("   1");
                sql.addSql(" from");
                sql.addSql("  CHT_GROUP_DATA_TEMP ATTACHBELONG");
                sql.addSql(" where ");
                sql.addSql("   ATTACHBELONG.CGD_SID = CHT_GROUP_DATA.CGD_SID");
                sql.addSql(" ) ");
            } else {
                sql.addSql("   CHT_GROUP_DATA.CGD_SID in (");

                sql.addSql(" select");
                sql.addSql("   ATTACHED.CGD_SID");
                sql.addSql(" from");
                sql.addSql("  CHT_GROUP_DATA ATTACHED");
                sql.addSql("  inner join CHT_GROUP_DATA_TEMP ATTACHBELONG");
                sql.addSql("    on ATTACHED.CGD_SID = ATTACHBELONG.CGD_SID");
                sql.addSql("  inner join CMN_BINF ATTACH");
                sql.addSql("    on ATTACHBELONG.BIN_SID = ATTACH.BIN_SID");
                sql.addSql(" where ");
                sql.addSql("   ATTACHED.CGI_SID = ?");
                sql.addIntValue(chatGroupSid);
                if (filter.getMode() != ChatMessageSearchFilter.EnumType.attachmentAll) {
                    sql.addSql("     and ATTACH.BIN_FILE_NAME like '%");
                    sql.addSql(JDBCUtil.escapeForLikeSearch(filter.getValue()));
                    sql.addSql("%' ESCAPE '");
                    sql.addSql(JDBCUtil.def_esc);
                    sql.addSql("'");
                }
                sql.addSql(" ) ");
            }
        }

        //URL検索
        for (ChatMessageSearchFilter filter : urlFilters) {
            if (filter.getMode() == ChatMessageSearchFilter.EnumType.urlAll) {
                sql.addSql(" and ");
                sql.addSql("   CHT_GROUP_DATA.CGD_URL_FLG = 1");
            } else if (filter.getMode() == ChatMessageSearchFilter.EnumType.urlLess) {
                sql.addSql(" and ");
                sql.addSql("   CHT_GROUP_DATA.CGD_URL_FLG = 0");
            } else {
                sql.addSql(" and (");
                sql.addSql("   CHT_GROUP_DATA.CGD_TEXT like '%http:%' ESCAPE '"
                + JDBCUtil.def_esc
                + "'");
                sql.addSql("   OR CHT_GROUP_DATA.CGD_TEXT like '%https:%' ESCAPE '"
                + JDBCUtil.def_esc
                + "'");
                sql.addSql(" ) ");
            }
        }

        //ユーザ検索
        if (senderFilters.size() > 0 || deleteUser) {
            sql.addSql(" and (");
            boolean first = true;
            for (ChatMessageSearchFilter filter : senderFilters) {
                if (!first) {
                    sql.addSql(" or ");
                }
                first = false;
                sql.addSql("   CHT_GROUP_DATA.CGD_SEND_UID = ?");
                sql.addIntValue(NullDefault.getInt(filter.getValue(), -1));
            }
            if (deleteUser) {
                if (senderFilters.size() > 0) {
                    sql.addSql(" or ");
                }
                sql.addSql(" ( ");
                sql.addSql("   CHT_GROUP_DATA.CGD_SEND_UID not in (");
                sql.addSql(" select");
                sql.addSql("  CMN_BELONGM.USR_SID ");
                sql.addSql("   ");
                sql.addSql(" from");
                sql.addSql("  CHT_GROUP_USER");
                sql.addSql("   inner join CMN_BELONGM");
                sql.addSql("   on CHT_GROUP_USER.CGU_SELECT_SID = CMN_BELONGM.GRP_SID");
                sql.addSql("     and CHT_GROUP_USER.CGU_KBN = ?");
                sql.addIntValue(GSConstChat.CHAT_KBN_GROUP);
                sql.addSql(" where ");
                sql.addSql("  CHT_GROUP_USER.CGI_SID = ?");
                sql.addIntValue(chatGroupSid);
                sql.addSql("     and CHT_GROUP_USER.CGU_KBN = ?");
                sql.addIntValue(GSConstChat.CHAT_KBN_GROUP);
                sql.addSql(" ) ");

                sql.addSql(" and ");
                sql.addSql("   CHT_GROUP_DATA.CGD_SEND_UID not in (");
                sql.addSql(" select");
                sql.addSql("   CHT_GROUP_USER.CGU_SELECT_SID ");
                sql.addSql(" from");
                sql.addSql("  CHT_GROUP_USER");
                sql.addSql(" where ");
                sql.addSql("  CHT_GROUP_USER.CGI_SID = ?");
                sql.addIntValue(chatGroupSid);
                sql.addSql("     and CHT_GROUP_USER.CGU_KBN = ?");
                sql.addIntValue(GSConstChat.CHAT_KBN_USER);
                sql.addSql(" ) ");
                sql.addSql(" ) ");

            }
            sql.addSql(" )");
        }
        //削除済みメッセージを除外
        if (deletedMessageLess) {
            sql.addSql(" and ");
            sql.addSql("   CHT_GROUP_DATA.CGD_STATE_KBN <> ?");
            sql.addIntValue(GSConst.JTKBN_DELETE);
        }
        //メッセージSID検索
        if (messageFilters.size() > 0) {
            sql.addSql(" and (");
            boolean first = true;

            for (ChatMessageSearchFilter filter : messageFilters) {
                if (!first) {
                    sql.addSql(" or ");
                }
                first = false;
                sql.addSql("   CHT_GROUP_DATA.CGD_SID = ?");
                sql.addIntValue(NullDefault.getInt(filter.getValue(), -1));
            }
            sql.addSql(" )");
        }
    }

    /**
     * <br>[機  能] メッセージに所属メンバー内フラグの情報を設定します。
     * <br>[解  説]
     * <br>[備  考]
     * @param cmList
     * @param cgiSid
     * @param chatKbn
     * @throws SQLException
     */
    private void __setOutMemberFlg(List<ChatMessageModel> cmList,
            int cgiSid,
            int chatKbn) throws SQLException {
        if (chatKbn == GSConstChat.CHAT_KBN_USER) {
            for (ChatMessageModel cm : cmList) {
                cm.setInMemberKbn(GSConstChat.SENDER_MEMBER_YES);
            }
            return;
        }

        Connection con = getCon();
        ChtGroupUserDao guDao = new ChtGroupUserDao(con);
        List<ChtGroupUserModel> uList = guDao.select(cgiSid);

        Set<Integer> usrSids = new HashSet<Integer>();
        usrSids.addAll(
            uList.stream()
                    .filter(u -> u.getCguKbn() == GSConstChat.CHAT_KBN_USER)
                    .map(u -> u.getCguSelectSid())
                    .collect(Collectors.toSet())
        );
        String[] gsids = uList.stream()
        .filter(u -> u.getCguKbn() == GSConstChat.CHAT_KBN_GROUP)
        .map(u -> String.valueOf(u.getCguSelectSid()))
        .toArray(String[]::new);
        if (gsids.length > 0) {
            usrSids.addAll(
                new CmnBelongmDao(con)
                .select(
                    gsids
                )
                .stream()
                .map(uStr -> Integer.valueOf(uStr))
                .collect(Collectors.toSet())
            );
        }
        //削除済みユーザを除外
        CmnUsrmDao cuDao = new CmnUsrmDao(con);
        usrSids =
        Stream.of(
            cuDao.getNoDeleteUser(
                usrSids.stream()
                    .map(sid -> String.valueOf(sid))
                    .toArray(String[]::new)
            )
        )
        .map(str -> Integer.parseInt(str))
        .collect(Collectors.toSet());

        for (ChatMessageModel cm : cmList) {
            cm.setInMemberKbn(GSConstChat.SENDER_MEMBER_YES);
            if (usrSids.contains(cm.getUsrSid()) == false) {
                cm.setInMemberKbn(GSConstChat.SENDER_MEMBER_NO);
            }
        }
    }

    /**
     * <p>指定したメッセージSIDのユーザ間チャット送信者と送信日付を取得する
     * <p>返信元の投稿情報を取得する際に使用する
     * @param messageSidList 返信元投稿SID
     * @return メッセージSID:メッセージ情報を格納したマップ
     * @throws SQLException SQL実行例外
     */
    public Map<Long, ChatMessageModel> getGroupReplyMap(
        List<Long> messageSidList) throws SQLException {

        Map<Long, ChatMessageModel> ret = new HashMap<Long, ChatMessageModel>();
        if (messageSidList == null || messageSidList.isEmpty()) {
            return ret;
        }

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        con = getCon();
        ArrayList<ChatMessageModel> cmList = new ArrayList<ChatMessageModel>();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            getGroupMessage(sql, GSConstChat.CHAT_MESSAGE_GET_ALL);
            sql.addSql(" where ");
            sql.addSql("   CHT_GROUP_DATA.CGD_SID in ( ");
            for (int idx = 0; idx < messageSidList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql(", ");
                }
                sql.addSql("?");
                sql.addLongValue(messageSidList.get(idx));
            }
            sql.addSql("   ) ");

            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                cmList.add(__getChatGrpMessageReplyModel(rs));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        __setTempInfo(cmList, GSConstChat.CHAT_KBN_GROUP);
        for (ChatMessageModel cmMdl : cmList) {
            ret.put(cmMdl.getMessageSid(), cmMdl);
        }

        return ret;
    }

    /**
     * <p>Create CHT_GROUP_DATA Data Bindding JavaBean From ResultSet
     * @param rs ResultSet
     * @param viewSid viewSid
     * @return created ChtGroupDataModel
     * @throws SQLException SQL実行例外
     */
    private ChatMessageModel __getChatGrpMessageModel(ResultSet rs,
            Long viewSid) throws SQLException {

        ChatMessageModel cmMdl = new ChatMessageModel();
        cmMdl.setMessageSid(rs.getLong("CGD_SID"));
        if (viewSid >= rs.getInt("CGD_SID")) {
            //自身の既読情報登録
            cmMdl.setOwnKidoku(GSConstChat.KIDOKU_DISP);
        }
        cmMdl.setMessageKbn(rs.getInt("CGD_STATE_KBN"));

        UDate date = UDate.getInstanceTimestamp(rs.getTimestamp("CGD_ADATE"));
        cmMdl.setEntryDay(date.getStrYear() + "/"
        + date.getStrMonth() + "/" + date.getStrDay());
        cmMdl.setEntryTime(date.getStrHour() + ":" + date.getStrMinute());

        UDate uDate = UDate.getInstanceTimestamp(rs.getTimestamp("CGD_EDATE"));
        cmMdl.setUpdateDay(
            uDate.getStrYear()
            + "/" + uDate.getStrMonth()
            + "/" + uDate.getStrDay());
        cmMdl.setUpdateTime(uDate.getStrHour() + ":" + uDate.getStrMinute());

        //削除された投稿の場合は、画面描画に必要ない情報を設定しない
        if (cmMdl.getMessageKbn() == GSConstChat.TOUKOU_STATUS_DELETE) {
            return cmMdl;
        }

        cmMdl.setSelectSid(rs.getInt("CGI_SID"));
        cmMdl.setMessageText(rs.getString("CGD_TEXT"));
        cmMdl.setStampSid(rs.getInt("CST_SID"));
        cmMdl.setStampBinSid(rs.getInt("STAMP_BIN"));
        cmMdl.setStampDefaultId(rs.getInt("CST_DEFSTAMP_ID"));
        cmMdl.setInsertUid(rs.getInt("CGD_AUID"));
        cmMdl.setUpdateUid(rs.getInt("CGD_EUID"));
        cmMdl.setUsrSid(rs.getInt("CGD_SEND_UID"));
        cmMdl.setUsrName(rs.getString("USI_SEI") + " " + rs.getString("USI_MEI"));
        cmMdl.setUsrId(rs.getString("USR_LGID"));
        cmMdl.setUsrJkbn(rs.getInt("USR_JKBN"));
        cmMdl.setUsrUkoFlg(rs.getInt("USR_UKO_FLG"));
        cmMdl.setUsrBinSid(rs.getInt("USR_BIN"));
        cmMdl.setUsrPictKf(rs.getInt("USI_PICT_KF"));
        cmMdl.setReplyMessageSid(rs.getLong("CGD_REPLY_SID"));
        return cmMdl;
    }

    /**
     * <p>SQL実行結果から、返信元投稿データを作成する
     * <p>返信元投稿は表示する内容のみを格納する
     * @param rs ResultSet
     * @return created ChtGroupDataModel
     * @throws SQLException SQL実行例外
     */
    private ChatMessageModel __getChatGrpMessageReplyModel(ResultSet rs) throws SQLException {

        ChatMessageModel cmMdl = new ChatMessageModel();
        cmMdl.setMessageSid(rs.getLong("CGD_SID"));
        cmMdl.setMessageKbn(rs.getInt("CGD_STATE_KBN"));
        cmMdl.setMessageText(rs.getString("CGD_TEXT"));
        cmMdl.setUsrSid(rs.getInt("CGD_SEND_UID"));
        cmMdl.setUsrName(UserUtil.makeName(rs.getString("USI_SEI"), rs.getString("USI_MEI")));
        cmMdl.setUsrId(rs.getString("USR_LGID"));
        cmMdl.setUsrJkbn(rs.getInt("USR_JKBN"));
        cmMdl.setUsrUkoFlg(rs.getInt("USR_UKO_FLG"));
        cmMdl.setUsrBinSid(rs.getInt("USR_BIN"));
        cmMdl.setUsrPictKf(rs.getInt("USI_PICT_KF"));
        cmMdl.setStampSid(rs.getInt("CST_SID"));
        cmMdl.setStampBinSid(rs.getInt("STAMP_BIN"));
        cmMdl.setStampDefaultId(rs.getInt("CST_DEFSTAMP_ID"));

        return cmMdl;
    }

    /**
     * ユーザ情報を取得する
     * @param usrSid ペアSID
     * @return CmnUsrmInfModel　情報クラス
     * @throws SQLException SQL実行例外
     */
    public CmnUsrmInfModel getUser(int usrSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        CmnUsrmInfModel ret = null;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   CMN_USRM_INF.USR_SID,");
            sql.addSql("   CMN_USRM_INF.USI_SEI,");
            sql.addSql("   CMN_USRM_INF.USI_MEI,");
            sql.addSql("   CMN_USRM.USR_JKBN");
            sql.addSql(" from ");
            sql.addSql("   CMN_USRM_INF");
            sql.addSql(" left join ");
            sql.addSql("   CMN_USRM");
            sql.addSql(" on ");
            sql.addSql("   CMN_USRM_INF.USR_SID = CMN_USRM.USR_SID");
            sql.addSql(" where ");
            sql.addSql("   CMN_USRM_INF.USR_SID = ?");

            sql.addIntValue(usrSid);
            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret = new CmnUsrmInfModel();
                ret.setUsrSid(rs.getInt("USR_SID"));
                ret.setUsiSei(rs.getString("USI_SEI"));
                ret.setUsiMei(rs.getString("USI_MEI"));
                ret.setUsrJkbn(rs.getInt("USR_JKBN"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * ユーザ情報を取得する
     * @param usrList ユーザリスよ
     * @return CmnUsrmInfModel　情報クラス
     * @throws SQLException SQL実行例外
     */
    public ArrayList<CmnUsrmInfModel> getUser(ArrayList<Integer> usrList) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<CmnUsrmInfModel> ret = new ArrayList<CmnUsrmInfModel>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   CMN_USRM_INF.USR_SID,");
            sql.addSql("   CMN_USRM_INF.USI_SEI,");
            sql.addSql("   CMN_USRM_INF.USI_MEI,");
            sql.addSql("   CMN_USRM.USR_JKBN");
            sql.addSql(" from ");
            sql.addSql("   CMN_USRM_INF");
            sql.addSql(" left join ");
            sql.addSql("   CMN_USRM");
            sql.addSql(" on ");
            sql.addSql("   CMN_USRM_INF.USR_SID = CMN_USRM.USR_SID");
            sql.addSql(" where ");
            sql.addSql("   CMN_USRM_INF.USR_SID IN (");
            for (int idx = 0; idx < usrList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql(" , ");
                }
                sql.addSql(" ? ");
                sql.addIntValue(usrList.get(idx));
            }
            sql.addSql("   )");
            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                CmnUsrmInfModel cuiMdl = new CmnUsrmInfModel();
                cuiMdl.setUsrSid(rs.getInt("USR_SID"));
                cuiMdl.setUsiSei(rs.getString("USI_SEI"));
                cuiMdl.setUsiMei(rs.getString("USI_MEI"));
                cuiMdl.setUsrJkbn(rs.getInt("USR_JKBN"));
                ret.add(cuiMdl);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * グループ情報情報を取得する
     * @param groupSid グループSID
     * @return ChatInformationModel　情報クラス
     * @throws SQLException SQL実行例外
     */
    public ChatInformationModel getGroupInfoData(int groupSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ChatInformationModel ret = new ChatInformationModel();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   CHT_GROUP_INF.CGI_SID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_ID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_NAME, ");
            sql.addSql("   CHT_GROUP_INF.CGI_CONTENT, ");
            sql.addSql("   CHT_GROUP_INF.CGI_COMP_FLG, ");
            sql.addSql("   CHT_GROUP_INF.CHC_SID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_ADATE, ");
            sql.addSql("   CHT_CATEGORY.CHC_NAME ");
            sql.addSql(" from ");
            sql.addSql("   CHT_GROUP_INF");
            sql.addSql(" left join ");
            sql.addSql("   CHT_CATEGORY");
            sql.addSql(" on ");
            sql.addSql("   CHT_GROUP_INF.CHC_SID = CHT_CATEGORY.CHC_SID");
            sql.addSql(" where ");
            sql.addSql("   CHT_GROUP_INF.CGI_SID = ?");

            sql.addIntValue(groupSid);
            pstmt = con.prepareStatement(sql.toSqlString());
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret.setChatSid(rs.getInt("CGI_SID"));
                ret.setChatName(rs.getString("CGI_NAME"));
                ret.setChatArchive(rs.getInt("CGI_COMP_FLG"));
                ret.setCategorySid(rs.getInt("CHC_SID"));
                ret.setCategoryName(rs.getString("CHC_NAME"));
                ret.setChatId(rs.getString("CGI_ID"));
                ret.setChatBiko(
                        StringUtilHtml.transToHTmlPlusAmparsant(rs.getString("CGI_CONTENT")));
                ret.setInsertDate(UDate.getInstanceTimestamp(rs.getTimestamp("CGI_ADATE")));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <p>Select CHT_GROUP_USER
     * @param groupSid CGI_SID
     * @param adminFlg 0:一般ユーザ 1:管理者ユーザ
     * @return CHT_GROUP_USERModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<CmnUsrmInfModel> getGroupUser(int groupSid, int adminFlg) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<CmnUsrmInfModel> ret = new ArrayList<CmnUsrmInfModel>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   CMN_USRM_INF.USR_SID,");
            sql.addSql("   CMN_USRM_INF.USI_SEI,");
            sql.addSql("   CMN_USRM_INF.USI_MEI,");
            sql.addSql("   CMN_USRM.USR_JKBN,");
            sql.addSql("   CMN_USRM.USR_UKO_FLG");
            sql.addSql(" from");
            sql.addSql("   CMN_USRM_INF");
            sql.addSql(" left join");
            sql.addSql("   CMN_USRM");
            sql.addSql(" on");
            sql.addSql("   CMN_USRM_INF.USR_SID = CMN_USRM.USR_SID");
            sql.addSql(" where ");
            sql.addSql("   CMN_USRM_INF.USR_SID IN (");
            sql.addSql("     select");
            sql.addSql("       CGU_SELECT_SID");
            sql.addSql("     from");
            sql.addSql("       CHT_GROUP_USER");
            sql.addSql("     where");
            sql.addSql("       CGI_SID = ?");
            sql.addSql("     and");
            sql.addSql("       CGU_ADMIN_FLG = ?");
            sql.addSql("     and");
            sql.addSql("       CGU_KBN = ?");
            sql.addSql("     and");
            sql.addSql("       CMN_USRM.USR_JKBN = ?");
            sql.addSql("   )");

            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addIntValue(groupSid);
            sql.addIntValue(adminFlg);
            sql.addIntValue(GSConstChat.CHAT_KBN_USER);
            sql.addIntValue(GSConst.JTKBN_TOROKU);

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                CmnUsrmInfModel usrMdl = new CmnUsrmInfModel();
                usrMdl.setUsrSid(rs.getInt("USR_SID"));
                usrMdl.setUsiSei(rs.getString("USI_SEI"));
                usrMdl.setUsiMei(rs.getString("USI_MEI"));
                usrMdl.setUsrJkbn(rs.getInt("USR_JKBN"));
                usrMdl.setUsrUkoFlg(rs.getInt("USR_UKO_FLG"));
                ret.add(usrMdl);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <p>Select CHT_GROUP_USER
     * @param groupSid CGI_SID
     * @param adminFlg 0:一般ユーザ 1:管理者ユーザ
     * @return CHT_GROUP_USERModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<CmnGroupmModel> getGroup(int groupSid, int adminFlg) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<CmnGroupmModel> ret = new ArrayList<CmnGroupmModel>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   CMN_GROUPM.GRP_SID,");
            sql.addSql("   CMN_GROUPM.GRP_ID,");
            sql.addSql("   CMN_GROUPM.GRP_NAME,");
            sql.addSql("   CMN_GROUPM.GRP_JKBN");
            sql.addSql(" from");
            sql.addSql("   CMN_GROUPM");
            sql.addSql(" where ");
            sql.addSql("   CMN_GROUPM.GRP_SID IN (");
            sql.addSql("     select");
            sql.addSql("       CGU_SELECT_SID");
            sql.addSql("     from");
            sql.addSql("       CHT_GROUP_USER");
            sql.addSql("     where");
            sql.addSql("       CGI_SID = ?");
            sql.addSql("     and");
            sql.addSql("       CGU_ADMIN_FLG = ?");
            sql.addSql("     and");
            sql.addSql("       CGU_KBN = ?");
            sql.addSql("   )");

            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addIntValue(groupSid);
            sql.addIntValue(adminFlg);
            sql.addIntValue(GSConstChat.CHAT_KBN_GROUP);

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                CmnGroupmModel grpMdl = new CmnGroupmModel();
                grpMdl.setGrpSid(rs.getInt("GRP_SID"));
                grpMdl.setGrpId(rs.getString("GRP_ID"));
                grpMdl.setGrpName(rs.getString("GRP_NAME"));
                grpMdl.setGrpJkbn(rs.getInt("GRP_JKBN"));
                ret.add(grpMdl);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <p>Select deleteUserSid
     * @param userSid ユーザSID
     * @return CHT_GROUP_USERModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<CmnUsrmInfModel> getDeleteUser(int userSid) throws SQLException {

        ArrayList<CmnUsrmInfModel> ret = new ArrayList<CmnUsrmInfModel>();
        ArrayList<Integer> sidList = deleteUserCount(userSid);
        if (sidList.size() != 0) {
            ret = getUser(sidList);
        }
        return ret;
    }

    /**
     * <p>Select deleteUserSid
     * @param userSid ユーザSID
     * @return CHT_GROUP_USERModel
     * @throws SQLException SQL実行例外
     */
    public ArrayList<Integer> deleteUserCount(int userSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<Integer> ret = new ArrayList<Integer>();
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql("  select");
            sql.addSql("    CUP_TABLE.USR_SID");
            sql.addSql("  from");
            sql.addSql("    (select");
            sql.addSql("       case ");
            sql.addSql("         when CHT_USER_PAIR.CUP_UID_F = ? then CHT_USER_PAIR.CUP_UID_S");
            sql.addSql("         else CHT_USER_PAIR.CUP_UID_F");
            sql.addSql("       end as USR_SID");
            sql.addSql("     from");
            sql.addSql("       CHT_USER_PAIR");
            sql.addSql("     where");
            sql.addSql("       (");
            sql.addSql("          CHT_USER_PAIR.CUP_UID_F = ?");
            sql.addSql("        or");
            sql.addSql("          CHT_USER_PAIR.CUP_UID_S = ?");
            sql.addSql("       )");
            sql.addSql("    ) CUP_TABLE");
            sql.addSql("  left join");
            sql.addSql("    CMN_USRM");
            sql.addSql("  on");
            sql.addSql("    CUP_TABLE.USR_SID = CMN_USRM.USR_SID");
            sql.addSql("  where");
            sql.addSql("    CMN_USRM.USR_JKBN = ?");

            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addIntValue(userSid);
            sql.addIntValue(userSid);
            sql.addIntValue(userSid);
            sql.addIntValue(GSConst.JTKBN_DELETE);
            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret.add(rs.getInt("USR_SID"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     *
     * <br>[機  能]未読件数を返す
     * <br>[解  説]
     * <br>[備  考]
     * @param usrSid ユーザSID
     * @return ret カウント
     * @throws SQLException SQLException
     */
    public int getMidokuCount(int usrSid) throws SQLException {

        int ret = 0;
        int groupCount = getGroupMidokuCount(usrSid);
        int userCount = getUserMidokuCount(usrSid);
        ret = groupCount + userCount;
        return ret;
    }

    /**
     *
     * <br>[機  能] タイムライン一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param usrSid ユーザSID
     * @param startIdx 検索開始位置
     * @param onlyNoRead 未読のみ表示
     * @param limit 取得件数
     * @return ret ChatMidokuModelList
     * @throws SQLException SQLException
     */
    public ArrayList<ChatMidokuModel> getTimelineList(
            int usrSid, UDate startIdx, boolean onlyNoRead, int limit)
            throws SQLException {

        ArrayList<ChatMidokuModel> ret = new ArrayList<ChatMidokuModel>();

        ArrayList<ChatMidokuModel> groupList =
            getGroupTimelineList(usrSid, startIdx, onlyNoRead, limit);

        ArrayList<ChatMidokuModel> userList =
            getUserTimelineList(usrSid, startIdx, onlyNoRead, limit);

        //startIdxから最新10件取得する
        boolean check = true;
        UDate chkDate = null;

        Iterator<ChatMidokuModel> itGroup = groupList.iterator();
        Iterator<ChatMidokuModel> itUser = userList.iterator();

        ChatMidokuModel stkUser = null;
        ChatMidokuModel stkGroup = null;

        while (check) {
            //ストックもイテレータもなければ走査終了
            if (!itGroup.hasNext() && !itUser.hasNext()
                    && stkUser == null && stkGroup == null) {
                check = false;
                break;
            }

            ChatMidokuModel chkUser = null;
            ChatMidokuModel chkGroup = null;
            //ストックがなければイテレータから取り出す
            if (stkGroup != null) {
                chkGroup = stkGroup;
            } else if (itGroup.hasNext()) {
                chkGroup = itGroup.next();
            }
            if (stkUser != null) {
                chkUser = stkUser;
            } else if (itUser.hasNext()) {
                chkUser = itUser.next();
            }
            stkGroup = chkGroup;
            stkUser = chkUser;
            //基準日がない場合基準日を設定
            if (chkDate == null) {
                if (chkUser == null) {
                    chkDate = chkGroup.getMidokuDate();
                } else if (chkGroup == null) {
                    chkDate = chkUser.getMidokuDate();
                } else if (chkGroup.getMidokuDate().compare(
                        chkGroup.getMidokuDate(), chkUser.getMidokuDate()) >= 0) {
                    chkDate = chkUser.getMidokuDate();
                } else {
                    chkDate = chkGroup.getMidokuDate();
                }
            }

            //タイムラインが基準日と一致する限り連続して同じイテレータから取り出す
            if (chkUser != null
                    && chkDate.equalsTimeStamp(chkUser.getMidokuDate())) {
                ret.add(chkUser);
                stkUser = null;
            } else if (chkGroup != null
                    && chkDate.equalsTimeStamp(chkGroup.getMidokuDate())) {
                ret.add(chkGroup);
                stkGroup = null;
            //基準日と一致するタイムラインがなければ次の基準日へ進めるためchkDateを空に
            } else {
                chkDate = null;
                //タイムラインのリミットを超えていればループ終了
                if (ret.size() >= limit) {
                    check = false;
                }
            }

        }
        return ret;
    }

    /**
    *
    * <br>[機  能] 相手別タイムライン件数を取得する
    * <br>[解  説]
    * <br>[備  考]
    * @param usrSid ユーザSID
    * @param startIdx 検索開始位置
    * @param onlyNoRead 未読のみ表示
    * @return ret ChatMidokuModelList
    * @throws SQLException SQLException
    */
   public int getTimelineListCount(int usrSid, UDate startIdx, boolean onlyNoRead)
           throws SQLException {

       int ret = 0;
       int groupCnt = getGroupTimelineListCount(usrSid, startIdx, onlyNoRead);
       int userCnt = getUserTimelineListCount(usrSid, startIdx, onlyNoRead);
       ret = groupCnt + userCnt;
       return ret;
   }

    /**
     * <br>[機  能] 未読グループ一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param usrSid ユーザSID
     * @param startIdx 検索開始位置
     * @param onlyNoRead 未読のみ表示
     * @param limit
     * @return グループ情報一覧の件数
     * @throws SQLException SQL実行例外
     */
    public ArrayList<ChatMidokuModel> getGroupTimelineList(
            int usrSid, UDate startIdx, boolean onlyNoRead, int limit)
            throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<ChatMidokuModel> ret = new ArrayList<ChatMidokuModel>();
        con = getCon();

        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   GROUP_DATA.GROUP_SID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_NAME, ");
            sql.addSql("   CHT_GROUP_INF.CGI_ID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_COMP_FLG, ");
            sql.addSql("   GROUP_DATA.CNT, ");
            sql.addSql("   GROUP_DATA.LAST_DATE, ");
            sql.addSql("   GROUP_DATA.CGUC_MUTE_KBN, ");
            sql.addSql("   GROUP_DATA.CGUC_FAVORITE_KBN ");
            sql.addSql(" from");
            sql.addSql("   (");
            writeQueryGroupTimeline(sql, usrSid);
            sql.addSql("   ) GROUP_DATA");
            sql.addSql(" left join ");
            sql.addSql("   CHT_GROUP_INF");
            sql.addSql(" on ");
            sql.addSql("   GROUP_DATA.GROUP_SID = CHT_GROUP_INF.CGI_SID");
            sql.addSql(" where");
            sql.addSql("   GROUP_DATA.LAST_DATE < ?");
            if (onlyNoRead) {
                sql.addSql(" and");
                sql.addSql("   GROUP_DATA.CNT > 0");
            }
            sql.addSql(" and");
            sql.addSql("   CHT_GROUP_INF.CGI_DEL_FLG = ?");
            sql.addSql(" order by");
            sql.addSql("   GROUP_DATA.LAST_DATE desc");

            // 検索条件
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addDateValue(startIdx);
            sql.addIntValue(GSConstChat.CHAT_MODE_ADD);
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();

            UDate chkDate = null;
            while (rs.next()) {
                ChatMidokuModel cmMdl = new ChatMidokuModel();
                cmMdl.setMidokuCount(rs.getInt("CNT"));
                cmMdl.setMidokuDate(UDate.getInstanceTimestamp(rs.getTimestamp("LAST_DATE")));
                cmMdl.setMidokuKbn(GSConstChat.CHAT_KBN_GROUP);
                cmMdl.setMidokuName(rs.getString("CGI_NAME"));
                cmMdl.setArchiveFlg(rs.getInt("CGI_COMP_FLG"));
                cmMdl.setUsrUkoFlg(0);
                cmMdl.setMidokuJkbn(0);
                cmMdl.setMidokuSid(rs.getInt("GROUP_SID"));
                cmMdl.setMidokuId(rs.getString("CGI_ID"));
                cmMdl.setMuteFlg(rs.getInt("CGUC_MUTE_KBN"));
                cmMdl.setFavoriteFlg(rs.getInt("CGUC_FAVORITE_KBN"));

                UDate now = new UDate();
                UDate lastDate = UDate.getInstanceTimestamp(rs.getTimestamp("LAST_DATE"));
                if (now.compareDateYMD(lastDate) == 0) {
                    cmMdl.setMidokuDispDate(lastDate.getStrHour() + ":" + lastDate.getStrMinute());
                } else {
                    cmMdl.setMidokuDispDate(lastDate.getStrMonth() + "/" + lastDate.getStrDay()
                    + " " + lastDate.getStrHour() + ":" + lastDate.getStrMinute());
                }
                if (ret.size() > limit) {
                    if (chkDate != null
                            && !chkDate.equalsTimeStamp(cmMdl.getMidokuDate())) {
                        break;
                    }
                }
                ret.add(cmMdl);

                chkDate = cmMdl.getMidokuDate();
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        return ret;
    }

    /**
     * <br>[機  能] タイムライン件数を取得する（グループ別）
     * <br>[解  説]
     * <br>[備  考]
     * @param usrSid ユーザSID
     * @param startIdx 検索開始位置
     * @param onlyNoRead 未読のみ表示
     * @return グループ情報一覧の件数
     * @throws SQLException SQL実行例外
     */
    public int getGroupTimelineListCount(
            int usrSid, UDate startIdx, boolean onlyNoRead)
            throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   COUNT(GROUP_DATA.GROUP_SID) as CNT ");
            sql.addSql(" from");
            sql.addSql("   (");
            writeQueryGroupTimeline(sql, usrSid);
            sql.addSql("   ) GROUP_DATA");
            sql.addSql(" where");
            sql.addSql("   GROUP_DATA.LAST_DATE < ?");
            if (onlyNoRead) {
                sql.addSql(" and");
                sql.addSql("   GROUP_DATA.CNT > 0");
            }
            // 検索条件
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addDateValue(startIdx);
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret = rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <br>[機  能] 未読ユーザ一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param usrSid ユーザSID
     * @param startIdx 検索開始位置
     * @param onlyNoRead 未読のみ表示
     * @param limit
     * @return グループ情報一覧の件数
     * @throws SQLException SQL実行例外
     */
    public ArrayList<ChatMidokuModel> getUserTimelineList(
            int usrSid, UDate startIdx, boolean onlyNoRead, int limit)
            throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        ArrayList<ChatMidokuModel> ret = new ArrayList<ChatMidokuModel>();
        con = getCon();

        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();

            sql.addSql(" select");
            sql.addSql("   USER_INFO.USID, ");
            sql.addSql("   USER_INFO.CNT, ");
            sql.addSql("   USER_INFO.LAST_DATE, ");
            sql.addSql("   USER_INFO.CUUC_MUTE_KBN, ");
            sql.addSql("   USER_INFO.CUUC_FAVORITE_KBN, ");
            sql.addSql("   CMN_USRM.USR_LGID, ");
            sql.addSql("   CMN_USRM.USR_JKBN, ");
            sql.addSql("   CMN_USRM.USR_UKO_FLG, ");
            sql.addSql("   CMN_USRM_INF.USI_SEI, ");
            sql.addSql("   CMN_USRM_INF.USI_MEI ");
            sql.addSql(" from");
            sql.addSql("   (");
            sql.addSql("     select");
            sql.addSql("       case ");
            sql.addSql("         when CHT_USER_PAIR.CUP_UID_F = ? then CHT_USER_PAIR.CUP_UID_S");
            sql.addIntValue(usrSid);
            sql.addSql("         else CHT_USER_PAIR.CUP_UID_F ");
            sql.addSql("       end as USID, ");
            sql.addSql("       USER_DATA.CNT, ");
            sql.addSql("       USER_DATA.LAST_DATE, ");
            sql.addSql("       USER_DATA.CUUC_MUTE_KBN, ");
            sql.addSql("       USER_DATA.CUUC_FAVORITE_KBN ");
            sql.addSql("     from");
            sql.addSql("       (");
            writeQueryUserChatExtra(sql, usrSid, true);
            sql.addSql("       ) USER_DATA");
            sql.addSql("     inner join ");
            sql.addSql("       CHT_USER_PAIR");
            sql.addSql("     on");
            sql.addSql("       USER_DATA.PAIR_SID = CHT_USER_PAIR.CUP_SID");
            sql.addSql("     where");
            sql.addSql("       USER_DATA.LAST_DATE < ?");
            sql.addSql("   ) USER_INFO");
            sql.addSql(" left join");
            sql.addSql("   CMN_USRM_INF");
            sql.addSql(" on");
            sql.addSql("   USER_INFO.USID = CMN_USRM_INF.USR_SID");
            sql.addSql(" left join");
            sql.addSql("   CMN_USRM");
            sql.addSql(" on");
            sql.addSql("   USER_INFO.USID = CMN_USRM.USR_SID");
            if (onlyNoRead) {
                sql.addSql(" where");
                sql.addSql("   USER_INFO.CNT > 0");
            }
            sql.addSql("     order by");
            sql.addSql("       USER_INFO.LAST_DATE desc");


            // 検索条件
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addDateValue(startIdx);
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();

            UDate chkDate = null;

            while (rs.next()) {
                ChatMidokuModel cmMdl = new ChatMidokuModel();

                cmMdl.setMidokuCount(rs.getInt("CNT"));
                cmMdl.setMidokuDate(UDate.getInstanceTimestamp(rs.getTimestamp("LAST_DATE")));
                cmMdl.setMidokuKbn(GSConstChat.CHAT_KBN_USER);
                cmMdl.setMidokuName(
                        UserUtil.makeName(rs.getString("USI_SEI"), rs.getString("USI_MEI")));
                cmMdl.setUsrUkoFlg(rs.getInt("USR_UKO_FLG"));
                cmMdl.setMidokuJkbn(rs.getInt("USR_JKBN"));
                cmMdl.setMidokuSid(rs.getInt("USID"));
                cmMdl.setMidokuId(rs.getString("USR_LGID"));

                cmMdl.setMuteFlg(rs.getInt("CUUC_MUTE_KBN"));
                cmMdl.setFavoriteFlg(rs.getInt("CUUC_FAVORITE_KBN"));

                UDate now = new UDate();
                UDate lastDate = UDate.getInstanceTimestamp(rs.getTimestamp("LAST_DATE"));
                if (now.compareDateYMD(lastDate) == 0) {
                    cmMdl.setMidokuDispDate(lastDate.getStrHour() + ":" + lastDate.getStrMinute());
                } else {
                    cmMdl.setMidokuDispDate(lastDate.getStrMonth() + "/" + lastDate.getStrDay()
                    + " " + lastDate.getStrHour() + ":" + lastDate.getStrMinute());
                }
                if (ret.size() > limit) {
                    if (chkDate != null
                            && !chkDate.equalsTimeStamp(cmMdl.getMidokuDate())) {
                        break;
                    }
                }

                ret.add(cmMdl);

                chkDate = cmMdl.getMidokuDate();
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <br>[機  能] タイムライン件数を取得する（ユーザ別）
     * <br>[解  説]
     * <br>[備  考]
     * @param usrSid ユーザSID
     * @param startIdx 検索開始位置
     * @param onlyNoRead 未読のみ表示
     * @return グループ情報一覧の件数
     * @throws SQLException SQL実行例外
     */
    public int getUserTimelineListCount(int usrSid, UDate startIdx, boolean onlyNoRead)
            throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();
        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   COUNT(USER_INFO.USID) as CNT ");
            sql.addSql(" from");
            sql.addSql("   (");
            sql.addSql("     select");
            sql.addSql("       case ");
            sql.addSql("         when CHT_USER_PAIR.CUP_UID_F = ? then CHT_USER_PAIR.CUP_UID_S");
            sql.addIntValue(usrSid);
            sql.addSql("         else CHT_USER_PAIR.CUP_UID_F ");
            sql.addSql("       end as USID, ");
            sql.addSql("       USER_DATA.CNT, ");
            sql.addSql("       USER_DATA.LAST_DATE ");
            sql.addSql("     from");
            sql.addSql("       (");
            writeQueryUserChatExtra(sql, usrSid, true);
            sql.addSql("       ) USER_DATA");
            sql.addSql("     left join ");
            sql.addSql("       CHT_USER_PAIR");
            sql.addSql("     on");
            sql.addSql("       USER_DATA.PAIR_SID = CHT_USER_PAIR.CUP_SID");
            sql.addSql("     where");
            sql.addSql("       USER_DATA.LAST_DATE < ?");
            sql.addDateValue(startIdx);
            sql.addSql("   ) USER_INFO");
            sql.addSql(" left join");
            sql.addSql("   CMN_USRM_INF");
            sql.addSql(" on");
            sql.addSql("   USER_INFO.USID = CMN_USRM_INF.USR_SID");
            if (onlyNoRead) {
                sql.addSql(" where");
                sql.addSql("   USER_INFO.CNT > 0");
            }
            // 検索条件
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret = rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }


    /**
     *
     * <br>[機  能] グループチャット先一覧取得用のクエリを書き込む
     * <br>[解  説] 未読件数込で取得できるクエリを書き込む
     * <br>[備  考] 処理内ではSQLパラメータの追加がある
     * @param sql SqlBuffer
     * @param usrSid ユーザSID
     * @return sql SqlBuffer
     * @throws SQLException SQLException
     */
    public SqlBuffer writeQueryGroupTimeline(SqlBuffer sql, int usrSid) throws SQLException {
        return writeQueryGroupChatExtra(sql, usrSid, null, true);
    }
    /**
     *
     * <br>[機  能] グループチャット先一覧取得用のクエリを書き込む
     * <br>[解  説] 未読件数込で取得できるクエリを書き込む
     * <br>[備  考] 処理内ではSQLパラメータの追加がある
     * @param sql SqlBuffer
     * @param usrSid ユーザSID
     * @param grpSids グループSID nullの場合は制限なし
     * @param forTimeline
     * @return sql SqlBuffer
     * @throws SQLException SQLException
     */
    public SqlBuffer writeQueryGroupChatExtra(SqlBuffer sql, int usrSid,
            Collection<Integer> grpSids, boolean forTimeline) {
        boolean all = true;
        if (grpSids != null) {
            all = false;
        }
        sql.addSql(" select");
        sql.addSql("      CGU.CGI_SID AS GROUP_SID,");
        sql.addSql("        case when DATA.LASTDATE is null then CHT_GROUP_DATA_SUM.CGS_LASTDATE");
        sql.addSql("          else DATA.LASTDATE end as LAST_DATE,");
        sql.addSql("        coalesce(CHT_GROUP_DATA_SUM.CGS_CNT, 0) +");
        sql.addSql("           coalesce(DATA.CNT, 0) -");
        sql.addSql("           coalesce(VIEW_EXIST.CGV_VIEWCNT, 0) as CNT,");
        sql.addSql("        CHT_GROUP_UCONF.CGUC_MUTE_KBN, ");
        sql.addSql("        CHT_GROUP_UCONF.CGUC_FAVORITE_KBN ");
        sql.addSql("    from");
        sql.addSql("      (");
        sql.addSql("        select distinct");
        sql.addSql("          CHT_GROUP_USER.CGI_SID");
        sql.addSql("        from");
        sql.addSql("          CHT_GROUP_USER");
        sql.addSql("           inner join CHT_GROUP_INF");
        sql.addSql("             on CHT_GROUP_USER.CGI_SID = CHT_GROUP_INF.CGI_SID");
        sql.addSql("        where");
        sql.addSql("          (");
        sql.addSql("              (");
        sql.addSql("                CGU_SELECT_SID = ?");
        sql.addIntValue(usrSid);
        sql.addSql("              and");
        sql.addSql("                CGU_KBN = ?");
        sql.addIntValue(GSConstChat.CHAT_KBN_USER);
        sql.addSql("              )");
        sql.addSql("              or");
        sql.addSql("              (");
        sql.addSql("                CGU_SELECT_SID in");
        sql.addSql("                 (");
        sql.addSql("                  select");
        sql.addSql("                    GRP_SID");
        sql.addSql("                  from");
        sql.addSql("                    CMN_BELONGM");
        sql.addSql("                  where");
        sql.addSql("                    CMN_BELONGM.USR_SID = ?");
        sql.addIntValue(usrSid);
        sql.addSql("                 )");
        sql.addSql("               and");
        sql.addSql("                 CGU_KBN = ?");
        sql.addIntValue(GSConstChat.CHAT_KBN_GROUP);
        sql.addSql("              )");
        sql.addSql("          )");
        sql.addSql("          and CHT_GROUP_INF.CGI_DEL_FLG <> ?");
        if (all == false) {
            sql.addSql("      and CHT_GROUP_INF.CGI_SID in (");
            sql.addSql(
                grpSids.stream()
                    .map(sid -> String.valueOf(sid))
                    .collect(Collectors.joining(",")));
            sql.addSql("      )");
        }
        sql.addIntValue(GSConstChat.CHAT_MODE_DELETE);
        sql.addSql("      ) CGU");
        sql.addSql("            left join");
        sql.addSql("         (");
        writeQueryGroupDataSumMisyukei(sql);
        sql.addSql("         ) DATA");
        sql.addSql("             on CGU.CGI_SID = DATA.CGI_SID");
        sql.addSql("         inner join CHT_GROUP_DATA_SUM");
        sql.addSql("             on CGU.CGI_SID = CHT_GROUP_DATA_SUM.CGI_SID");
        sql.addSql("         left join");
        sql.addSql("         (");
        sql.addSql("            select ");
        sql.addSql("                CHT_GROUP_VIEW.CGI_SID as CGI_SID,");
        sql.addSql("                CHT_GROUP_VIEW.CGD_SID as CGD_SID,");
        sql.addSql("                CHT_GROUP_VIEW.CGV_VIEWCNT as CGV_VIEWCNT");
        sql.addSql("            from CHT_GROUP_VIEW");
        sql.addSql("            where");
        sql.addSql("                CHT_GROUP_VIEW.CGV_UID = ?");
        sql.addIntValue(usrSid);
        sql.addSql("         ) VIEW_EXIST");
        sql.addSql("            on CGU.CGI_SID = VIEW_EXIST.CGI_SID");
        sql.addSql(" left join ");
        sql.addSql("   CHT_GROUP_UCONF");
        sql.addSql(" on ");
        sql.addSql("   CGU.CGI_SID = CHT_GROUP_UCONF.CGI_SID");
        sql.addSql(" and ");
        sql.addSql("   CHT_GROUP_UCONF.USR_SID = ?");
        sql.addIntValue(usrSid);
        if (forTimeline) {
            sql.addSql("    where");
            sql.addSql("      CHT_GROUP_DATA_SUM.CGS_LASTDATE is not null");
            sql.addSql("      or DATA.CGI_SID is not null");
        }
        return sql;
    }

    /**
    *
    * <br>[機  能] 投稿集計情報の未集計範囲の集計を取得するクエリを書き込む
    * <br>[解  説]
    * <br>[備  考]
    * @param sql SqlBuffer
    * @return sql
    */
    public SqlBuffer writeQueryGroupDataSumMisyukei(SqlBuffer sql) {
        sql.addSql("             select");
        sql.addSql("                 CGI_SID,");
        sql.addSql("                 count(1) as CNT,");
        sql.addSql("                 max(CGD_ADATE) as LASTDATE,");
        sql.addSql("                 max(CGD_SID) as LASTSID");
        sql.addSql("             from");
        sql.addSql("              CHT_GROUP_DATA");
        sql.addSql("             where CGD_SID > coalesce((");
        sql.addSql("                 select max(CGS_LASTSID) LAST_SID from CHT_GROUP_DATA_SUM");
        sql.addSql("               ), 0)");
        sql.addSql("             group by CGI_SID");
        return sql;
    }

    /**
     * <p>Select count
     * @param usrSid ユーザSID
     * @return count
     * @throws SQLException SQL実行例外
     */
    public int getGroupMidokuCount(int usrSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();

            writeQueryGroupTimeline(sql, usrSid);
            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret += rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }
    /**
     * 指定したグループチャットにおける未読件数を取得
     * ただしメッセージ受信ユーザ側から見た未読件数を取得する

     * @param usrSid メッセージ受信ユーザ
     * @param grpSids 指定グループSID
     * @return チャットグループSIDと未読件数のMap
     * @throws SQLException
     */
    public Map<Integer, Integer> getGroupMidokuCountMap(
        int usrSid, Collection<Integer> grpSids) throws SQLException {

        Connection con = null;
        Map<Integer, Integer> ret = new HashMap<>();
        if (grpSids == null || grpSids.isEmpty()) {
            return ret;
        }

        List<Integer> exeList = new ArrayList<>();
        Iterator<Integer> itr = grpSids.iterator();
        StringBuilder sb = new StringBuilder();
        sb.append(" select");
        sb.append("   DATA.GROUP_SID, ");
        sb.append("   DATA.CNT, ");
        sb.append("   DATA.LAST_DATE ");
        sb.append(" from ");

        con = getCon();

        while (itr.hasNext()) {
            exeList.add(itr.next());
            if (exeList.size() < 500
                    && itr.hasNext()) {
                continue;
            }


            //500件毎に実行
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(sb.toString());
            sql.addSql("   (");
            sql = writeQueryGroupChatExtra(sql, usrSid, exeList, true);
            sql.addSql("   ) DATA");

            try (PreparedStatement pstmt = con.prepareStatement(sql.toSqlString());) {
                sql.setParameter(pstmt);
                log__.info(sql.toLogString());
                try (ResultSet rs = pstmt.executeQuery();) {

                    while (rs.next()) {
                        ret.put(rs.getInt("GROUP_SID"), rs.getInt("CNT"));
                    }

                }

            }
            exeList.clear();
        }
        return ret;
    }

    /**
     *
     * <br>[機  能] ユーザチャット先一覧取得用のクエリを書き込む
     * <br>[解  説] 未読件数込のユーザチャット先一覧が取得できる
     * <br>        チャット履歴をがあるもののみを取得する
     * <br>[備  考] 処理内ではSQLパラメータの追加がある
     * @param sql SqlBuffer
     * @param usrSid ユーザSID
     * @param forTimeline タイムライン用 trueで履歴のないものを除外
     * @return sql SqlBuffer
     * @throws SQLException SQLException
     */
    public SqlBuffer writeQueryUserChatExtra(
        SqlBuffer sql, int usrSid, boolean forTimeline) throws SQLException {

        sql.addSql(" select");
        sql.addSql("        PAIR.CUP_SID as PAIR_SID,");
        sql.addSql("        case when DATA.LASTDATE is null then CHT_USER_DATA_SUM.CUS_LASTDATE");
        sql.addSql("          else DATA.LASTDATE end as LAST_DATE,");
        sql.addSql("        coalesce(CHT_USER_DATA_SUM.CUS_CNT, 0) +");
        sql.addSql("           coalesce(DATA.CNT, 0) -");
        sql.addSql("           coalesce( VIEW_EXIST.CUV_VIEWCNT, 0) as CNT,");
        sql.addSql("       CHT_USER_UCONF.CUUC_MUTE_KBN, ");
        sql.addSql("       CHT_USER_UCONF.CUUC_FAVORITE_KBN ");
        sql.addSql("    from");
        sql.addSql("        (");
        sql.addSql("            select");
        sql.addSql("                    CUP_SID");
        sql.addSql("                from");
        sql.addSql("                    CHT_USER_PAIR");
        sql.addSql("                WHERE");
        sql.addSql("                    CUP_UID_F = ?");
        sql.addSql("                    OR CUP_UID_S = ?");
        sql.addIntValue(usrSid);
        sql.addIntValue(usrSid);
        sql.addSql("        ) PAIR");
        sql.addSql("            left join");
        sql.addSql("         (");
        writeQueryUserDataSumMisyukei(sql);
        sql.addSql("         ) DATA");
        sql.addSql("             on PAIR.CUP_SID = DATA.CUP_SID");
        sql.addSql("            inner join CHT_USER_DATA_SUM");
        sql.addSql("                on PAIR.CUP_SID = CHT_USER_DATA_SUM.CUP_SID");
        sql.addSql("            left join");
        sql.addSql("            (");
        sql.addSql("                select ");
        sql.addSql("                 CHT_USER_VIEW.CUP_SID as CUP_SID,");
        sql.addSql("                 CHT_USER_VIEW.CUD_SID as CUD_SID,");
        sql.addSql("                 CHT_USER_VIEW.CUV_VIEWCNT as CUV_VIEWCNT");
        sql.addSql("             from CHT_USER_VIEW");
        sql.addSql("             where");
        sql.addSql("                    CHT_USER_VIEW.CUV_UID = ?");
        sql.addIntValue(usrSid);
        sql.addSql("            ) VIEW_EXIST");
        sql.addSql("                on PAIR.CUP_SID = VIEW_EXIST.CUP_SID");
        sql.addSql("     left join ");
        sql.addSql("       CHT_USER_UCONF");
        sql.addSql("     on ");
        sql.addSql("       PAIR.CUP_SID = CHT_USER_UCONF.CUP_SID");
        sql.addSql("     and ");
        sql.addSql("       CHT_USER_UCONF.USR_SID = ?");
        sql.addIntValue(usrSid);
        if (forTimeline) {
            sql.addSql("    where");
            sql.addSql("      CHT_USER_DATA_SUM.CUS_LASTDATE is not null");
            sql.addSql("      or DATA.CUP_SID is not null");
        }



        return sql;
    }


    /**
     *
     * <br>[機  能] 投稿集計情報の未集計範囲の集計を取得するクエリを書き込む
     * <br>[解  説]
     * <br>[備  考]
     * @param sql SqlBuffer
     * @return sql
     */
    public SqlBuffer writeQueryUserDataSumMisyukei(SqlBuffer sql) {
        sql.addSql("             select");
        sql.addSql("                 CUP_SID,");
        sql.addSql("                 count(1) as CNT,");
        sql.addSql("                 max(CUD_ADATE) as LASTDATE,");
        sql.addSql("                 max(CUD_SID) as LASTSID");
        sql.addSql("             from");
        sql.addSql("              CHT_USER_DATA");
        sql.addSql("             where CUD_SID > coalesce((");
        sql.addSql("                     select max(CUS_LASTSID) LAST_SID from CHT_USER_DATA_SUM");
        sql.addSql("                   ),0)");
        sql.addSql("             group by CUP_SID");
        return sql;
    }

    /**
     * <p>Select count
     * @param usrSid ユーザSID
     * @return count
     * @throws SQLException SQL実行例外
     */
    public int getUserMidokuCount(int usrSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql = writeQueryUserChatExtra(sql, usrSid, true);
            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret += rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <p>ユーザ間チャットにおいて各それぞれでの未読件数を取得する
     * @param sessionUsrSid ユーザSID
     * @param uList 相手ユーザSID
     * @return ret ユーザリスト
     * @throws SQLException SQL実行例外
     */
    public ArrayList<ChatUserInfModel> getUserCntList(int sessionUsrSid,
            List<CmnUsrmInfModel> uList) throws SQLException {

        ArrayList<ChatUserInfModel> ret = new ArrayList<ChatUserInfModel>();
        if (uList == null || uList.isEmpty()) {
            return ret;
        }
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        con = getCon();

        if (uList == null || uList.size() == 0) {
            return ret;
        }

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   case");
            sql.addSql("      when CHT_USER_PAIR.CUP_UID_F = ?");
            sql.addIntValue(sessionUsrSid);
            sql.addSql("      then CHT_USER_PAIR.CUP_UID_S");
            sql.addSql("      else CHT_USER_PAIR.CUP_UID_F");
            sql.addSql("   end as USR_SID,");
            sql.addSql("   CNT_DATA.PAIR_SID,");
            sql.addSql("   CNT_DATA.CNT,");
            sql.addSql("   CNT_DATA.LAST_DATE,");
            sql.addSql("   CNT_DATA.CUUC_MUTE_KBN, ");
            sql.addSql("   CNT_DATA.CUUC_FAVORITE_KBN ");
            sql.addSql(" from");
            sql.addSql("   CHT_USER_PAIR");
            sql.addSql(" left join");
            sql.addSql("   (");
            writeQueryUserChatExtra(sql, sessionUsrSid, false);
            sql.addSql("   ) CNT_DATA");
            sql.addSql(" on");
            sql.addSql("   CHT_USER_PAIR.CUP_SID = CNT_DATA.PAIR_SID");
            sql.addSql(" where");
            sql.addSql("   (");
            sql.addSql("      CHT_USER_PAIR.CUP_UID_F = ?");
            sql.addIntValue(sessionUsrSid);
            sql.addSql("    and");
            sql.addSql("      CHT_USER_PAIR.CUP_UID_S IN (");
            for (int idx = 0; idx < uList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql("    ,");
                }
                sql.addSql("    ?");
                sql.addIntValue(uList.get(idx).getUsrSid());
            }
            sql.addSql("      )");
            sql.addSql("   )");
            sql.addSql(" or");
            sql.addSql("   (");
            sql.addSql("      CHT_USER_PAIR.CUP_UID_S = ?");
            sql.addIntValue(sessionUsrSid);
            sql.addSql("    and");
            sql.addSql("      CHT_USER_PAIR.CUP_UID_F IN (");
            for (int idx = 0; idx < uList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql("    ,");
                }
                sql.addSql("    ?");
                sql.addIntValue(uList.get(idx).getUsrSid());
            }
            sql.addSql("      )");
            sql.addSql("   )");

            pstmt = con.prepareStatement(sql.toSqlString());
            sql.setParameter(pstmt);
            log__.info(sql.toLogString());
            rs = pstmt.executeQuery();

            Map<Integer, ChatUserInfModel> mdlMap = new HashMap<>();
            while (rs.next()) {
                ChatUserInfModel bean = new ChatUserInfModel();
                int usrSid = rs.getInt("USR_SID");
                bean.setChtUserCount(rs.getInt("CNT"));
                bean.setChtLastDate(
                    UDate.getInstanceTimestamp(rs.getTimestamp("LAST_DATE"))
                    );

                ChtUserUconfModel uconf = new ChtUserUconfModel();
                uconf.setUsrSid(sessionUsrSid);
                uconf.setCupSid(rs.getInt("PAIR_SID"));
                uconf.setCuucMuteKbn(rs.getInt("CUUC_MUTE_KBN"));
                uconf.setCuucFavoriteKbn(rs.getInt("CUUC_FAVORITE_KBN"));
                bean.setUconfModel(uconf);

                mdlMap.put(usrSid, bean);
            }
            for (CmnUsrmInfModel cuiMdl : uList) {
                ChatUserInfModel bean = mdlMap.get(cuiMdl.getUsrSid());
                if (bean == null) {
                    bean = new ChatUserInfModel();
                    bean.setChtUserCount(0);
                }
                bean.setBinSid(cuiMdl.getBinSid());
                bean.setUsiPictKf(cuiMdl.getUsiPictKf());
                bean.setUsrSid(cuiMdl.getUsrSid());
                bean.setUsrJkbn(cuiMdl.getUsrJkbn());
                bean.setUsrUkoFlg(cuiMdl.getUsrUkoFlg());
                bean.setUsiSei(cuiMdl.getUsiSei());
                bean.setUsiMei(cuiMdl.getUsiMei());
                ret.add(bean);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }


    /**
     * 指定したグループチャットにおける未読件数を取得
     * ただしメッセージ受信ユーザ側から見た未読件数を取得する
     * @param usrSid 受信者SID
     * @param cgiSid ペアSID
     * @return 未読件数
     * @throws SQLException SQL実行例外
     * */
    public int getMidokuCountGroup(int usrSid, int cgiSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   DATA.GROUP_SID, ");
            sql.addSql("   DATA.CNT, ");
            sql.addSql("   DATA.LAST_DATE ");
            sql.addSql(" from");
            sql.addSql("   (");
            sql = writeQueryGroupTimeline(sql, usrSid);
            sql.addSql("   ) DATA");
            sql.addSql(" where");
            sql.addSql("   DATA.GROUP_SID = ? ");
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addIntValue(cgiSid);

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret = rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * 指定したユーザ間チャットにおける未読件数を取得
     * ただしメッセージ受信ユーザ側から見た未読件数を取得する
     * @param usrSid 受信者SID
     * @param pairSid ペアSID
     * @return 未読件数
     * @throws SQLException SQL実行例外
     * */
    public int getMidokuCountPair(int usrSid, int pairSid) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 0;
        con = getCon();

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql("     select");
            sql.addSql("       USER_DATA.CNT, ");
            sql.addSql("       USER_DATA.LAST_DATE ");
            sql.addSql("     from");
            sql.addSql("       (");
            writeQueryUserChatExtra(sql, usrSid, false);
            sql.addSql("       ) USER_DATA");
            sql.addSql(" where ");
            sql.addSql("   USER_DATA.PAIR_SID = ? ");
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addIntValue(pairSid);

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
                ret = rs.getInt("CNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }


    /**
     * <br>[機  能] チャットグループ情報一覧の件数を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param searchMdl 検索情報
     * @return グループ情報一覧の件数
     * @throws SQLException SQL実行例外
     */
    public int getChatGroupDataCount(ChatSearchModel searchMdl) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int count = 0;
        con = getCon();

        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   count(CHT_GROUP_INF.CGI_SID) as COUNT");
            sql.addSql(" from");
            sql.addSql("   CHT_GROUP_INF");
            sql.addSql(" left join ");
            sql.addSql("   (");
            sql.addSql("   select");
            sql.addSql("     CGI_SID,");
            sql.addSql("     MAX(CGD_ADATE) as CGD_ADATE");
            sql.addSql("   from");
            sql.addSql("     CHT_GROUP_DATA");
            sql.addSql("   group by");
            sql.addSql("     CGI_SID");
            sql.addSql("   )LAST_DAY");
            sql.addSql(" on ");
            sql.addSql("    CHT_GROUP_INF.CGI_SID = LAST_DAY.CGI_SID");
            // 検索条件
            __searchWhere(searchMdl, sql);
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();

            if (rs.next()) {
                count = rs.getInt("COUNT");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        return count;
    }

    /**
     * <br>[機  能] チャットグループ情報一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param searchMdl 検索情報
     * @return グループ情報一覧
     * @throws SQLException SQL実行例外
     */
    public List <ChatInformationModel> getChtGrpDataList(
            ChatSearchModel searchMdl) throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        con = getCon();
        List<ChatInformationModel> ret = new ArrayList<ChatInformationModel>();
        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select ");
            sql.addSql("   CHT_GROUP_INF.CGI_SID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_ID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_NAME, ");
            sql.addSql("   CHT_GROUP_INF.CGI_CONTENT, ");
            sql.addSql("   CHT_GROUP_INF.CGI_COMP_FLG, ");
            sql.addSql("   CHT_GROUP_INF.CHC_SID, ");
            sql.addSql("   CHT_GROUP_INF.CGI_ADATE, ");
            sql.addSql("   CHT_GROUP_INF.CGI_EDATE,");
            sql.addSql("   LAST_DAY.CGD_ADATE");
            sql.addSql(" from ");
            sql.addSql("   CHT_GROUP_INF");
            sql.addSql(" left join ");
            sql.addSql("   (");
            sql.addSql("   select");
            sql.addSql("     CGI_SID,");
            sql.addSql("     MAX(CGD_ADATE) as CGD_ADATE");
            sql.addSql("   from");
            sql.addSql("     CHT_GROUP_DATA");
            sql.addSql("   group by");
            sql.addSql("     CGI_SID");
            sql.addSql("   ) as LAST_DAY");
            sql.addSql(" on ");
            sql.addSql("    CHT_GROUP_INF.CGI_SID = LAST_DAY.CGI_SID");
            // 検索条件
            __searchWhere(searchMdl, sql);
            __searchSort(searchMdl, sql);
            if (searchMdl.getOffset() > 0) {
                sql.setPagingValue(searchMdl.getOffset(), searchMdl.getMaxCnt());
            } else {
                __page(searchMdl, sql);
            }
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ChatInformationModel mdl = new ChatInformationModel();
                mdl.setChatSid(rs.getInt("CGI_SID"));
                mdl.setChatName(rs.getString("CGI_NAME"));
                mdl.setChatArchive(rs.getInt("CGI_COMP_FLG"));
                mdl.setCategorySid(rs.getInt("CHC_SID"));
                mdl.setChatId(rs.getString("CGI_ID"));
                mdl.setChatBiko(rs.getString("CGI_CONTENT"));
                mdl.setInsertDate(UDate.getInstanceTimestamp(rs.getTimestamp("CGI_ADATE")));
                mdl.setStrInsertDate(__createDateStr(mdl.getInsertDate()));
                mdl.setLastSendDate(UDate.getInstanceTimestamp(rs.getTimestamp("CGD_ADATE")));
                mdl.setStrLastSendDate(__createDateStr(mdl.getLastSendDate()));
                ret.add(mdl);
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }

        return ret;
    }


    /**
     * <br>[機  能] 検索where
     * <br>[解  説]
     * <br>[備  考]
     * @param searchMdl 検索情報
     * @param sql sql
     * @return グループ情報一覧
     * @throws SQLException SQL実行例外
     */
    public SqlBuffer __searchWhere(
            ChatSearchModel searchMdl, SqlBuffer sql) throws SQLException {
        sql.addSql("   where");
        sql.addSql(" 1 = 1");

        // キーワード検索
        List < String > keywordList = searchMdl.getKeywordList();

        if (keywordList != null && !keywordList.isEmpty()) {
            String keywordJoin = "   and";
            if (searchMdl.getAndOr() == GSConstChat.KEYWORDKBN_OR) {
                keywordJoin = "   or";
            }

            boolean kakko = false;
            boolean isGroupId = (searchMdl.getGroupId() ==  GSConstChat.SEARCH_GROUPID_IN);
            boolean isGroupName = (searchMdl.getGroupName() == GSConstChat.SEARCH_GROUPNAME_IN);
            boolean isGroupInfo = (searchMdl.getGroupInfo() == GSConstChat.SEARCH_GROUPINFO_IN);
            int chkNum = searchMdl.getGroupId()
                    + searchMdl.getGroupName() + searchMdl.getGroupInfo();

            if (chkNum >= 2) {
                kakko = true;
                sql.addSql("  and");
                sql.addSql("   (");
            }

            // グループID
            if (isGroupId) {
                if (chkNum == 1) {
                    sql.addSql("  and");
                }
                  sql.addSql("     (");
                for (int i = 0; i < keywordList.size(); i++) {
                    if (i > 0) {
                        sql.addSql(keywordJoin);
                    }
                    sql.addSql("       CGI_ID like '%"
                            + JDBCUtil.escapeForLikeSearch(keywordList.get(i))
                            + "%' ESCAPE '"
                            + JDBCUtil.def_esc
                            + "'");
                }
                sql.addSql("     )");
            }

            // グループ名
            if (isGroupName) {
                if (isGroupId) {
                    sql.addSql("    or");
                }
                if (chkNum == 1) {
                    sql.addSql("  and");
                }
                sql.addSql("     (");
                for (int i = 0; i < keywordList.size(); i++) {
                    if (i > 0) {
                        sql.addSql(keywordJoin);
                    }
                    sql.addSql("       CGI_NAME like '%"
                            + JDBCUtil.escapeForLikeSearch(keywordList.get(i))
                            + "%' ESCAPE '"
                            + JDBCUtil.def_esc
                            + "'");
                }
                sql.addSql("     )");
            }

            // 備考
            if (isGroupInfo) {

                if (isGroupId || isGroupName) {
                    sql.addSql("    or");
                }
                if (chkNum == 1) {
                    sql.addSql("  and");
                }

                sql.addSql("     (");
                for (int i = 0; i < keywordList.size(); i++) {
                    if (i > 0) {
                        sql.addSql(keywordJoin);
                    }
                    sql.addSql("       CGI_CONTENT like '%"
                            + JDBCUtil.escapeForLikeSearch(keywordList.get(i))
                            + "%' ESCAPE '"
                            + JDBCUtil.def_esc
                            + "'");
                }
                sql.addSql("     )");
            }
            if (kakko) {
                sql.addSql("     )");
            }
        }
        // カテゴリ
        if (searchMdl.getCategory() != GSConstChat.CHAT_CHC_SID_ALL) {
            sql.addSql("  and");
            sql.addSql("   CHC_SID = ?");
            sql.addIntValue(searchMdl.getCategory());
        }
        // アーカイブ
        if (searchMdl.getStatusKbn() != GSConstChat.SEARCH_STATUSKBN_ALL) {
                sql.addSql("  and");
                switch (searchMdl.getStatusKbn()) {
                case GSConstChat.SEARCH_STATUSKBN_NOT_ARCHIVE:
                    sql.addSql("   CGI_COMP_FLG = ?");
                    sql.addIntValue(GSConstChat.CHAT_ARCHIVE_NOT_MODE);
                    break;
                case GSConstChat.SEARCH_STATUSKBN_ARCHIVE_ONLY:
                    sql.addSql("   CGI_COMP_FLG = ?");
                    sql.addIntValue(GSConstChat.CHAT_ARCHIVE_MODE);
                    break;
                default:
                    sql.addSql("   CGI_COMP_FLG = ?");
                    sql.addIntValue(GSConstChat.CHAT_ARCHIVE_NOT_MODE);
                    break;
            }
        }

        // メンバー(グループ検索画面)
        if (searchMdl.getSelectUser() != -1 && searchMdl.getSelectGroup() != -1) {
            sql.addSql("  and");
            sql.addSql("   CHT_GROUP_INF.CGI_SID  in (");
            sql.addSql("    select");
            sql.addSql("      CGI_SID");
            sql.addSql("    from");
            sql.addSql("      CHT_GROUP_USER");
            sql.addSql("    where");
            sql.addSql("      CGU_SELECT_SID = ?");
            sql.addSql("    and");
            sql.addSql("      CGU_KBN = ?");
            sql.addIntValue(searchMdl.getSelectUser());
            sql.addIntValue(GSConstChat.CHAT_KBN_USER);
            if (searchMdl.getAdminMember() == GSConstChat.SEARCH_GROUPADMIN_IN
                    && searchMdl.getGeneralMember() == GSConstChat.SEARCH_GENERALUSER_OUT) {
                sql.addSql("  and");
                sql.addSql("  CGU_ADMIN_FLG = ?");
                sql.addIntValue(GSConstChat.SEARCH_GROUPADMIN_IN);
            } else if (searchMdl.getAdminMember() == GSConstChat.SEARCH_GROUPADMIN_OUT
                    && searchMdl.getGeneralMember() == GSConstChat.SEARCH_GENERALUSER_IN) {
                sql.addSql("  and");
                sql.addSql("  CGU_ADMIN_FLG = ?");
                sql.addIntValue(GSConstChat.SEARCH_GROUPADMIN_OUT);
            }
            sql.addSql("   )");
        } else if (searchMdl.getSelectGroup() != -1 && searchMdl.getSelectUser() == -1) {
            sql.addSql("  and");
            sql.addSql("   CHT_GROUP_INF.CGI_SID  in (");
            sql.addSql("    select");
            sql.addSql("      CGI_SID");
            sql.addSql("    from");
            sql.addSql("      CHT_GROUP_USER");
            sql.addSql("    where");
            sql.addSql("      CGU_SELECT_SID in (");
            sql.addSql("    select");
            sql.addSql("      USR_SID");
            sql.addSql("    from");
            sql.addSql("      CMN_BELONGM");
            sql.addSql("    where");
            sql.addSql("      GRP_SID = ?");
            sql.addIntValue(searchMdl.getSelectGroup());
            sql.addSql("   )");
            if (searchMdl.getAdminMember() == GSConstChat.SEARCH_GROUPADMIN_IN
                    && searchMdl.getGeneralMember() == GSConstChat.SEARCH_GENERALUSER_OUT) {
                sql.addSql("  and");
                sql.addSql("  CGU_ADMIN_FLG = ?");
                sql.addIntValue(GSConstChat.SEARCH_GROUPADMIN_IN);
            } else if (searchMdl.getAdminMember() == GSConstChat.SEARCH_GROUPADMIN_OUT
                    && searchMdl.getGeneralMember() == GSConstChat.SEARCH_GENERALUSER_IN) {
                sql.addSql("  and");
                sql.addSql("  CGU_ADMIN_FLG = ?");
                sql.addIntValue(GSConstChat.SEARCH_GROUPADMIN_OUT);
            }
            sql.addSql("   )");
        }

        // メンバー(API)
        if (searchMdl.getBelongMemberSid() > 0) {
            sql.addSql("  and");
            sql.addSql("   CHT_GROUP_INF.CGI_SID IN (");
            sql.addSql("     select");
            sql.addSql("       CGI_SID");
            sql.addSql("     from");
            sql.addSql("       CHT_GROUP_USER");
            sql.addSql("     where");
            sql.addSql("       (");
            sql.addSql("          CGU_SELECT_SID = ?");
            sql.addSql("        and");
            sql.addSql("          CGU_KBN = ?");
            sql.addSql("       )");
            sql.addSql("       or");
            sql.addSql("       (");
            sql.addSql("          CGU_KBN = ?");
            sql.addSql("        and");
            sql.addSql("          CGU_SELECT_SID IN (");
            sql.addSql("            select");
            sql.addSql("              GRP_SID");
            sql.addSql("            from");
            sql.addSql("              CMN_BELONGM");
            sql.addSql("            where");
            sql.addSql("              CMN_BELONGM.USR_SID = ?");
            sql.addSql("          )");
            sql.addSql("       )");
            sql.addSql("    )");

            sql.addIntValue(searchMdl.getBelongMemberSid());
            sql.addIntValue(GSConstChat.CHAT_KBN_USER);
            sql.addIntValue(GSConstChat.CHAT_KBN_GROUP);
            sql.addIntValue(searchMdl.getBelongMemberSid());

        }
        // 作成日時
        if (searchMdl.getCreateDateFr() != null) {
            sql.addSql(" and");
            sql.addSql("   CHT_GROUP_INF.CGI_ADATE >= ?");
            UDate date = UDate.getInstance(searchMdl.getCreateDateFr().getTimeMillis());
            date.setHour(0);
            date.setMinute(0);
            date.setSecond(0);
            date.setMilliSecond(0);
            sql.addDateValue(date);
        }
        if (searchMdl.getCreateDateTo() != null) {
            sql.addSql(" and");
            sql.addSql("   CGI_ADATE <= ?");
            UDate date = UDate.getInstance(searchMdl.getCreateDateTo().getTimeMillis());
            date.setHour(23);
            date.setMinute(59);
            date.setSecond(59);
            date.setMilliSecond(999);
            sql.addDateValue(date);
        }
        // 最終投稿日時
        if (searchMdl.getUpDateFr() != null) {
            sql.addSql(" and");
            sql.addSql("   CGD_ADATE >= ?");
            UDate date = UDate.getInstance(searchMdl.getUpDateFr().getTimeMillis());
            date.setHour(0);
            date.setMinute(0);
            date.setSecond(0);
            date.setMilliSecond(0);
            sql.addDateValue(date);
        }
        if (searchMdl.getUpDateTo() != null) {
            sql.addSql(" and");
            sql.addSql("   CGD_ADATE <= ?");
            UDate date = UDate.getInstance(searchMdl.getUpDateTo().getTimeMillis());
            date.setHour(23);
            date.setMinute(59);
            date.setSecond(59);
            date.setMilliSecond(999);
            sql.addDateValue(date);
        }
        //削除
        sql.addSql(" and");
        sql.addSql("   CGI_DEL_FLG <> ?");
        sql.addIntValue(GSConstChat.CHAT_GROUP_LOGIC_DELETE);
        return sql;
    }

    /**
     * <br>[機  能] sort
     * <br>[解  説]
     * <br>[備  考]
     * @param searchMdl 検索情報
     * @param sql sql
     * @return グループ情報一覧
     * @throws SQLException SQL実行例外
     */
    private SqlBuffer __searchSort(ChatSearchModel searchMdl, SqlBuffer sql) throws SQLException {
        sql.addSql("   order by ");
        String order = "asc";
        if (searchMdl.getOrderKey() == GSConst.ORDER_KEY_DESC) {
            order = "desc";
        }
        switch (searchMdl.getSortKey()) {
            case GSConstChat.CHAT_SORT_GRPID:
                sql.addSql("   CGI_ID " + order);
                break;
            case GSConstChat.CHAT_SORT_GRPNAME:
                sql.addSql("   CGI_NAME " + order);
                break;
            case GSConstChat.CHAT_SORT_ADATE:
                sql.addSql("   CGI_ADATE " + order);
                break;
            case GSConstChat.CHAT_SORT_EDATE:
                sql.addSql("   CGD_ADATE " + order);
                break;
            default:
                sql.addSql("   CGI_SID asc");
                break;
        }
        return sql;
    }

    /**
     * <br>[機  能] page
     * <br>[解  説]
     * <br>[備  考]
     * @param searchMdl 検索情報
     * @param sql sql
     * @return グループ情報一覧
     * @throws SQLException SQL実行例外
     */
    private SqlBuffer __page(ChatSearchModel searchMdl, SqlBuffer sql) throws SQLException {
        int page = searchMdl.getPage();
        if (page <= 0) {
            page = 1;
        }
        int maxCnt = searchMdl.getMaxCnt();
        sql.setPagingValue(PageUtil.getRowNumber(page, maxCnt) - 1, maxCnt);
        return sql;
    }

    /**
     * 日付文字列を取得する
     * @param date 日付
     * @return 日付文字列
     */
    private String __createDateStr(UDate date) {
        if (date == null) {
            return null;
        }

        StringBuilder strDate = new StringBuilder("");
        strDate.append(UDateUtil.getSlashYYMD(date));
        strDate.append(" ");
        strDate.append(UDateUtil.getSeparateHMS(date));

        return strDate.toString();
    }

    /**
     * <br>[機  能] グループSID内に所属するユーザを取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param grpList グループリスト
     * @return ユーザSIDリスト
     * @throws SQLException SQL実行例外
     */
    public List<Integer> getUsrSid(List<Integer> grpList)
            throws SQLException {

        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        List<Integer> ret = new ArrayList<Integer>();
        con = getCon();

        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" select");
            sql.addSql("   USR_SID ");
            sql.addSql(" from");
            sql.addSql("   CMN_BELONGM");
            sql.addSql(" where");
            sql.addSql("   GRP_SID IN (");
            for (int idx = 0; idx < grpList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql(" , ");
                }
                sql.addSql(" ? ");
                sql.addIntValue(grpList.get(idx));
            }
            sql.addSql("   )");


            // 検索条件
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            while (rs.next()) {
                ret.add(rs.getInt("USR_SID"));
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <br>[機  能] グループSIDの採番値を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @return グループSID
     * @throws SQLException SQL実行例外
     */
    public int getSaibanCgiSid()
            throws SQLException {
        PreparedStatement pstmt = null;
        ResultSet rs = null;
        Connection con = null;
        int ret = 1;
        con = getCon();

        try {
            // SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql("select");
            sql.addSql(" SBN_NUMBER ");
            sql.addSql("from");
            sql.addSql(" CMN_SAIBAN ");
            sql.addSql("where");
            sql.addSql(" SBN_SID = ? ");
            sql.addSql("and");
            sql.addSql(" SBN_SID_SUB = ? ");
            log__.info(sql.toLogString());
            pstmt = con.prepareStatement(sql.toSqlString());
            sql.addStrValue(GSConstChat.SBNSID_CHAT);
            sql.addStrValue(GSConstChat.SBNSID_SUB_CHAT_GROUP);
            sql.setParameter(pstmt);
            rs = pstmt.executeQuery();
            if (rs.next()) {
              ret = rs.getInt("SBN_NUMBER");
            }
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeResultSet(rs);
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <p>update
     * @param sidList バイナリSIDリスト
     * @throws SQLException SQL実行例外
     * @return count
     */
    public int updateTempJkbn(List<Long> sidList) throws SQLException {

        PreparedStatement pstmt = null;
        Connection con = getCon();
        int ret = 0;

        try {
            //SQL文
            SqlBuffer sql = new SqlBuffer();
            sql.addSql(" update");
            sql.addSql("   CMN_BINF");
            sql.addSql(" set");
            sql.addSql("   BIN_JKBN = ?");
            sql.addIntValue(GSConst.JTKBN_DELETE);
            sql.addSql(" where ");
            sql.addSql("   BIN_SID IN ( ");
            for (int idx = 0; idx < sidList.size(); idx++) {
                if (idx != 0) {
                    sql.addSql("  , ");
                }
                sql.addSql("  ? ");
                sql.addLongValue(sidList.get(idx));
            }
            sql.addSql("   ) ");
            pstmt = con.prepareStatement(sql.toSqlString());

            log__.info(sql.toLogString());
            sql.setParameter(pstmt);
            ret = pstmt.executeUpdate();
        } catch (SQLException e) {
            throw e;
        } finally {
            JDBCUtil.closeStatement(pstmt);
        }
        return ret;
    }

    /**
     * <br>[機  能] メッセージに添付ファイル情報を設定する
     * <br>[解  説]
     * <br>[備  考]
     * @param cmList メッセージ情報
     * @param type 1:ユーザ, 2:グループ
     * @return 添付ファイルを設定したメッセージ情報
     * @throws SQLException SQL実行時例外
     */
    private ArrayList<ChatMessageModel> __setTempInfo(
        ArrayList<ChatMessageModel> cmList, int type) throws SQLException {

        if (cmList == null || cmList.isEmpty()) {
            return cmList;
        }

        List<Long> sidList = cmList.stream()
            .map(mdl -> mdl.getMessageSid())
            .collect(Collectors.toList());

        Map<Long, List<CmnBinfModel>> binMap = null;
        Connection con = getCon();
        if (type == GSConstChat.CHAT_KBN_USER) {
            ChtUserDataTempDao cutDao = new ChtUserDataTempDao(con);
            binMap = cutDao.selectBinMap(sidList);
        } else {
            ChtGroupDataTempDao cgtDao = new ChtGroupDataTempDao(con);
            binMap = cgtDao.selectBinMap(sidList);
        }

        if (!binMap.isEmpty()) {
            List<CmnBinfModel> binMdlList = null;
            for (ChatMessageModel messageMdl : cmList) {
                binMdlList = binMap.get(messageMdl.getMessageSid());
                messageMdl.setBinMdlList(binMdlList);
            }
        }

        return cmList;
    }

    /**
     * <br>[機  能] グループチャットのメッセージにリアクション情報を設定します。
     * <br>[解  説]
     * <br>[備  考]
     *
     * @param cmList メッセージ情報
     * @throws SQLException
     */
    private void __setGroupReactionInfo(
        ArrayList<ChatMessageModel> cmList) throws SQLException {

        if (cmList == null || cmList.isEmpty()) {
            return;
        }

        List<Long> messageSid = cmList.stream()
            .map(mdl -> mdl.getMessageSid())
            .collect(Collectors.toList());

        ChtReactionBiz reactionBiz = new ChtReactionBiz();
        Map<Long, List<ChtReactionModel>> reactionMap
            = reactionBiz.getGroupReactionData(messageSid, getCon());
        for (ChatMessageModel messageMdl : cmList) {
            messageMdl.setReactionList(reactionMap.get(messageMdl.getMessageSid()));
        }
    }

    /**
     * <br>[機  能] ユーザチャットのメッセージにリアクション情報を設定します。
     * <br>[解  説]
     * <br>[備  考]
     *
     * @param cmList メッセージ情報
     * @throws SQLException
     */
    private void __setUserReactionInfo(
        ArrayList<ChatMessageModel> cmList) throws SQLException {

        if (cmList == null || cmList.isEmpty()) {
            return;
        }

        List<Long> messageSid = cmList.stream()
            .map(mdl -> mdl.getMessageSid())
            .collect(Collectors.toList());

        ChtReactionBiz reactionBiz = new ChtReactionBiz();
        Map<Long, List<ChtReactionModel>> reactionMap
            = reactionBiz.getUserReactionData(messageSid, getCon());
        for (ChatMessageModel messageMdl : cmList) {
            messageMdl.setReactionList(reactionMap.get(messageMdl.getMessageSid()));
        }
    }

    /**
     * <br>[機  能] メッセージに返信元投稿の情報を設定します。
     * <br>[解  説]
     * <br>[備  考]
     *
     * @param cmList メッセージ情報
     * @param type 1:ユーザ, 2:グループ
     * @throws SQLException
     */
    private void __setReplyMessage(
        ArrayList<ChatMessageModel> cmList, int type) throws SQLException {

        if (cmList == null || cmList.isEmpty()) {
            return;
        }

        List<Long> messageSid = cmList.stream()
            .map(mdl -> mdl.getReplyMessageSid())
            .collect(Collectors.toList());

        Map<Long, ChatMessageModel> replyMap = null;
        if (type == GSConstChat.CHAT_KBN_GROUP) {
            replyMap = getGroupReplyMap(messageSid);
        } else {
            replyMap = getUserReplyMap(messageSid);
        }

        if (replyMap == null || replyMap.isEmpty()) {
            return;
        }

        ChatMessageModel replyMessage;
        ChatMessageModel mapMsg;
        for (ChatMessageModel cmMdl : cmList) {
            replyMessage = null;
            mapMsg = replyMap.get(cmMdl.getReplyMessageSid());
            if (mapMsg != null) {
                replyMessage = new ChatMessageModel();
                CloneableUtil.copyField(replyMessage, mapMsg);
            }
            cmMdl.setReplyMessageInfo(replyMessage);
        }
    }

    /**
     * <br>[機  能] メッセージにメンション情報を設定します
     * <br>[解  説]
     * <br>[備  考]
     *
     * @param cmList メッセージ情報
     * @param type 1:ユーザ, 2:グループ
     * @throws SQLException
     */
    private void __setMentionInfo(
        ArrayList<ChatMessageModel> cmList, int type) throws SQLException {

        if (cmList == null || cmList.isEmpty()) {
            return;
        }

        List<Long> messageSid = cmList.stream()
            .map(mdl -> mdl.getMessageSid())
            .collect(Collectors.toList());

        ChtMentionBiz mentionBiz = new ChtMentionBiz();
        Map<Long, List<CmnUserModel>> mentionMap;
        if (type == GSConstChat.CHAT_KBN_GROUP) {
            mentionMap = mentionBiz.getGroupMentionData(messageSid, getCon());
        } else {
            mentionMap = mentionBiz.getUserMentionData(messageSid, getCon());
        }
        for (ChatMessageModel messageMdl : cmList) {
            messageMdl.setMentionUserInfo(mentionMap.get(messageMdl.getMessageSid()));
        }
    }

    /**
     * <br>[機  能] チャットのメッセージにピンどめ情報を設定します。
     * <br>[解  説]
     * <br>[備  考]
     * @param cmList メッセージ一覧
     * @param chatKbn チャット区分 0:ユーザ 1:グループ
     * @param selectSid 対象SID ユーザ:対象ユーザSID 1:グループSID
     * @param userSid セッションユーザSID
     * @throws SQLException
     */
    private void __setMessagePinInfo(
        ArrayList<ChatMessageModel> cmList, int chatKbn,
        int selectSid, int userSid) throws SQLException {
        Connection con = getCon();
        if (cmList == null || cmList.isEmpty()) {
            return;
        }
        List<Long> messageSidList = cmList.stream()
            .map(mdl -> mdl.getMessageSid())
            .collect(Collectors.toList());

        if (chatKbn == GSConstChat.CHAT_KBN_USER) {
            ChtUserDataPinDao uPinDao = new ChtUserDataPinDao(con);
            List<ChtUserDataPinModel> uPinMdl = uPinDao.selectList(
                userSid, selectSid, messageSidList);
            Map<Long, ChtUserDataPinModel> pinedMap =
             uPinMdl.stream()
                .collect(Collectors.toMap(
                    m -> m.getCudSid(),
                    m -> m,
                    (m1, m2) -> m2));
            if (pinedMap.size() > 0) {
                //ピンどめされた投稿にピンどめ有効区分を設定する。
                for (ChatMessageModel messageMdl : cmList) {
                    ChtUserDataPinModel mdl = pinedMap.get(messageMdl.getMessageSid());
                    if (mdl != null) {
                        messageMdl.setMessagePinKbn(GSConstChat.CHAT_MESSAGE_PIN_ON);
                        messageMdl.setMessagePinSortNum(mdl.getCudpSort());
                    }
                }
            }
        } else if (chatKbn == GSConstChat.CHAT_KBN_GROUP) {
            ChtGroupDataPinDao gPinDao = new ChtGroupDataPinDao(con);
            List<ChtGroupDataPinModel> gPinMdl = gPinDao.selectList(
                userSid, selectSid, messageSidList);
            Map<Long, ChtGroupDataPinModel> pinedMap =
            gPinMdl.stream()
                .collect(Collectors.toMap(
                    m -> m.getCgdSid(),
                    m -> m,
                    (m1, m2) -> m2));
            if (pinedMap.size() > 0) {
                //ピンどめされた投稿にピンどめ有効区分を設定する。
                for (ChatMessageModel messageMdl : cmList) {
                    ChtGroupDataPinModel mdl = pinedMap.get(messageMdl.getMessageSid());
                    if (mdl != null) {
                        messageMdl.setMessagePinKbn(GSConstChat.CHAT_MESSAGE_PIN_ON);
                        messageMdl.setMessagePinSortNum(mdl.getCgdpSort());
                    }
                }
            }
        }
    }

    /**
     * <br>[機  能] お気に入りチャットグループ検索SQLのwhere SQLを設定する
     * <br>[解  説]
     * <br>[備  考]
     * @param sql SqlBuffer
     * @param usrSid ユーザSID
     * @param wdrSidList ディレクトリSID一覧
     * @return SqlBuffer
     */
    private SqlBuffer __setFavoriteGroupWhereSql(SqlBuffer sql, int usrSid) {

        sql.addSql(" where ");
        sql.addSql("   CHT_GROUP_INF.CGI_SID IN (");
        sql.addSql("     select");
        sql.addSql("       CGI_SID");
        sql.addSql("     from");
        sql.addSql("       CHT_GROUP_UCONF");
        sql.addSql("     where");
        sql.addSql("       USR_SID = ?");
        sql.addSql("     and");
        sql.addSql("       CGUC_FAVORITE_KBN = ?");
        sql.addSql("     and");
        sql.addSql("       exists (");
        sql.addSql("         select 1 from CHT_GROUP_USER");
        sql.addSql("         where");
        sql.addSql("           CHT_GROUP_UCONF.CGI_SID = CHT_GROUP_USER.CGI_SID");
        sql.addSql("         and");
        sql.addSql("           (");
        sql.addSql("             (");
        sql.addSql("               CHT_GROUP_USER.CGU_KBN = ?");
        sql.addSql("             and");
        sql.addSql("               CHT_GROUP_USER.CGU_SELECT_SID = ?");
        sql.addSql("             )");
        sql.addSql("           or");
        sql.addSql("             (");
        sql.addSql("               CHT_GROUP_USER.CGU_KBN = ?");
        sql.addSql("             and");
        sql.addSql("               CHT_GROUP_USER.CGU_SELECT_SID in (");
        sql.addSql("                 select GRP_SID from CMN_BELONGM");
        sql.addSql("                 where USR_SID = ?");
        sql.addSql("               )");
        sql.addSql("             )");
        sql.addSql("           )");
        sql.addSql("         )");
        sql.addSql("    )");
        sql.addSql(" and ");
        sql.addSql("   CGI_DEL_FLG = ? ");

        sql.addIntValue(usrSid);
        sql.addIntValue(GSConstChat.CHAT_FAVORITE);
        sql.addIntValue(GSConstChat.CHAT_KBN_USER);
        sql.addIntValue(usrSid);
        sql.addIntValue(GSConstChat.CHAT_KBN_GROUP);
        sql.addIntValue(usrSid);
        sql.addIntValue(GSConstChat.CHAT_GROUP_LOGIC_NOT_DELETE);

        return sql;
    }
}
