package jp.groupsession.v2.cht.search;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import jp.groupsession.v2.cht.GSConstChat;
import jp.groupsession.v2.cht.dao.ChatDao;
import jp.groupsession.v2.cht.dao.ChtUserViewDao;
import jp.groupsession.v2.cht.model.ChatMessageModel;
import jp.groupsession.v2.cht.model.ChtAdmConfModel;
import jp.groupsession.v2.cht.search.ChatMessageSearchRequest.EnumTarget;
import jp.groupsession.v2.cht.search.position.EnumPositionType;
import jp.groupsession.v2.cht.search.position.OffsetPosition;
import jp.groupsession.v2.cmn.GSConst;

/**
 * <p> チャット検索 ビジネスロジック
 * <p> ChatMessageSearchRequestで検索を行う
 * <p>
 */
public class ChatMessageSearcher {
    /** コネクション */
    private final Connection con__;
    /** 検索設定パラメータ */
    private final ChatMessageSearchRequest searchReq__;

    /** チャット管理者設定 既読表示判定に使用
    */
    private final ChtAdmConfModel adminMdl__;

    /**
     * @param con
     * @param searchReq
     * @param adminMdl
     */
    public ChatMessageSearcher(
            Connection con,
            ChatMessageSearchRequest searchReq,
            ChtAdmConfModel adminMdl) {
        con__ = con;
        searchReq__ = searchReq;
        adminMdl__ = adminMdl;
    }
    /**
     * 検索を実行する
     * @return 実行結果
     * @throws SQLException
    */
    public ChatMessageSearchResult execute() throws SQLException, TargetMessageWrongException {
        ChatDao chtDao = new ChatDao(con__);

        ChatMessageSearchResult ret = new ChatMessageSearchResult();
        ChatMessageSearchInfo info = null;
        int partnerViewSid = 0;
        boolean kidokuFlg = false;

        int finalOrder = searchReq__.getOrder();
        EnumPositionType reqPos = searchReq__.getPosition().getType();

        //検索件数の取得
        if (searchReq__.getTarget() == EnumTarget.USERCHAT) {
            info = chtDao.getSearchInfoUserMessage(searchReq__);
        }
        if (searchReq__.getTarget() == EnumTarget.GROUPCHAT) {
            info = chtDao.getSearchInfoGroupMessage(searchReq__);
        }

        ret.hitCount__ = info.getHitCnt();
        if (ret.hitCount__ == 0) {
            return ret;
        }

        if (searchReq__.getTarget() == EnumTarget.USERCHAT) {
            //既読の表示が許可されている場合相手の閲覧情報を取得
            ChtUserViewDao cuvDao = new ChtUserViewDao(con__);
            if (adminMdl__.getCacReadFlg() == 1) {
                partnerViewSid = cuvDao.selectCudSid(
                    searchReq__.getChatPairSid(),
                    searchReq__.getChatTargetUserSid());
            }

            if (adminMdl__.getCacReadFlg() == GSConstChat.KIDOKU_DISP) {
                kidokuFlg = true;
            }
        }

        //検索リクエストの取得位置で検索方法を変える(未読メッセージ)
        if (searchReq__.getPosition().getType() == EnumPositionType.MIDOKUSTART) {

            //未読メッセージが取得件数以下の場合 最終行から取得
            if (info.getHitCnt() <= info.getViewCnt() + searchReq__.getLimit()) {
                searchReq__.setPositionStr(EnumPositionType.NEWEST.name());
                if (info.getHitCnt() < searchReq__.getLimit()) {
                    ret.resultOffset__ = 0;
                } else {
                    ret.resultOffset__ = info.getHitCnt() - searchReq__.getLimit();
                }

            } else {
                if (info.getViewSid() > 0) {

                    //未読メッセージの先頭位置を取得する
                    ChatMessageSearchRequest midokuTopReq = searchReq__.clone();
                    midokuTopReq.setPositionStr(
                        String.format(
                            "%s:%d",
                            EnumPositionType.AFTER_MESSAGE.name(),
                            info.getViewSid()));
                    midokuTopReq.setLimit(1);
                    List<ChatMessageModel> topMsg = List.of();
                    if (searchReq__.getTarget() == EnumTarget.USERCHAT) {
                        topMsg = chtDao.searchUserMessage(
                            info.getViewSid(),
                            partnerViewSid,
                            kidokuFlg,
                            midokuTopReq);
                    }
                    if (searchReq__.getTarget() == EnumTarget.GROUPCHAT) {
                        topMsg = chtDao.searchGroupMessage(
                            info.getViewSid(),
                            midokuTopReq);
                    }
                    //1件も取得できない場合は全件既読状態
                    if (topMsg.size() == 0) {
                        searchReq__.setPositionStr(EnumPositionType.NEWEST.name());
                        ret.resultOffset__ = info.getHitCnt() - searchReq__.getLimit();
                    } else {
                    //最終既読メッセージの次のメッセージ周辺取得
                        searchReq__.setPositionStr(
                            String.format(
                                "%s:%d",
                                EnumPositionType.ARROUND_MESSAGE.name(),
                                topMsg.get(0).getMessageSid()));
                        searchReq__.setOrder(GSConst.ORDER_KEY_ASC);
                        if (searchReq__.getTarget() == EnumTarget.USERCHAT) {
                            info = chtDao.getSearchInfoUserMessage(searchReq__);
                        }
                        if (searchReq__.getTarget() == EnumTarget.GROUPCHAT) {
                            info = chtDao.getSearchInfoGroupMessage(searchReq__);
                        }
                    }
                //1件も閲覧していない
                } else {
                    ret.resultOffset__ = 0;
                    searchReq__.setPosition(new OffsetPosition(ret.resultOffset__));
                }
            }
        }

        //検索リクエストの取得位置で検索方法を変える(オフセット)
        if (searchReq__.getPosition().getType() == EnumPositionType.OFFSET) {
            OffsetPosition position = (OffsetPosition) searchReq__.getPosition();
            ret.resultOffset__ = position.getOffset();
            //最大件数を超える場合は検索しない
            if (position.getOffset() > ret.hitCount__) {
                return ret;
            }
        }
        //指定メッセージSIDポジション指定で指定メッセージSIDが検索結果に含まれない
        if (info.getTargetMessageHave() == 0
            && (searchReq__.getPosition().getType() == EnumPositionType.BEFORE_MESSAGE
                || searchReq__.getPosition().getType() == EnumPositionType.AFTER_MESSAGE
                || searchReq__.getPosition().getType() == EnumPositionType.ARROUND_MESSAGE)
            ) {
            throw new TargetMessageWrongException();
        }
        //検索リクエストの取得位置で検索方法を変える(指定メッセージ周辺)
        if (searchReq__.getPosition().getType() == EnumPositionType.ARROUND_MESSAGE) {
            if (info.getTargetMessageOffset() <= searchReq__.getLimit() / 2) {
                ret.resultOffset__ = 0;
                searchReq__.setPosition(new OffsetPosition(0));
                searchReq__.setOrder(GSConst.ORDER_KEY_ASC);

            } else if (
                //取得位置が最大件数-15より高い時、新しい方から30件を取得
                info.getTargetMessageOffset()
                     >= info.getHitCnt() - searchReq__.getLimit() / 2) {
                ret.resultOffset__ = (info.getHitCnt() - searchReq__.getLimit());
                searchReq__.setPositionStr(EnumPositionType.NEWEST.name());
            } else {
                ret.resultOffset__ =
                    info.getTargetMessageOffset()
                     - searchReq__.getLimit() / 2;

                searchReq__.setPosition(new OffsetPosition(ret.resultOffset__));

            }
        }


        //並び順を設定
        Comparator<ChatMessageModel> comp = Comparator.comparing(ChatMessageModel::getMessageSid);
        if (searchReq__.getTargetPinKbn() == GSConstChat.CHAT_MESSAGE_GET_ALL) {
            if (finalOrder == GSConst.ORDER_KEY_DESC) {
                comp = comp.reversed();
            }
        } else {
            //ピンどめ取得用
            comp = new Comparator<ChatMessageModel>() {
                @Override
                public int compare(ChatMessageModel o1, ChatMessageModel o2) {
                    return 1;
                }
            };
        }

        if (searchReq__.getTarget() == EnumTarget.USERCHAT) {
            ret.list__ =
                chtDao.searchUserMessage(
                    info.getViewSid(),
                    partnerViewSid,
                    kidokuFlg,
                    searchReq__)
                .stream()
                .sorted(comp)
                .collect(Collectors.toList());

        }

        if (searchReq__.getTarget() == EnumTarget.GROUPCHAT) {
            ret.list__ =
                chtDao.searchGroupMessage(
                    info.getViewSid(),
                    searchReq__)
                .stream()
                .sorted(comp)
                .collect(Collectors.toList());
        }

        //前方取得の場合 offset指定時での開始位置を計算
        if (reqPos == EnumPositionType.BEFORE_MESSAGE) {
            if (finalOrder == 1) {
                ret.resultOffset__ = info.getHitCnt() - info.getTargetMessageOffset();
            } else {
                ret.resultOffset__ = info.getTargetMessageOffset()
                    - ret.list__.size();
            }
        //後ろ取得の場合 offset指定時での開始位置を計算
        } else if (reqPos == EnumPositionType.AFTER_MESSAGE) {
            if (finalOrder == 1) {
                ret.resultOffset__ =  info.getHitCnt()
                    - info.getTargetMessageOffset()
                    - ret.list__.size() - 1;
            } else {
                ret.resultOffset__ = info.getTargetMessageOffset() + 1;
            }
        //逆順取得時のオフセット位置に調整
        } else if (reqPos != EnumPositionType.OFFSET
            && finalOrder == 1) {
            ret.resultOffset__ = info.getHitCnt() - ret.resultOffset__ - ret.list__.size();
        }

        return ret;

    }





}
