package jp.groupsession.v2.rng;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import jp.groupsession.v2.cmn.GSConst;
import jp.groupsession.v2.cmn.GSConstCommon;
import jp.groupsession.v2.cmn.GSContext;
import jp.groupsession.v2.cmn.GroupSession;
import jp.groupsession.v2.cmn.IPositionListener;
import jp.groupsession.v2.cmn.biz.CommonBiz;
import jp.groupsession.v2.cmn.biz.UserBiz;
import jp.groupsession.v2.cmn.config.PluginConfig;
import jp.groupsession.v2.cmn.dao.MlCountMtController;
import jp.groupsession.v2.cmn.dao.base.CmnPluginAdminDao;
import jp.groupsession.v2.cmn.dao.base.CmnPositionDao;
import jp.groupsession.v2.cmn.model.RequestModel;
import jp.groupsession.v2.cmn.model.base.CmnPositionModel;
import jp.groupsession.v2.cmn.model.base.CmnUsrmInfModel;
import jp.groupsession.v2.rng.biz.RngTemplateBiz;
import jp.groupsession.v2.rng.dao.RngActionparamDao;
import jp.groupsession.v2.rng.dao.RngTemplateActionDao;
import jp.groupsession.v2.rng.dao.RngTemplateDao;
import jp.groupsession.v2.rng.dao.RngTemplatecategoryAdmDao;
import jp.groupsession.v2.rng.model.RngActionparamModel;
import jp.groupsession.v2.rng.model.RngTemplateActionModel;
import jp.groupsession.v2.rng.model.RngTemplateModel;
import jp.groupsession.v2.rng.model.RngTemplatecategoryAdmModel;
import jp.groupsession.v2.sml.GSConstSmail;
import jp.groupsession.v2.sml.SmlSender;
import jp.groupsession.v2.sml.model.SmlSenderModel;
import jp.groupsession.v2.struts.msg.GsMessage;
import jp.groupsession.v2.usr.GSConstUser;
/**
 * <br>[機  能]
 * <br>[解  説]
 * <br>[備  考]
 *
 * @author  JTS
 */
public class RngIPositionListenerImpl implements IPositionListener {

    /** ログ */
    private Log log__ = LogFactory.getLog(getClass());

    /**
     * <p>役職削除時に実行される
     * @param con DBコネクション
     * @param cntCon 採番コントローラ
     * @param reqMdl リクエスト情報
     * @param posSid 役職SID
     * @param eusid 更新者ユーザSID
     * @throws Exception
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    public void deletePosition(
        Connection con, MlCountMtController cntCon,
        RequestModel reqMdl, int posSid, int eusid) throws Exception {

        RngActionparamDao rapDao = new RngActionparamDao(con);
        List<RngActionparamModel> rapMdlList = rapDao.select();

        if (rapMdlList == null || rapMdlList.isEmpty()) {
            return;
        }

        CmnPositionDao positionDao = new CmnPositionDao(con);
        CmnPositionModel posMdl = positionDao.getPosInfo(posSid);
        String posName = posMdl.getPosName();

        ObjectMapper mapper = new ObjectMapper();
        List<RngActionparamModel> updateList = new ArrayList<>();
        for (RngActionparamModel rapMdl : rapMdlList) {
            String conditionJson = rapMdl.getRapConditionJson();
            JsonNode node;
            try {
                node = mapper.readTree(conditionJson);
            } catch (JsonProcessingException e) {
                //条件JSONが不正な場合はエラーを足せないためスキップ
                continue;
            }

            JsonNode conditionListNode = node.get("conditionList");
            if (conditionListNode == null || conditionListNode.isEmpty()) {
                continue;
            }

            boolean updateFlg = false;
            for (JsonNode conditionNode : conditionListNode) {
                if (conditionNode.get("paramValue") != null
                    && conditionNode.get("paramValue").asInt()
                        == RngConst.API_COMPARE_PARAM_POSITION
                    && conditionNode.get("compareTarget") != null
                    && conditionNode.get("compareTarget").asInt() == posSid) {
                    ObjectNode replaceNode = (ObjectNode) conditionNode;
                    replaceNode.put("errorTargetName", posName);
                    replaceNode.put("errorFlg", true);
                    updateFlg = true;
                }
            }
            if (updateFlg) {
                rapMdl.setRapConditionJson(node.toString());
                updateList.add(rapMdl);
            }
        }
        if (updateList.isEmpty()) {
            return;
        }

        for (RngActionparamModel mdl : updateList) {
            rapDao.update(mdl);
        }

        RngTemplateActionDao rtaDao = new RngTemplateActionDao(con);
        List<Integer> rtpaSidList = updateList.stream()
            .map(RngActionparamModel::getRtpaSid)
            .distinct()
            .collect(Collectors.toList());
        List<RngTemplateActionModel> rtaMdlList = rtaDao.select(rtpaSidList);
        rtaMdlList = new ArrayList<>(rtaMdlList.stream()
            .collect(Collectors.toMap(
                mdl -> mdl.getRtpSid(),
                mdl -> mdl,
                (mdl1, mdl2) -> {
                    if (mdl1.getRtpVer() < mdl2.getRtpVer()) {
                        return mdl2;
                    }
                    return mdl1;
                })
            ).values());

        RngTemplateDao templateDao = new RngTemplateDao(con);
        RngTemplatecategoryAdmDao rtcaDao = new RngTemplatecategoryAdmDao(con);
        ArrayList<RngTemplateModel> templateModelList = new ArrayList<>();
        List<Integer> categorySidList = new ArrayList<>();

        for (RngTemplateActionModel rtaMdl : rtaMdlList) {
            RngTemplateModel templateMdl
                = templateDao.select(rtaMdl.getRtpSid(), rtaMdl.getRtpVer());
            templateModelList.add(templateMdl);
            ArrayList<RngTemplatecategoryAdmModel> rtaList
                = rtcaDao.select(templateMdl.getRtcSid());
            categorySidList.addAll(rtaList.stream()
                .map(RngTemplatecategoryAdmModel::getRtcSid)
                .collect(Collectors.toList()));
        }

        UserBiz userBiz = new UserBiz();
        List<CmnUsrmInfModel> userList
                = userBiz.getBelongUserList(con, GSConstUser.SID_ADMIN, new ArrayList<>());
        List<Integer> sysAdminUserList = userList.stream()
            .map(CmnUsrmInfModel::getUsrSid)
            .collect(Collectors.toList());
        CmnPluginAdminDao cpaDao = new CmnPluginAdminDao(con);
        List<Integer> pluginAdminUserList = cpaDao.getPluginAdminUsrSid(RngConst.PLUGIN_ID_RINGI);

        List<Integer> adminUserList = new ArrayList<>();
        adminUserList.addAll(sysAdminUserList);
        adminUserList.addAll(pluginAdminUserList);
        adminUserList = adminUserList.stream()
            .distinct()
            .collect(Collectors.toList());
        adminUserList.removeIf(sid -> sid == GSConst.SYSTEM_USER_ADMIN);
        adminUserList.removeIf(sid -> sid == GSConst.SYSTEM_USER_MAIL);

        //システム管理者, プラグイン管理者ユーザにショートメールを送信
        __sendSmailAdmin(adminUserList, templateModelList, con, cntCon, reqMdl);

        //カテゴリ管理者にショートメールを送信
        __sendSmailCategory(adminUserList, templateModelList, categorySidList, con, cntCon, reqMdl);
    }

    /**
     * <br>[機  能] システム管理者，稟議のプラグイン管理者にメールを送信する
     * <br>[解  説]
     * <br>[備  考]
     * @param adminUserList システム管理者，稟議のプラグイン管理者のユーザSID一覧
     * @param templateModelList テンプレート情報一覧
     * @param con DBコネクション
     * @param cntCon 採番コントローラ
     * @param reqMdl リクエスト情報
     * @throws Exception
     */
    private void __sendSmailAdmin(
        List<Integer> adminUserList,
        List<RngTemplateModel> templateModelList,
        Connection con, MlCountMtController cntCon, RequestModel reqMdl) throws Exception {

        if (adminUserList.isEmpty()) {
            return;
        }

        //ショートメール送信用モデルを作成する。
        SmlSenderModel smlModel = new SmlSenderModel();
        //送信者(システムメールを指定)
        smlModel.setSendUsid(GSConst.SYSTEM_USER_MAIL);
        //TO
        smlModel.setSendToUsrSidArray(adminUserList);

        //タイトル
        GsMessage gsMsg = new GsMessage(reqMdl);
        String msg = gsMsg.getMessage("rng.71");
        String msg2 = gsMsg.getMessage("rng.129");
        String title = msg + " " + msg2;
        smlModel.setSendTitle(title);

        //本文
        RngTemplateBiz templateBiz = new RngTemplateBiz();
        String bodyText = templateBiz.getTemplateSmailTuutiBody(reqMdl, templateModelList, true);
        if (bodyText.length() > GSConstCommon.MAX_LENGTH_SMLBODY) {
            String textMessage = gsMsg.getMessage("cmn.mail.omit");
            bodyText = textMessage + "\r\n\r\n" + bodyText;
            bodyText = bodyText.substring(0, GSConstCommon.MAX_LENGTH_SMLBODY);
        }
        smlModel.setSendBody(bodyText);
        //メール形式
        smlModel.setSendType(GSConstSmail.SAC_SEND_MAILTYPE_NORMAL);
        //マーク
        smlModel.setSendMark(GSConstSmail.MARK_KBN_NONE);

        //メール送信処理開始
        CommonBiz cmnBiz = new CommonBiz();
        PluginConfig pluginConfig = cmnBiz.getPluginConfigForMain(con, reqMdl);
        String appRootPath = (String) GroupSession.getContext().get(GSContext.APP_ROOT_PATH);
        SmlSender sender =
            new SmlSender(
                    con,
                    cntCon,
                    smlModel, pluginConfig,
                    appRootPath,
                    reqMdl);
        sender.execute();
    }

    /**
     * <br>[機  能] カテゴリ管理者にメールを送信する
     * <br>[解  説]
     * <br>[備  考]
     * @param adminUserList システム管理者，稟議のプラグイン管理者のユーザSID一覧
     * @param templateModelList テンプレート情報一覧
     * @param categorySidList カテゴリSID一覧
     * @param con DBコネクション
     * @param cntCon 採番コントローラ
     * @param reqMdl リクエスト情報
     * @throws Exception
     */
    private void __sendSmailCategory(
        List<Integer> adminUserList,
        List<RngTemplateModel> templateModelList,
        List<Integer> categorySidList,
        Connection con, MlCountMtController cntCon, RequestModel reqMdl) throws Exception {

        if (categorySidList.isEmpty()) {
            return;
        }

        log__.error("adminUserListの中身："
            + String.join(",",
                adminUserList.stream().map(String::valueOf).collect(Collectors.toList())));

        RngTemplateBiz templateBiz = new RngTemplateBiz();
        //ユーザ指定部分を取得 (キー：ユーザSID，値：カテゴリSID一覧)
        Map<Integer, Set<Integer>> userCategoryMap
            = templateBiz.getUserCategoryMap(con, categorySidList, adminUserList);

        if (userCategoryMap.isEmpty()) {
            return;
        }

        Map<Integer, List<RngTemplateModel>> categoryTemplateNameMap = templateModelList.stream()
            .collect(Collectors.groupingBy(
                RngTemplateModel::getRtcSid,
                Collectors.mapping(mdl -> mdl, Collectors.toList())
            ));

        //ショートメール送信用モデルを作成する。
        SmlSenderModel smlModel = new SmlSenderModel();
        //送信者(システムメールを指定)
        smlModel.setSendUsid(GSConst.SYSTEM_USER_MAIL);

        //タイトル
        GsMessage gsMsg = new GsMessage(reqMdl);
        String msg = gsMsg.getMessage("rng.71");
        String msg2 = gsMsg.getMessage("rng.129");
        String title = msg + " " + msg2;
        smlModel.setSendTitle(title);

        //メール形式
        smlModel.setSendType(GSConstSmail.SAC_SEND_MAILTYPE_NORMAL);
        //マーク
        smlModel.setSendMark(GSConstSmail.MARK_KBN_NONE);

        CommonBiz cmnBiz = new CommonBiz();
        PluginConfig pluginConfig = cmnBiz.getPluginConfigForMain(con, reqMdl);
        String appRootPath = (String) GroupSession.getContext().get(GSContext.APP_ROOT_PATH);

        List<Integer> doneUser = new ArrayList<>();
        for (Entry<Integer, Set<Integer>> entry : userCategoryMap.entrySet()) {
            Set<Integer> sendTarget = new HashSet<>();
            int usrSid = entry.getKey();
            if (doneUser.contains(usrSid)) {
                continue;
            }
            sendTarget.add(usrSid);

            //送信先のユーザと全く同じカテゴリの管理者の場合は一緒にメールを送信する
            userCategoryMap.entrySet().stream()
                .filter(e -> e.getValue().equals(entry.getValue()))
                .map(e -> e.getKey())
                .forEach(e -> {
                    sendTarget.add(e);
                });

            //本文
            List<RngTemplateModel> targetTemplateList = new ArrayList<>();
            for (int categorySid : entry.getValue()) {
                targetTemplateList.addAll(categoryTemplateNameMap.get(categorySid));
            }
            String bodyText
                = templateBiz.getTemplateSmailTuutiBody(reqMdl, targetTemplateList, false);
            if (bodyText.length() > GSConstCommon.MAX_LENGTH_SMLBODY) {
                String textMessage = gsMsg.getMessage("cmn.mail.omit");
                bodyText = textMessage + "\r\n\r\n" + bodyText;
                bodyText = bodyText.substring(0, GSConstCommon.MAX_LENGTH_SMLBODY);
            }
            smlModel.setSendBody(bodyText);
            //TO
            log__.error("sendTarget:"
                + sendTarget.stream().map(String::valueOf).collect(Collectors.toList()));
            smlModel.setSendToUsrSidArray(new ArrayList<Integer>(sendTarget));

            SmlSender sender =
            new SmlSender(
                    con,
                    cntCon,
                    smlModel, pluginConfig,
                    appRootPath,
                    reqMdl);
            sender.execute();

            doneUser.addAll(sendTarget);
        }
    }
}
