package jp.groupsession.v2.rng;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.action.ActionErrors;
import org.apache.struts.action.ActionMessage;

import jp.co.sjts.util.NullDefault;
import jp.co.sjts.util.StringUtil;
import jp.co.sjts.util.ValidateUtil;
import jp.co.sjts.util.struts.StrutsUtil;
import jp.groupsession.v2.cmn.GSConst;
import jp.groupsession.v2.cmn.GSConstCommon;
import jp.groupsession.v2.cmn.GSValidateUtil;
import jp.groupsession.v2.cmn.dao.GroupDao;
import jp.groupsession.v2.cmn.dao.base.CmnApiConnectParamDao;
import jp.groupsession.v2.cmn.dao.base.CmnPositionDao;
import jp.groupsession.v2.cmn.dao.base.CmnUsrmDao;
import jp.groupsession.v2.cmn.formbuilder.EnumFormModelKbn;
import jp.groupsession.v2.cmn.formbuilder.FormBuilder;
import jp.groupsession.v2.cmn.formbuilder.FormCell;
import jp.groupsession.v2.cmn.formmodel.CheckBox;
import jp.groupsession.v2.cmn.formmodel.ComboBox;
import jp.groupsession.v2.cmn.formmodel.GroupComboModel;
import jp.groupsession.v2.cmn.formmodel.RadioButton;
import jp.groupsession.v2.cmn.formmodel.SimpleUserSelect;
import jp.groupsession.v2.cmn.formmodel.UserGroupSelectModel;
import jp.groupsession.v2.cmn.model.RequestModel;
import jp.groupsession.v2.cmn.model.base.CmnApiConnectParamModel;
import jp.groupsession.v2.cmn.model.base.CmnGroupmModel;
import jp.groupsession.v2.cmn.model.base.CmnPositionModel;
import jp.groupsession.v2.rng.biz.RngTemplateBiz;
import jp.groupsession.v2.rng.model.AddRngActionModel;
import jp.groupsession.v2.rng.model.ChildParamInfoModel;
import jp.groupsession.v2.rng.model.ListParamInfoModel;
import jp.groupsession.v2.rng.model.ParamInfoModel;
import jp.groupsession.v2.rng.model.RngRapConditionJsonModel;
import jp.groupsession.v2.rng.model.RngRapConditionModel;
import jp.groupsession.v2.rng.model.RngRapParamJsonModel;
import jp.groupsession.v2.rng.model.RngRapParamModel;
import jp.groupsession.v2.rng.model.RngValidateParamModel;
import jp.groupsession.v2.struts.msg.GsMessage;
import jp.groupsession.v2.usr.GSConstUser;

/**
 * <br>[機  能] 稟議の入力チェックを行うクラス
 * <br>[解  説]
 * <br>[備  考]
 *
 * @author JTS
 */
public class RngValidate {

    /** Logging インスタンス */
    private static Log log__ = LogFactory.getLog(RngValidate.class);

    /** epefix 稟議*/
    private static final String RNG_E_RINGI__ = "ringi";
    /** 稟議テンプレートタイトル文字数 */
    protected static final int RNG_TPL_TITLE = 100;
    /** 稟議テンプレートフォーマット文字数 */
    protected static final int RNG_TPL_FORMAT = 1000;

    /** GSメッセージ */
    private final GsMessage gsMsg__;

    /**
     * @param gsMsg
     */
    public RngValidate(GsMessage gsMsg) {
        gsMsg__ = gsMsg;
    }

    /**
     * <br>[機  能] テキストフィールドの入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors ActionErrors
     * @param checkObject エラーメッセージ表示テキスト(例："名前" 例："コメント"）
     * @param field チェックするフィールド
     * @param strField チェックするフィールドの文字列
     * @param maxLength 最大文字数
     * @param hisuFlg 入力が必須か true:必須 false:必須ではない
     * @return ActionErrors
     */
    public static ActionErrors validateCmnFieldText(
        ActionErrors errors,
        String checkObject,
        String field,
        String strField,
        int maxLength,
        boolean hisuFlg) {
        log__.debug(strField + " のチェックを行います。");
        ActionMessage msg = null;

        String eprefix = RNG_E_RINGI__;
        String fieldfix = checkObject + ".";

        if (StringUtil.isNullZeroString(field)) {
            if (hisuFlg) {
                //未入力チェック
                msg = new ActionMessage("error.input.required.text", checkObject);
                StrutsUtil.addMessage(
                        errors, msg, eprefix + fieldfix + strField);
            }
            return errors;
        }

        if (ValidateUtil.isSpace(field)) {
            //スペースのみチェック
            msg = new ActionMessage("error.input.spase.only", checkObject);
            StrutsUtil.addMessage(
                    errors, msg, eprefix + fieldfix + strField);
            return errors;
        }

        if (ValidateUtil.isSpaceStart(field)) {
            //先頭スペースチェック
            msg = new ActionMessage("error.input.spase.start", checkObject);
            StrutsUtil.addMessage(errors, msg,
                    eprefix + fieldfix + strField);
            return errors;
        }

        if (ValidateUtil.isTab(field)) {
            //タブスペースチェック
            msg = new ActionMessage("error.input.tab.text", checkObject);
            StrutsUtil.addMessage(errors, msg,
                    eprefix + fieldfix + strField);
            return errors;
        }

        if (field.length() > maxLength) {
            //MAX桁チェック
            msg = new ActionMessage("error.input.length.text", checkObject,
                    maxLength);
            StrutsUtil.addMessage(
                    errors, msg, eprefix + fieldfix + strField);
            return errors;
        }

        if (!GSValidateUtil.isGsJapaneaseStringTextArea(field)) {
            //利用不可能な文字を入力した場合
            String nstr = GSValidateUtil.getNotGsJapaneaseStringTextArea(field);
            msg = new ActionMessage("error.input.njapan.text", checkObject, nstr);
            StrutsUtil.addMessage(
                    errors, msg, eprefix + fieldfix + strField);
            return errors;
        }
        return errors;
    }

    /**
     * <br>[機  能] テキストエリアの入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors ActionErrors
     * @param checkObject エラーメッセージ表示テキスト(例："名前" 例："コメント"）
     * @param field チェックするフィールド
     * @param strField チェックするフィールドの文字列
     * @param maxLength 最大文字数
     * @param hisuFlg 入力が必須か true:必須 false:必須ではない
     * @return ActionErrors
     */
    public static ActionErrors validateFieldTextArea(
            ActionErrors errors,
            String checkObject,
            String field,
            String strField,
            int maxLength,
            boolean hisuFlg) {
        log__.debug(strField + " のチェックを行います。");
        ActionMessage msg = null;

        String eprefix = RNG_E_RINGI__;
        String fieldfix = checkObject + ".";

        if (StringUtil.isNullZeroString(field)) {
            if (hisuFlg) {
                //未入力チェック
                msg = new ActionMessage("error.input.required.text", checkObject);
                StrutsUtil.addMessage(
                        errors, msg, eprefix + fieldfix + strField);
            }
            return errors;
        }

        if (field.length() > maxLength) {
            //MAX桁チェック
            msg = new ActionMessage("error.input.length.textarea", checkObject,
                    maxLength);
            StrutsUtil.addMessage(
                    errors, msg, eprefix + fieldfix + strField);

        //スペース、改行のみチェック
        } else if (ValidateUtil.isSpaceOrKaigyou(field)) {
            msg = new ActionMessage("error.input.spase.cl.only", checkObject);
            StrutsUtil.addMessage(errors, msg,
                    eprefix + fieldfix + strField);

        } else if (!GSValidateUtil.isGsJapaneaseStringTextArea(field)) {
            //利用不可能な文字を入力した場合
            String nstr = GSValidateUtil.getNotGsJapaneaseStringTextArea(field);
            msg = new ActionMessage("error.input.njapan.text", checkObject, nstr);
            StrutsUtil.addMessage(
                    errors, msg, eprefix + fieldfix + strField);
        }

        return errors;
    }

    /**
     * <br>[機  能] 削除する稟議の選択チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors ActionErrors
     * @param checkObject エラーメッセージ表示テキスト(例："名前" 例："コメント"）
     * @param delRng 削除する稟議
     * @param strDelRng 削除する稟議の文字列
     * @return errors
     */
    public static ActionErrors validateDeleteRng(
            ActionErrors errors,
            String checkObject,
            String[] delRng,
            String strDelRng) {
        ActionMessage msg = null;
        String eprefix = RNG_E_RINGI__;
        String fieldfix = checkObject + ".";
        if (delRng == null || delRng.length <= 0) {
            //未選択チェック
            msg = new ActionMessage("error.select.required.text", checkObject);
            log__.debug("msg  = " + msg);
            log__.debug("eprefix  = " + eprefix);
            log__.debug("fieldfix  = " + fieldfix);
            log__.debug("strDelRng  = " + strDelRng);

            StrutsUtil.addMessage(errors, msg, eprefix + fieldfix + strDelRng);
        }

        return errors;
    }

    /**
     * <br>[機能] 数量の入力チェックを行う
     * <br>[解説]
     * <br>[備考]
     * @param errors ActionErrors
     * @param nameStr テキストの入力値
     * @param checkObject チェック対象
     * @param maxLength 最大文字数
     * @return ActionErrors
     */
    public static ActionErrors validateNumberInt(
        ActionErrors errors,
        String nameStr,
        String checkObject,
        int maxLength) {

        ActionMessage msg = null;

        //未入力チェック
        if (StringUtil.isNullZeroString(nameStr)) {
            msg = new ActionMessage("error.input.required.text", checkObject);
            StrutsUtil.addMessage(errors, msg, checkObject);
            return errors;
        }

        //数値チェック
        if (!ValidateUtil.isNumber(nameStr)) {
            msg = new ActionMessage("error.input.number.text", checkObject);
            StrutsUtil.addMessage(errors, msg, checkObject);
        }

        //MAX桁チェック
        if (nameStr.length() > maxLength) {
            msg = new ActionMessage(
                    "error.input.length.text", checkObject, String.valueOf(maxLength));
            StrutsUtil.addMessage(errors, msg, checkObject);
        }
        //スペース（改行）のみチェック
        if (ValidateUtil.isSpace(nameStr)) {
            msg = new ActionMessage("error.input.spase.only", checkObject);
            StrutsUtil.addMessage(errors, msg, checkObject);
        }
        //先頭スペースチェック
        if (ValidateUtil.isSpaceStart(nameStr)) {
            msg = new ActionMessage("error.input.spase.start", checkObject);
            StrutsUtil.addMessage(errors, msg, checkObject);
        }
        //JIS第2水準チェック
        if (!GSValidateUtil.isGsJapaneaseString(nameStr)) {
            //利用不可能な文字を入力した場合
            String nstr = GSValidateUtil.getNotGsJapaneaseString(nameStr);
            msg = new ActionMessage("error.input.njapan.text", checkObject, nstr);
            StrutsUtil.addMessage(errors, msg, checkObject);
        }

        return errors;
    }

    /**
     * <br>[機能] ラジオの入力チェックを行う
     * <br>[解説]
     * <br>[備考]
     * @param errors ActionErrors
     * @param nameStr テキストの入力値
     * @param checkObject チェック対象
     * @return ActionErrors
     */
    public static ActionErrors validateRadio(
        ActionErrors errors,
        String nameStr,
        String checkObject) {

        ActionMessage msg = null;

        //未入力チェック
        if (nameStr.equals("0")) {
            msg = new ActionMessage("error.input.required.text", checkObject);
            StrutsUtil.addMessage(errors, msg, checkObject);
            return errors;
        }
        return errors;
    }

    /**
     * <br>[機能] 実行条件>比較文字列（入力フォーム）の入力チェックを行う
     * <br>[解説]
     * <br>[備考]
     * @param errors エラーメッセージリスト
     * @param compareTarget 比較文字列
     * @param row 行数
     * @param reqMdl リクエストモデル
     * @param numFlg 数値入力フラグ（true:数値入力, false:文字列入力）
     * @return true:入力エラーなし, false:入力エラーあり
     */
    public List<String> validateCompareTargetForm(
        List<String> errors,
        String compareTarget,
        int row,
        RequestModel reqMdl,
        boolean numFlg) {

        GsMessage gsMsg = new GsMessage(reqMdl);

        String conditionStr = "[" + gsMsg__.getMessage("rng.rng310.8")
            + "&nbsp;(" + gsMsg__.getMessage("cmn.line",
                new String[] {String.valueOf(row)}) + ")]:";

        if (numFlg) {
            //数値入力の場合
            if (!ValidateUtil.isNumber(compareTarget)) {
                //数値入力チェック
                errors.add(conditionStr
                    + gsMsg.getMessage("cmn.error.number",
                        new String[] {gsMsg.getMessage("rng.rng310.53")}));
            }
        } else {
            //文字列入力の場合
            if (compareTarget.length() > RngConst.API_COMPARE_TARGET_MAXLENGTH) {
                //最大文字数チェック
                errors.add(conditionStr
                    + gsMsg.getMessage("cmn.plz.max",
                        new String[] {gsMsg.getMessage("rng.rng310.53"),
                    String.valueOf(RngConst.API_COMPARE_TARGET_MAXLENGTH)}));
            }
            //スペースチェック
            if (ValidateUtil.isSpace(compareTarget)) {
                //スペースのみチェック
                errors.add(conditionStr
                    + gsMsg.getMessage("cmn.error.space.only",
                        new String[] {gsMsg.getMessage("rng.rng310.53")}));
            } else if (ValidateUtil.isSpaceStart(compareTarget)) {
                //先頭スペースチェック
                errors.add(conditionStr
                    + gsMsg.getMessage("cmn.error.space.start",
                        new String[] {gsMsg.getMessage("rng.rng310.53")}));
            }
            if (ValidateUtil.isTab(compareTarget)) {
                //タブスペースチェック
                errors.add(conditionStr
                    + gsMsg.getMessage("cmn.error.tab",
                        new String[] {gsMsg.getMessage("rng.rng310.53")}));
            }
            if (!GSValidateUtil.isGsJapaneaseString(compareTarget)) {
                //使用禁止文字列チェック
                String nstr = GSValidateUtil.getNotGsJapaneaseString(compareTarget);
                errors.add(conditionStr
                    + gsMsg.getMessage("cmn.plz.input.available",
                        new String[] {gsMsg.getMessage("rng.rng310.53"), nstr}));
            }
        }
        return errors;
    }

    /**
     * <br>[機能] パラメータ情報の入力チェックを行う
     * <br>[解説]
     * <br>[備考]
     * @param errors エラー文字列リスト
     * @param rapParamMdlList パラメータ情報
     * @param araMdl 決裁後アクション情報
     * @param templateJson Json文字列
     * @param paramKbn パラメータ区分（0:パスパラメータ, 1:クエリパラメータ, 2:リクエストボディ）
     * @param con コネクション
     * @param reqMdl リクエストモデル
     * @return エラー文字列リスト
     * @throws SQLException
     */
    public List<String> validateParam(
        List<String> errors,
        List<RngRapParamModel> rapParamMdlList,
        AddRngActionModel araMdl,
        String templateJson,
        int paramKbn,
        Connection con,
        RequestModel reqMdl) throws SQLException {

        CmnApiConnectParamDao cacpDao = new CmnApiConnectParamDao(con);
        int cacSid = 0;
        int repeatKbn = 0;
        int repeatType = 0;
        String repeatFormId = null;
        if (araMdl != null) {
            repeatKbn = araMdl.getRepeatKbn();
            repeatType = araMdl.getRepeatType();
            cacSid = araMdl.getApiConnectSid();
            if (repeatKbn == RngConst.API_REPEAT_KBN_ON
                && repeatType == RngConst.API_REPEAT_TYPE_FORM) {
                //繰り返し実行対象がフォーム要素の場合、繰り返し実行対象フォームIDを取得
                repeatFormId = araMdl.getRepeatFormId();
            }
        }

        List<String> paramNameList
            = rapParamMdlList.stream()
                .map(r -> r.getName())
                .collect(Collectors.toList());
        List<CmnApiConnectParamModel> cacpMdlList = cacpDao.select(cacSid, paramKbn, paramNameList);

        for (RngRapParamModel rapParamMdl : rapParamMdlList) {
            CmnApiConnectParamModel cacpMdl
                = cacpMdlList.stream()
                .filter(c -> c.getCacpName().equals(rapParamMdl.getName()))
                .findFirst()
                .orElse(null);

            //必須区分
            boolean requiredFlg = false;
            if (cacpMdl != null
                && cacpMdl.getCacpRequiredKbn() == GSConstCommon.PARAM_REQUIRE_ON) {
                requiredFlg = true;
            }
            //パラメータ形式
            boolean fileFlg = false;
            if (paramKbn == GSConstCommon.PARAM_KBN_BODY
                && cacpMdl != null
                && cacpMdl.getCacpParamType() == GSConstCommon.PARAM_TYPE_FILE) {
                fileFlg = true;
            }
            RngValidateParamModel validateMdl = new RngValidateParamModel();
            if (cacpMdl != null) {
                validateMdl.setParamNameDsp(cacpMdl.getCacpNameDisp());
            }
            validateMdl.setFileFlg(fileFlg);
            validateMdl.setRequiredFlg(requiredFlg);
            validateMdl.setRepeatKbn(repeatKbn);
            validateMdl.setRepeatType(repeatType);
            validateMdl.setRepeatFormId(repeatFormId);
            validateMdl.setTemplateJson(templateJson);
            errors = __validateRapParamModel(errors, rapParamMdl, validateMdl, cacSid, con);
        }
        return errors;
    }

    /**
     * <br>[機能] アクションパラメータ（パラメータ情報）の入力チェックを行い、エラーが1つでもあるかを確認する
     * <br>[解説]
     * <br>[備考]
     * @param rapParamJsonMdl アクションパラメータ（パラメータ情報）
     * @param araMdl 決裁後アクション情報
     * @param templateJson Json文字列
     * @param con コネクション
     * @param reqMdl リクエストモデル
     * @return true:アクションパラメータにエラーが存在する, false:アクションパラメータにエラーが存在しない
     * @throws SQLException
     */
    public boolean isValidateParamError(
        RngRapParamJsonModel rapParamJsonMdl,
        AddRngActionModel araMdl,
        String templateJson,
        Connection con,
        RequestModel reqMdl) throws SQLException {

        List<String> errors = new ArrayList<>();
        CmnApiConnectParamDao cacpDao = new CmnApiConnectParamDao(con);
        int cacSid = 0;
        List<CmnApiConnectParamModel> cacpMdlList = new ArrayList<>();
        if (araMdl != null) {
            cacSid = araMdl.getApiConnectSid();
            cacpMdlList = cacpDao.select(cacSid);
        }
        List<CmnApiConnectParamModel> hissuParamList = cacpMdlList.stream()
            .filter(mdl -> mdl.getCacpRequiredKbn() == GSConstCommon.PARAM_REQUIRE_ON)
            .collect(Collectors.toList());

        if (rapParamJsonMdl != null) {
            //パスパラメータのタイプチェック
            for (RngRapParamModel pathMdl : rapParamJsonMdl.getPathParamList()) {
                String paramType = "parameter";
                if (!Objects.equals(pathMdl.getParamType(), paramType)) {
                    return true;
                }
                hissuParamList.removeIf(mdl ->
                    mdl.getCacpKbn() == GSConstCommon.PARAM_KBN_PATH
                        && Objects.equals(mdl.getCacpName(), pathMdl.getName()));
            }

            //クエリパラメータのタイプチェック
            for (RngRapParamModel queryMdl : rapParamJsonMdl.getQueryParamList()) {
                CmnApiConnectParamModel cacpMdl = cacpMdlList.stream()
                    .filter(mdl -> mdl.getCacpKbn() == 1)
                    .filter(mdl -> Objects.equals(queryMdl.getName(), mdl.getCacpName()))
                    .findFirst()
                    .orElse(null);
                String paramType = "";
                if (cacpMdl != null) {
                    if (cacpMdl.getCacpListKbn() == 1) {
                        paramType = "list";
                    } else {
                        paramType = "parameter";
                    }
                    if (!Objects.equals(queryMdl.getParamType(), paramType)) {
                        return true;
                    }
                }
                hissuParamList.removeIf(mdl ->
                    mdl.getCacpKbn() == GSConstCommon.PARAM_KBN_QUERY
                        && Objects.equals(mdl.getCacpName(), queryMdl.getName()));
            }

            //ボディパラメータのタイプチェック
            Set<String> parentNameSet = cacpMdlList.stream()
                .filter(mdl -> mdl.getCacpKbn() == 2)
                .filter(mdl -> mdl.getCacpName().contains("."))
                .map(mdl -> mdl.getCacpName().split("\\.")[0])
                .collect(Collectors.toSet());
            Map<String, CmnApiConnectParamModel> bodyParamMap = cacpMdlList.stream()
                .filter(mdl -> mdl.getCacpKbn() == 2)
                .collect(Collectors.toMap(
                    mdl -> mdl.getCacpName(),
                    mdl -> mdl,
                    (mdl1, mdl2) -> mdl2
                ));
            for (RngRapParamModel bodyMdl : rapParamJsonMdl.getBodyParamList()) {
                CmnApiConnectParamModel cacpMdl = bodyParamMap.get(bodyMdl.getName());
                String paramType = "";
                if (cacpMdl != null) {
                    if (cacpMdl.getCacpListKbn() == 1) {
                        paramType = "list";
                    } else if (parentNameSet.contains(cacpMdl.getCacpName())) {
                        paramType = "model";
                    } else {
                        paramType = "parameter";
                    }
                    if (!Objects.equals(bodyMdl.getParamType(), paramType)) {
                        return true;
                    }
                }
                if (Objects.equals(bodyMdl.getParamType(), "model")) {
                    for (Entry<Integer, ChildParamInfoModel> entry
                        : bodyMdl.getChildParamInfoMap().entrySet()) {
                        ChildParamInfoModel childValueMdl = entry.getValue();
                        CmnApiConnectParamModel childMdl
                            = bodyParamMap.get(bodyMdl.getName() + "." + childValueMdl.getName());
                        String childParamType = "";
                        if (childMdl != null) {
                            if (childMdl.getCacpListKbn() == 1) {
                                childParamType = "list";
                            } else {
                                childParamType = "parameter";
                            }
                            if (!Objects.equals(childValueMdl.getParamType(), childParamType)) {
                                return true;
                            }
                            hissuParamList.removeIf(mdl ->
                                mdl.getCacpKbn() == GSConstCommon.PARAM_KBN_BODY
                                    && Objects.equals(mdl.getCacpName(), childMdl.getCacpName()));
                        }
                    }
                }
                hissuParamList.removeIf(mdl ->
                    mdl.getCacpKbn() == GSConstCommon.PARAM_KBN_BODY
                        && Objects.equals(mdl.getCacpName(), bodyMdl.getName()));
            }
            if (!hissuParamList.isEmpty()) {
                //必須パラメータがJSON内に記載されていない場合
                return true;
            }

            //パスパラメータ
            errors = validateParam(
                errors, rapParamJsonMdl.getPathParamList(),
                araMdl, templateJson, GSConstCommon.PARAM_KBN_PATH, con, reqMdl);
            if (!errors.isEmpty()) {
                return true;
            }

            //クエリパラメータ
            errors = validateParam(
                errors, rapParamJsonMdl.getQueryParamList(),
                araMdl, templateJson, GSConstCommon.PARAM_KBN_QUERY, con, reqMdl);
            if (!errors.isEmpty()) {
                return true;
            }
            //リクエストボディ
            errors = validateParam(
                errors, rapParamJsonMdl.getBodyParamList(),
                araMdl, templateJson, GSConstCommon.PARAM_KBN_BODY, con, reqMdl);

            if (!errors.isEmpty()) {
                return true;
            }
        }
        return false;
    }

    /**
     * <br>[機  能] 1つのパラメータに設定されたパラメータ情報の入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param rapParamMdl パラメータ情報
     * @param validateMdl パラメータ情報入力チェック用モデル
     * @param cacSid 連携APISID
     * @param con コネクション
     * @return エラー文字列リスト
     * @throws SQLException
     */
    private List<String> __validateRapParamModel(
        List<String> errors,
        RngRapParamModel rapParamMdl,
        RngValidateParamModel validateMdl,
        int cacSid,
        Connection con) throws SQLException {

        if (rapParamMdl.getParamType().equals(RngConst.API_PARAMTYPE_LIST)) {
            //配列パラメータ
            validateMdl.setUseKbn(rapParamMdl.getUseKbn());
            validateMdl.setArrayFlg(true);
            errors = __validateListParam(errors, validateMdl, rapParamMdl.getListParamInfoMap());

        } else if (rapParamMdl.getParamType().equals(RngConst.API_PARAMTYPE_MODEL)) {
            //モデルパラメータ

            //親パラメータ必須チェック
            boolean parentReqFlg = validateMdl.isRequiredFlg();
            if (parentReqFlg && rapParamMdl.getUseKbn() != RngConst.API_PARAM_USE) {
                errors.add(__createErrorMsgForUseKbn(validateMdl.getParamNameDsp()));
            }
            CmnApiConnectParamDao cacpDao = new CmnApiConnectParamDao(con);
            List<String> childParamNameList
                = rapParamMdl.getChildParamInfoMap().values()
                    .stream()
                    .map(c -> rapParamMdl.getName() + "." + c.getName())
                    .collect(Collectors.toList());
            List<CmnApiConnectParamModel> cacpMdlList
                = cacpDao.select(
                cacSid,
                GSConstCommon.PARAM_KBN_BODY,
                childParamNameList);

            for (Entry<Integer, ChildParamInfoModel> childParamInfo
                : rapParamMdl.getChildParamInfoMap().entrySet()) {

                CmnApiConnectParamModel cacpMdl
                    = cacpMdlList.stream()
                        .filter(c -> c.getCacpName().equals(
                            rapParamMdl.getName() + "." + childParamInfo.getValue().getName()))
                        .findFirst()
                        .orElse(null);

                //子パラメータの必須区分を判定
                boolean requiredFlg = false;
                if ((rapParamMdl.getUseKbn() == RngConst.API_PARAM_USE
                    || parentReqFlg)
                    && cacpMdl != null
                    && cacpMdl.getCacpRequiredKbn()
                        == GSConstCommon.PARAM_REQUIRE_ON) {
                    //親パラメータの使用区分が「使用する」 または 必須 かつ 子パラメータが必須の場合
                    requiredFlg = true;
                }

                if (childParamInfo.getValue().getParamType().equals(RngConst.API_PARAMTYPE_LIST)) {
                    //配列パラメータ
                    if (rapParamMdl.getUseKbn() == RngConst.API_PARAM_NOT_USE) {
                        //親パラメータが「使用しない」の場合、子パラメータも「使用しない」
                        validateMdl.setUseKbn(RngConst.API_PARAM_NOT_USE);
                    } else if (rapParamMdl.getUseKbn() == RngConst.API_PARAM_USE) {
                        validateMdl.setUseKbn(childParamInfo.getValue().getUseKbn());
                    }
                    validateMdl.setArrayFlg(true);
                    validateMdl.setRequiredFlg(requiredFlg);
                    if (cacpMdl != null) {
                        validateMdl.setParamNameDsp(cacpMdl.getCacpNameDisp());
                    }
                    errors = __validateListParam(
                        errors, validateMdl, childParamInfo.getValue().getListParamInfoMap());
                } else if (childParamInfo.getValue().getParamType().equals(
                    RngConst.API_PARAMTYPE_PARAMETER)) {
                    //通常パラメータ
                    if (rapParamMdl.getUseKbn() == RngConst.API_PARAM_NOT_USE) {
                        //親パラメータが「使用しない」の場合、子パラメータも「使用しない」
                        validateMdl.setUseKbn(RngConst.API_PARAM_NOT_USE);
                    } else if (rapParamMdl.getUseKbn() == RngConst.API_PARAM_USE) {
                        validateMdl.setUseKbn(childParamInfo.getValue().getUseKbn());
                    }
                    validateMdl.setArrayFlg(false);
                    validateMdl.setRequiredFlg(requiredFlg);
                    if (cacpMdl != null) {
                        validateMdl.setParamNameDsp(cacpMdl.getCacpNameDisp());
                    }
                    errors = __validateNormalParam(
                        errors, validateMdl, childParamInfo.getValue().getParamInfoMap());
                }
            }

        } else if (rapParamMdl.getParamType().equals(RngConst.API_PARAMTYPE_PARAMETER)) {
            //通常パラメータ
            validateMdl.setUseKbn(rapParamMdl.getUseKbn());
            validateMdl.setArrayFlg(false);
            errors = __validateNormalParam(errors, validateMdl, rapParamMdl.getParamInfoMap());
        }
        return errors;
    }

    /**
     * <br>[機  能] 配列パラメータの入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param validateMdl パラメータ情報入力チェック用モデル
     * @param listParamMap 配列パラメータ情報
     * @return エラー文字列リスト
     */
    private List<String> __validateListParam(
        List<String> errors,
        RngValidateParamModel validateMdl,
        TreeMap<Integer, ListParamInfoModel> listParamMap) {

        //使用区分チェック
        errors = __validateParamUseKbn(
            errors,
            validateMdl.getUseKbn(),
            validateMdl.getParamNameDsp(),
            validateMdl.isRequiredFlg());
        //必須チェック
        boolean emptyArrayFlg = true;
        int arrayIndex = 1;
        for (Entry<Integer, ListParamInfoModel> listParamInfo : listParamMap.entrySet()) {

            List<ParamInfoModel> paramInfoList
                = listParamInfo.getValue().getParamInfoMap().values()
                    .stream().collect(Collectors.toList());
            //必須チェック
            List<ParamInfoModel> notDelParamList
                = paramInfoList.stream()
                    .filter(p -> !p.isErrorFlg())
                    .collect(Collectors.toList());
            if (validateMdl.isRequiredFlg() && notDelParamList.size() > 0) {
                //1つの配列要素に設定値が1つ以上設定されている場合
                emptyArrayFlg = false;
            }
            //パラメータ情報チェック
            if (!validateMdl.isRequiredFlg()
                && validateMdl.getUseKbn() == RngConst.API_PARAM_NOT_USE) {
                int paramType = GSConstCommon.PARAM_TYPE_TEXT;
                if (validateMdl.isFileFlg()) {
                    paramType = GSConstCommon.PARAM_TYPE_FILE;
                }
                errors.addAll(
                    checkParamType(paramType, paramInfoList,
                        validateMdl.getTemplateJson(),
                        validateMdl.getParamNameDsp(), arrayIndex, RngConst.API_PARAM_NOT_USE));
            } else if (validateMdl.isRequiredFlg()
                || validateMdl.getUseKbn() == RngConst.API_PARAM_USE) {
                errors = __validateParamInfo(errors, paramInfoList, validateMdl, arrayIndex);
            }
            arrayIndex++;
        }
        //必須チェック
        if (validateMdl.isRequiredFlg() && emptyArrayFlg) {
            errors.add(__createErrorMsgForRequired(validateMdl.getParamNameDsp()));
        }

        return errors;
    }

    /**
     * <br>[機  能] 通常パラメータの入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param validateMdl パラメータ情報入力チェック用モデル
     * @param paramMap パラメータ情報
     * @return エラー文字列リスト
     */
    private List<String> __validateNormalParam(
        List<String> errors,
        RngValidateParamModel validateMdl,
        TreeMap<Integer, ParamInfoModel> paramMap) {

        List<ParamInfoModel> paramInfoList
            = paramMap.values().stream().collect(Collectors.toList());
        //使用区分チェック
        errors = __validateParamUseKbn(
            errors,
            validateMdl.getUseKbn(),
            validateMdl.getParamNameDsp(),
            validateMdl.isRequiredFlg());
        //必須チェック
        List<ParamInfoModel> notDelParamList
            = paramInfoList.stream()
                .filter(p -> !p.isErrorFlg())
                .collect(Collectors.toList());
        if (validateMdl.isRequiredFlg() && notDelParamList.size() <= 0) {
            errors.add(__createErrorMsgForRequired(validateMdl.getParamNameDsp()));
        }
        if (!validateMdl.isRequiredFlg() && validateMdl.getUseKbn() == RngConst.API_PARAM_NOT_USE) {
            int paramType = GSConstCommon.PARAM_TYPE_TEXT;
            if (validateMdl.isFileFlg()) {
                paramType = GSConstCommon.PARAM_TYPE_FILE;
            }
            errors.addAll(
                checkParamType(paramType, paramInfoList,
                    validateMdl.getTemplateJson(),
                    validateMdl.getParamNameDsp(), -1, RngConst.API_PARAM_NOT_USE));
        } else if (validateMdl.isRequiredFlg()
            || validateMdl.getUseKbn() == RngConst.API_PARAM_USE) {
            //パラメータ情報チェック
            errors = __validateParamInfo(errors, paramInfoList, validateMdl, -1);
        }

        return errors;
    }

    /**
     * <br>[機  能] パラメータの使用区分の入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param useKbn 使用区分
     * @param paramName パラメータ名
     * @param requiredFlg 必須フラグ（true;必須, false:必須ではない）
     * @return エラー文字列リスト
     */
    private List<String> __validateParamUseKbn(
        List<String> errors,
        int useKbn,
        String paramName,
        boolean requiredFlg) {

        //必須の場合
        if (requiredFlg
            && useKbn != RngConst.API_PARAM_USE) {
            errors.add(__createErrorMsgForUseKbn(paramName));
        }
        //必須ではない場合
        if (!requiredFlg
            && useKbn != RngConst.API_PARAM_USE
            && useKbn != RngConst.API_PARAM_NOT_USE) {
            errors.add(__createErrorMsgForUseKbn(paramName));
        }
        return errors;
    }

    /**
     * <br>[機  能] パラメータ情報の入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramInfoList パラメータ情報リスト
     * @param validateMdl 入力チェック用モデル
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private List<String> __validateParamInfo(
        List<String> errors,
        List<ParamInfoModel> paramInfoList,
        RngValidateParamModel validateMdl,
        int arrayIndex) {

        String paramName = validateMdl.getParamNameDsp();
        boolean fileFlg = validateMdl.isFileFlg();
        boolean arrayFlg = validateMdl.isArrayFlg();
        String repeatFormId = validateMdl.getRepeatFormId();
        String templateJson = validateMdl.getTemplateJson();

        //1つの配列要素に複数パラメータが設定されている場合
        boolean multiListFlg = false;
        if (arrayFlg && paramInfoList.size() > 1) {
            multiListFlg = true;
        }

        //ファイル形式パラメータに複数パラメータが設定されている場合
        if (fileFlg && paramInfoList.size() > 1) {
            errors.add(__createErrorMsgForMultiParam(paramName, arrayIndex));
        }

        int row = 1;
        for (ParamInfoModel paramInfo : paramInfoList) {

            //フォーム要素が変更された場合
            if (paramInfo.isErrorFlg()) {
                errors.add(__createErrorMsgForParam(paramName, row, arrayIndex));
            }

            int paramKbn = paramInfo.getParamKbn();
            String formId = paramInfo.getParamFormId();
            int paramValue = paramInfo.getParamValue();
            String paramValueManual = paramInfo.getParamValueManual();
            int paramIndex = paramInfo.getParamIndex();

            if (paramKbn != RngConst.API_PARAMKBN_FORM) {
                //フォーム要素以外

                //要素区分範囲チェック
                errors = __checkParamKbnRange(
                    errors, paramKbn, paramName, row, fileFlg, arrayIndex);

                //フォームIDチェック
                if (formId != null && formId.length() > 0) {
                    //フォームIDが入力されている場合
                    errors.add(
                        __createErrorMsgForParam(paramName, row, arrayIndex));
                }

                //要素詳細範囲チェック
                if (paramKbn == RngConst.API_PARAMKBN_ADDUSER
                    || paramKbn == RngConst.API_PARAMKBN_LETUSER) {
                    //申請者、最終承認者
                    errors = __checkParamValueRangeForUser(
                        errors, paramValue, paramName, row, arrayIndex);
                } else if (paramKbn == RngConst.API_PARAMKBN_ADDDATE
                    || paramKbn == RngConst.API_PARAMKBN_LETDATE) {
                    //申請日時、最終承認日時
                    errors = __checkParamValueRangeForDate(
                        errors, paramValue, paramName, row, false, arrayIndex);
                } else if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO) {
                    //稟議情報
                    boolean repeatFlg = false;
                    if (validateMdl.getRepeatKbn() == RngConst.API_REPEAT_KBN_ON
                        && validateMdl.getRepeatType() == RngConst.API_REPEAT_TYPE_FILE) {
                        repeatFlg = true;
                    }
                    errors = __checkParamValueRangeForRingiInfo(
                        errors, paramValue, paramName, row, fileFlg, arrayIndex, repeatFlg);
                } else if (paramKbn == RngConst.API_PARAMKBN_MANUAL) {
                    //手入力
                    if (paramValue != RngConst.API_PARAMVALUE_NOSELECT) {
                        //要素詳細に0以外が指定されている場合
                        errors.add(
                            __createErrorMsgForParam(paramName, row, arrayIndex));
                    }
                    //手入力値チェック
                    errors = __checkParamValueManual(
                        errors, paramValueManual, paramName, row, arrayIndex);
                }

                //手入力値チェック
                if (paramKbn != RngConst.API_PARAMKBN_MANUAL
                    && paramValueManual != null && paramValueManual.length() > 0) {
                    //手入力以外選択時、手入力値が入力されている場合
                    errors.add(
                        __createErrorMsgForParam(paramName, row, arrayIndex));
                }

                //使用するボディ行範囲チェック
                if (paramIndex != -1) {
                    //値が設定されている場合
                    errors.add(
                        __createErrorMsgForParam(paramName, row, arrayIndex));
                }

            } else if (paramKbn == RngConst.API_PARAMKBN_FORM) {
                //フォーム要素

                //フォームIDチェック
                if (formId == null || formId.length() < 1) {
                    //未入力チェック
                    errors.add(
                        __createErrorMsgForParam(paramName, row, arrayIndex));
                } else {
                    //存在チェック
                    FormBuilder formBuilder = new FormBuilder();
                    formBuilder.setFormTable(templateJson);
                    List<FormCell> formList = formBuilder.getFormCellList();
                    boolean existFlg = false;
                    for (FormCell cell : formList) {
                        if (formId.equals(cell.getFormID())) {
                            existFlg = true;
                            break;
                        }
                    }
                    if (!existFlg) {
                        //フォームIDが存在しない
                        errors.add(
                            __createErrorMsgForParam(paramName, row, arrayIndex));
                        continue;
                    }
                    //フォームIDが存在する
                    for (FormCell cell : formList) {
                        if (!cell.getFormID().equals(formId)) {
                            continue;
                        }

                        //要素区分範囲チェック
                        if ((fileFlg && cell.getType() != EnumFormModelKbn.file)
                            || (!fileFlg && cell.getType() == EnumFormModelKbn.file)) {
                            errors.add(
                                __createErrorMsgForParam(paramName, row, arrayIndex));
                        }

                        //1つの配列要素に複数パラメータが設定されている場合
                        if (multiListFlg) {
                            if (cell.getType() == EnumFormModelKbn.user
                                && ((SimpleUserSelect) cell.getBody()).getMultiFlg()
                                    == UserGroupSelectModel.FLG_MULTI_ON) {
                                //ユーザ選択（複数）
                                errors.add(
                                    __createErrorMsgForMultiParam(
                                        paramName, arrayIndex));
                            } else if (cell.getType() == EnumFormModelKbn.group
                                && ((GroupComboModel) cell.getBody()).getMultiFlg()
                                == UserGroupSelectModel.FLG_MULTI_ON) {
                                //グループ選択（複数）
                                errors.add(
                                    __createErrorMsgForMultiParam(
                                        paramName, arrayIndex));
                            } else if (cell.getType() == EnumFormModelKbn.check) {
                                //チェックボックス
                                errors.add(
                                    __createErrorMsgForMultiParam(
                                        paramName, arrayIndex));
                            }
                        }

                        //要素詳細範囲チェック
                        if (cell.getType() == EnumFormModelKbn.user) {
                            //ユーザ選択
                            errors = __checkParamValueRangeForUser(
                                errors, paramValue, paramName, row, arrayIndex);
                        } else if (cell.getType() == EnumFormModelKbn.group) {
                            //グループ選択
                            errors = __checkParamValueRangeForGroup(
                                errors, paramValue, paramName, row, arrayIndex);
                        } else if (cell.getType() == EnumFormModelKbn.date) {
                            //日付選択
                            errors = __checkParamValueRangeForDate(
                                errors, paramValue, paramName, row, false, arrayIndex);
                        } else if (cell.getType() == EnumFormModelKbn.time) {
                            //時間選択
                            errors = __checkParamValueRangeForDate(
                                errors, paramValue, paramName, row, true, arrayIndex);
                        } else if (paramValue != RngConst.API_PARAMVALUE_NOSELECT) {
                            //その他のフォーム要素選択時、要素詳細が未選択以外の場合
                            errors.add(
                                __createErrorMsgForParam(paramName, row, arrayIndex));
                        }

                        //手入力値チェック
                        if (paramValueManual != null && paramValueManual.length() > 0) {
                            //手入力値が入力されている場合
                            errors.add(
                                __createErrorMsgForParam(paramName, row, arrayIndex));
                        }

                        //使用するボディ行範囲チェック
                        //表要素ボディ内要素を取得
                        RngTemplateBiz rtpBiz = new RngTemplateBiz();
                        List<String> bodyFormIdList
                            = rtpBiz.getFormCellFromBlockListBody(templateJson)
                                .stream()
                                .map(b -> b.getFormID())
                                .collect(Collectors.toList());

                        //繰り返し実行対象要素を取得
                        List<String> repeatFormIdList = new ArrayList<String>();
                        if (repeatFormId != null) {
                            //繰り返し実行対象表要素内のフォームID
                            repeatFormIdList
                                = rtpBiz.getFormCellFromBlockListBody(
                                    repeatFormId,
                                    templateJson)
                                    .stream()
                                    .map(b -> b.getFormID())
                                    .collect(Collectors.toList());
                        }

                        if (cell.getType() == EnumFormModelKbn.file
                            && formId.equals(repeatFormId)) {
                            //要素が繰り返し実行対象の場合（添付ファイルのみ対象）
                            if (paramIndex != -1) {
                                errors.add(
                                    __createErrorMsgForParam(paramName, row, arrayIndex));
                            }
                        } else if (repeatFormIdList.contains(formId)) {
                            //要素が繰り返し実行対象表要素内の場合
                            if (paramIndex != -1) {
                                errors.add(
                                    __createErrorMsgForParam(paramName, row, arrayIndex));
                            }
                        } else if (arrayFlg && bodyFormIdList.contains(formId)) {
                            //要素が表要素内 かつ 配列パラメータの場合
                            if (paramIndex != RngConst.PARAMETER_INDEX_ALL
                                && paramIndex != RngConst.PARAMETER_INDEX_TOP
                                && paramIndex != RngConst.PARAMETER_INDEX_BOTTOM) {
                                errors.add(
                                    __createErrorMsgForUseLine(paramName, row, arrayIndex));
                            }
                        } else if (!arrayFlg && bodyFormIdList.contains(formId)) {
                            //要素が表要素内 かつ 通常パラメータの場合
                            if (paramIndex != RngConst.PARAMETER_INDEX_TOP
                                && paramIndex != RngConst.PARAMETER_INDEX_BOTTOM) {
                                errors.add(
                                    __createErrorMsgForUseLine(paramName, row, arrayIndex));
                            }
                        } else {
                            if (paramIndex != -1) {
                                errors.add(
                                    __createErrorMsgForParam(paramName, row, arrayIndex));
                            }
                        }
                        break;
                    }
                }
            }
            row++;
        }
        return errors;
    }

    /**
     * <br>[機  能] 要素区分の範囲チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramKbn 要素区分選択値
     * @param cacpName パラメータ名
     * @param row 行数
     * @param fileFlg ファイル形式フラグ（true:ファイル形式, false:テキスト形式）
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private List<String> __checkParamKbnRange(
        List<String> errors,
        int paramKbn,
        String cacpName,
        int row,
        boolean fileFlg,
        int arrayIndex) {

        if (!fileFlg
            && paramKbn != RngConst.API_PARAMKBN_ADDUSER
            && paramKbn != RngConst.API_PARAMKBN_ADDDATE
            && paramKbn != RngConst.API_PARAMKBN_LETUSER
            && paramKbn != RngConst.API_PARAMKBN_LETDATE
            && paramKbn != RngConst.API_PARAMKBN_RINGIINFO
            && paramKbn != RngConst.API_PARAMKBN_MANUAL) {

            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }

        if (fileFlg && paramKbn != RngConst.API_PARAMKBN_RINGIINFO) {

            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }
        return errors;
    }

    /**
     * <br>[機  能] ユーザ選択時の要素詳細の範囲チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramValue 要素詳細選択値
     * @param cacpName パラメータ名
     * @param row 行数
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private List<String> __checkParamValueRangeForUser(
        List<String> errors,
        int paramValue,
        String cacpName,
        int row,
        int arrayIndex) {

        if (paramValue != RngConst.API_PARAMVALUE_USER_SID
            && paramValue != RngConst.API_PARAMVALUE_USER_ID
            && paramValue != RngConst.API_PARAMVALUE_USER_SEI
            && paramValue != RngConst.API_PARAMVALUE_USER_MEI
            && paramValue != RngConst.API_PARAMVALUE_USER_SEIMEI
            && paramValue != RngConst.API_PARAMVALUE_USER_SEI_KANA
            && paramValue != RngConst.API_PARAMVALUE_USER_MEI_KANA
            && paramValue != RngConst.API_PARAMVALUE_USER_SEIMEI_KANA
            && paramValue != RngConst.API_PARAMVALUE_SYAIN_NO
            && paramValue != RngConst.API_PARAMVALUE_BELONG_GROUP_SID
            && paramValue != RngConst.API_PARAMVALUE_BELONG_GROUP_ID
            && paramValue != RngConst.API_PARAMVALUE_BELONG_GROUP_NAME
            && paramValue != RngConst.API_PARAMVALUE_BELONG_GROUP_NAME_KANA
            && paramValue != RngConst.API_PARAMVALUE_MAILADDRESS1
            && paramValue != RngConst.API_PARAMVALUE_MAILADDRESS2
            && paramValue != RngConst.API_PARAMVALUE_MAILADDRESS3) {

            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }
        return errors;
    }

    /**
     * <br>[機  能] ユーザ選択時の要素詳細の範囲チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramValue 要素詳細選択値
     * @param cacpName パラメータ名
     * @param row 行数
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private List<String> __checkParamValueRangeForGroup(
        List<String> errors,
        int paramValue,
        String cacpName,
        int row,
        int arrayIndex) {

        if (paramValue != RngConst.API_PARAMVALUE_GROUP_SID
            && paramValue != RngConst.API_PARAMVALUE_GROUP_ID
            && paramValue != RngConst.API_PARAMVALUE_GROUP_NAME
            && paramValue != RngConst.API_PARAMVALUE_GROUP_NAME_KANA) {

            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }
        return errors;
    }

    /**
     * <br>[機  能] 日時選択時の要素詳細の範囲チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramValue 要素詳細選択値
     * @param cacpName パラメータ名
     * @param row 行数
     * @param timeFlg フォーム要素時間入力フラグ（true:時間選択, false:それ以外）
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private List<String> __checkParamValueRangeForDate(
        List<String> errors,
        int paramValue,
        String cacpName,
        int row,
        boolean timeFlg,
        int arrayIndex) {

        if (timeFlg
            && paramValue != RngConst.API_PARAMVALUE_FULLTIME_COLON
            && paramValue != RngConst.API_PARAMVALUE_FULLTIME_TEXT
            && paramValue != RngConst.API_PARAMVALUE_HHMM_COLON
            && paramValue != RngConst.API_PARAMVALUE_HHMM_TEXT
            && paramValue != RngConst.API_PARAMVALUE_HOUR
            && paramValue != RngConst.API_PARAMVALUE_MINUTE
            && paramValue != RngConst.API_PARAMVALUE_SECOND) {

            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        } else if (!timeFlg
            && paramValue != RngConst.API_PARAMVALUE_FULLDATE_SLASH
            && paramValue != RngConst.API_PARAMVALUE_FULLDATE_HYPHEN
            && paramValue != RngConst.API_PARAMVALUE_FULLDATE_TEXT
            && paramValue != RngConst.API_PARAMVALUE_DATE_SLASH
            && paramValue != RngConst.API_PARAMVALUE_DATE_HEPHEN
            && paramValue != RngConst.API_PARAMVALUE_DATE_TEXT
            && paramValue != RngConst.API_PARAMVALUE_FULLTIME_COLON
            && paramValue != RngConst.API_PARAMVALUE_FULLTIME_TEXT
            && paramValue != RngConst.API_PARAMVALUE_HHMM_COLON
            && paramValue != RngConst.API_PARAMVALUE_HHMM_TEXT
            && paramValue != RngConst.API_PARAMVALUE_YEAR
            && paramValue != RngConst.API_PARAMVALUE_MONTH
            && paramValue != RngConst.API_PARAMVALUE_DAY
            && paramValue != RngConst.API_PARAMVALUE_HOUR
            && paramValue != RngConst.API_PARAMVALUE_MINUTE
            && paramValue != RngConst.API_PARAMVALUE_SECOND) {

            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }
        return errors;
    }

    /**
     * <br>[機  能] 稟議情報の要素詳細の範囲チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramValue 要素詳細選択値
     * @param cacpName パラメータ名
     * @param row 行数
     * @param fileFlg ファイル形式フラグ（true:ファイル形式, false:テキスト形式）
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @param repeatFlg 確認時添付ファイル繰り返し実行対象フラグ（true:対象, false:対象ではない）
     * @return エラー文字列リスト
     */
    private List<String> __checkParamValueRangeForRingiInfo(
        List<String> errors,
        int paramValue,
        String cacpName,
        int row,
        boolean fileFlg,
        int arrayIndex,
        boolean repeatFlg) {

        //ファイル形式 かつ 配列要素の場合
        if (fileFlg && arrayIndex > 0
            && paramValue != RngConst.API_PARAMVALUE_FILE) {
            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }

        //ファイル形式 かつ 通常パラメータ かつ 確認時添付ファイルが繰り返し実行対象ではない場合
        if (fileFlg && arrayIndex == -1 && !repeatFlg
            && paramValue != RngConst.API_PARAMVALUE_FILE_FIRST
            && paramValue != RngConst.API_PARAMVALUE_FILE_LAST) {
            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }

        //ファイル形式 かつ 通常パラメータ かつ 確認時添付ファイルが繰り返し実行対象の場合
        if (fileFlg && arrayIndex == -1 && repeatFlg
            && paramValue != RngConst.API_PARAMVALUE_FILE) {
            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }

        //テキスト形式の場合
        if (!fileFlg
            && paramValue != RngConst.API_PARAMVALUE_TITLE
            && paramValue != RngConst.API_PARAMVALUE_SINSEIID) {
            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }
        return errors;
    }

    /**
     * <br>[機  能] 手入力値の入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param paramValueManual 要素詳細手入力値
     * @param cacpName パラメータ名
     * @param row 行数
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private List<String> __checkParamValueManual(
        List<String> errors,
        String paramValueManual,
        String cacpName,
        int row,
        int arrayIndex) {

        //未入力チェック
        if (paramValueManual == null || paramValueManual.length() <= 0) {
            errors.add(__createErrorMsgForParam(cacpName, row, arrayIndex));
        }
        //最大文字数チェック
        if (paramValueManual != null
            && paramValueManual.length() > RngConst.API_PARAM_MANUAL_MAXLENGTH) {
            String msgText = cacpName + "&nbsp;(";
            if (arrayIndex > 0) {
                msgText += gsMsg__.getMessage("rng.rng320.7")
                    + String.valueOf(arrayIndex) + "&nbsp;";
            }
            msgText += gsMsg__.getMessage("cmn.line", new String[] {String.valueOf(row)});
            msgText += ")&nbsp;:&nbsp;";
            msgText += gsMsg__.getMessage("cmn.plz.max",
                new String[] {gsMsg__.getMessage("rng.rng090.19"),
                String.valueOf(RngConst.API_PARAM_MANUAL_MAXLENGTH)});
            errors.add(msgText);
        }
        return errors;
    }

    /**
     * <br>[機  能] 必須パラメータのエラー文字列を作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param cacpName パラメータ名
     * @return エラー文字列リスト
     */
    private String __createErrorMsgForRequired(
        String cacpName) {

        String msgText = cacpName + "&nbsp;";
        msgText += gsMsg__.getMessage("cmn.select.4",
            new String[] {gsMsg__.getMessage("rng.rng090.04")});
        return msgText;
    }

    /**
     * <br>[機  能] 使用区分のエラー文字列を作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param cacpName パラメータ名
     * @return エラー文字列リスト
     */
    private String __createErrorMsgForUseKbn(
        String cacpName) {

        String msgText = cacpName + "&nbsp;:&nbsp;";
        msgText += gsMsg__.getMessage("cmn.select.4",
            new String[] {gsMsg__.getMessage("rng.rng320.13")});
        return msgText;
    }

    /**
     * <br>[機  能] 要素区分のエラー文字列を作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param cacpName パラメータ名
     * @param row 行数
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private String __createErrorMsgForParam(
        String cacpName,
        int row,
        int arrayIndex) {

        String msgText = cacpName + "&nbsp;(";
        if (arrayIndex > 0) {
            msgText += gsMsg__.getMessage("rng.rng320.7") + String.valueOf(arrayIndex) + "&nbsp;";
        }
        msgText += gsMsg__.getMessage("cmn.line", new String[] {String.valueOf(row)});
        msgText += ")&nbsp;:&nbsp;";
        msgText += gsMsg__.getMessage("cmn.select.4",
            new String[] {gsMsg__.getMessage("rng.rng090.04")});
        return msgText;
    }

    /**
     * <br>[機  能] 使用するボディ行のエラー文字列を作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param cacpName パラメータ名
     * @param row 行数
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private String __createErrorMsgForUseLine(
        String cacpName,
        int row,
        int arrayIndex) {

        String msgText = cacpName + "&nbsp;(";
        if (arrayIndex > 0) {
            msgText += gsMsg__.getMessage("rng.rng320.7") + String.valueOf(arrayIndex) + "&nbsp;";
        }
        msgText += gsMsg__.getMessage("cmn.line", new String[] {String.valueOf(row)});
        msgText += ")&nbsp;:&nbsp;";
        msgText += gsMsg__.getMessage("cmn.select.4",
            new String[] {gsMsg__.getMessage("rng.rng320.12")});
        return msgText;
    }

    /**
     * <br>[機  能] 複数要素選択不可の場合のエラー文字列を作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param cacpName パラメータ名
     * @param arrayIndex 配列要素インデックス（配列要素以外は-1）
     * @return エラー文字列リスト
     */
    private String __createErrorMsgForMultiParam(
        String cacpName,
        int arrayIndex) {

        String msgText = cacpName;
        if (arrayIndex > 0) {
            msgText += "&nbsp;("
                + gsMsg__.getMessage("rng.rng320.7")
                + String.valueOf(arrayIndex) + ")";
        }
        msgText += "&nbsp;:&nbsp;";
        msgText += gsMsg__.getMessage("rng.rng320.26");
        return msgText;
    }

    /**
     * <br>[機  能] アクションパラメータ（実行条件）の入力チェックを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param errors エラー文字列リスト
     * @param reqMdl リクエストモデル
     * @param con コネクション
     * @param rapConditionJsonMdl 実行条件JSONモデル
     * @param templateJson 申請内容JSON文字列
     * @return エラー文字列リスト
     * @throws SQLException
     */
    public List<String> validateCondition(
        List<String> errors,
        RngRapConditionJsonModel rapConditionJsonMdl,
        String templateJson,
        Connection con,
        RequestModel reqMdl) throws SQLException {

        //実行条件区分
        if (rapConditionJsonMdl.getConditionType() != RngConst.EXECUTE_CONDITION_AND
            && rapConditionJsonMdl.getConditionType() != RngConst.EXECUTE_CONDITION_OR) {
            //範囲チェック
            errors.add(gsMsg__.getMessage("cmn.select.4",
                new String[] {gsMsg__.getMessage("rng.rng310.50")}));
        }

        int row = 1;
        for (RngRapConditionModel conditionMdl : rapConditionJsonMdl.getConditionList()) {

            //フォーム要素が変更された場合
            if (conditionMdl.isErrorFlg()) {
                errors.add(__createErrorMsgForConditionCondition(row));
            }

            int paramKbn = conditionMdl.getParamKbn();
            String formId = conditionMdl.getParamFormId();
            int paramValue = conditionMdl.getParamValue();
            String compareTarget = conditionMdl.getCompareTarget();
            int compareType = conditionMdl.getCompareType();

            if (paramKbn != RngConst.API_PARAMKBN_FORM) {
                //フォーム要素以外

                //条件対象コンボチェック
                if (paramKbn != RngConst.API_PARAMKBN_ADDUSER
                    && paramKbn != RngConst.API_PARAMKBN_ADDDATE
                    && paramKbn != RngConst.API_PARAMKBN_LETUSER
                    && paramKbn != RngConst.API_PARAMKBN_LETDATE
                    && paramKbn != RngConst.API_PARAMKBN_RINGIINFO) {
                    //範囲チェック
                    errors.add(__createErrorMsgForConditionCondition(row));
                }

                //設定値詳細チェック
                if ((paramKbn == RngConst.API_PARAMKBN_ADDUSER
                    || paramKbn == RngConst.API_PARAMKBN_LETUSER)
                    && paramValue != RngConst.API_COMPARE_PARAM_NOSELECT
                    && paramValue != RngConst.API_COMPARE_PARAM_POSITION) {
                    //申請者、最終承認者
                    errors.add(__createErrorMsgForConditionCondition(row));
                } else if ((paramKbn == RngConst.API_PARAMKBN_ADDDATE
                    || paramKbn == RngConst.API_PARAMKBN_LETDATE)
                    && paramValue != RngConst.API_COMPARE_PARAM_NOSELECT) {
                    //申請日時、最終承認日時
                    errors.add(__createErrorMsgForConditionCondition(row));
                } else if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO
                    && paramValue != RngConst.API_COMPARE_PARAM_TITLE
                    && paramValue != RngConst.API_COMPARE_PARAM_SINSEIID
                    && paramValue != RngConst.API_COMPARE_PARAM_FILE) {
                    //稟議情報
                    errors.add(__createErrorMsgForConditionCondition(row));
                }

                //比較文字列
                if (compareTarget == null || compareTarget.length() < 1) {
                    //未入力チェック
                    if (paramKbn == RngConst.API_PARAMKBN_ADDUSER
                        || paramKbn == RngConst.API_PARAMKBN_LETUSER) {
                        //申請者、最終承認者
                        errors.add(__createErrorMsgForConditionCondition(row));
                    } else if (paramKbn == RngConst.API_PARAMKBN_ADDDATE
                        || paramKbn == RngConst.API_PARAMKBN_LETDATE
                        || (paramKbn == RngConst.API_PARAMKBN_RINGIINFO
                            && (paramValue == RngConst.API_COMPARE_PARAM_TITLE
                                || paramValue == RngConst.API_COMPARE_PARAM_SINSEIID))) {
                        //申請日時、最終承認日時、稟議情報（タイトル、申請ID）
                        errors.add(__createErrorMsgForConditionCondition(row));
                    }
                } else {
                    if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO
                        && (paramValue == RngConst.API_COMPARE_PARAM_TITLE
                            || paramValue == RngConst.API_COMPARE_PARAM_SINSEIID)) {
                        //稟議情報（タイトル、申請ID）
                        validateCompareTargetForm(errors, compareTarget, row, reqMdl, false);
                    } else if (paramKbn == RngConst.API_PARAMKBN_ADDDATE
                        || paramKbn == RngConst.API_PARAMKBN_LETDATE) {
                        //申請日時、最終承認日時
                        if (!ValidateUtil.isSlashDateTimeFormat(compareTarget)) {
                            //日付フォーマットチェック
                            errors.add(__createErrorMsgForConditionCondition(row));
                        }

                    } else if ((paramKbn == RngConst.API_PARAMKBN_ADDUSER
                        || paramKbn == RngConst.API_PARAMKBN_LETUSER)
                        && paramValue == RngConst.API_COMPARE_PARAM_NOSELECT) {
                        //申請者（未選択）、最終承認者（未選択）
                        int userSid = NullDefault.getInt(compareTarget, -1);
                        CmnUsrmDao cumDao = new CmnUsrmDao(con);
                        if (userSid < GSConstUser.USER_RESERV_SID
                            || (userSid > GSConstUser.USER_RESERV_SID
                                && cumDao.getUserJkbn(userSid)
                                != GSConst.JTKBN_TOROKU)) {
                            //範囲チェック
                            errors.add(__createErrorMsgForConditionCondition(row));
                        }
                    } else if ((paramKbn == RngConst.API_PARAMKBN_ADDUSER
                        || paramKbn == RngConst.API_PARAMKBN_LETUSER)
                        && paramValue == RngConst.API_COMPARE_PARAM_POSITION) {
                        //申請者（役職）、最終承認者（役職）
                        int posSid = NullDefault.getInt(compareTarget, -1);
                        CmnPositionDao cpsDao = new CmnPositionDao(con);
                        CmnPositionModel cpsMdl = cpsDao.getPosInfo(posSid);
                        if (cpsMdl == null) {
                            //範囲チェック
                            errors.add(__createErrorMsgForConditionCondition(row));
                        }
                    } else if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO
                        && paramValue == RngConst.API_COMPARE_PARAM_FILE) {
                        //稟議情報（確認時添付ファイル）
                        errors.add(__createErrorMsgForConditionCondition(row));
                    }
                }

                //比較条件
                if (paramKbn == RngConst.API_PARAMKBN_ADDUSER
                    || paramKbn == RngConst.API_PARAMKBN_LETUSER) {
                    //申請者、最終承認者
                    if (compareType != RngConst.API_COMPARE_CONDITION_EQUAL
                        && compareType != RngConst.API_COMPARE_CONDITION_NOTEQUAL) {
                        errors.add(__createErrorMsgForConditionCondition(row));
                    }

                } else if (paramKbn == RngConst.API_PARAMKBN_ADDDATE
                    || paramKbn == RngConst.API_PARAMKBN_LETDATE) {
                    //申請日時、最終承認日時
                    if (compareType != RngConst.API_COMPARE_CONDITION_MORETHAN
                        && compareType != RngConst.API_COMPARE_CONDITION_MORE
                        && compareType != RngConst.API_COMPARE_CONDITION_LESS
                        && compareType != RngConst.API_COMPARE_CONDITION_LESSTHAN) {
                        errors.add(__createErrorMsgForConditionCondition(row));
                    }
                } else if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO
                    && (paramValue == RngConst.API_COMPARE_PARAM_TITLE
                    || paramValue == RngConst.API_COMPARE_PARAM_SINSEIID)) {
                    //稟議情報（タイトル、申請ID）
                    if (compareType != RngConst.API_COMPARE_CONDITION_CONTAIN
                        && compareType != RngConst.API_COMPARE_CONDITION_NOTCONTAIN) {
                        errors.add(__createErrorMsgForConditionCondition(row));
                    }
                } else if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO
                    && paramValue == RngConst.API_COMPARE_PARAM_FILE) {
                    //稟議情報（確認時添付ファイル）
                    if (compareType != RngConst.API_COMPARE_CONDITION_EXISTFILE
                        && compareType != RngConst.API_COMPARE_CONDITION_NOTEXISTFILE) {
                        errors.add(__createErrorMsgForConditionCondition(row));
                    }
                }

            } else if (paramKbn == RngConst.API_PARAMKBN_FORM) {
                //フォーム要素

                if (formId == null || formId.length() < 1) {
                    //フォームID未入力チェック（条件対象コンボチェック）
                    errors.add(__createErrorMsgForConditionCondition(row));
                } else {
                    //フォームID存在チェック（条件対象コンボチェック）
                    FormBuilder formBuilder = new FormBuilder();
                    formBuilder.setFormTable(templateJson);
                    List<FormCell> formList = formBuilder.getFormCellList();
                    boolean existFlg = false;
                    for (FormCell cell : formList) {
                        if (formId.equals(cell.getFormID())) {
                            existFlg = true;
                            break;
                        }
                    }
                    if (!existFlg) {
                        //フォームIDが存在しない
                        errors.add(__createErrorMsgForConditionCondition(row));
                    } else {
                        //フォームIDが存在する
                        for (FormCell cell : formList) {
                            if (cell.getFormID().equals(formId)) {
                                //設定値詳細
                                if (cell.getType() == EnumFormModelKbn.user) {
                                    //ユーザ選択
                                    if (paramValue != RngConst.API_COMPARE_PARAM_NOSELECT
                                        && paramValue != RngConst.API_COMPARE_PARAM_POSITION) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                } else {
                                    //ユーザ選択以外
                                    if (paramValue != RngConst.API_COMPARE_PARAM_NOSELECT) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                }

                                //比較文字列
                                if (compareTarget == null || compareTarget.length() < 1) {
                                    //未入力チェック
                                    if (cell.getType() != EnumFormModelKbn.file) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                } else {
                                    if (cell.getType() == EnumFormModelKbn.label
                                        || cell.getType() == EnumFormModelKbn.textbox
                                        || cell.getType() == EnumFormModelKbn.textarea) {
                                        //コメント、テキスト入力、テキスト入力（複数行）
                                        validateCompareTargetForm(
                                            errors, compareTarget, row, reqMdl, false);
                                    } else if (cell.getType() == EnumFormModelKbn.number
                                        || cell.getType() == EnumFormModelKbn.sum
                                        || cell.getType() == EnumFormModelKbn.calc) {
                                        //数値入力、自動計算（合計）、自動計算（その他）
                                        validateCompareTargetForm(
                                            errors, compareTarget, row, reqMdl, true);
                                    } else if (cell.getType() == EnumFormModelKbn.radio) {
                                        //ラジオボタン
                                        List<String> selectValueList
                                            = ((RadioButton) cell.getBody()).getList();
                                        selectValueList.add(
                                            String.valueOf(RngConst.API_COMPARE_TARGET_NOSELECT));
                                        if (!selectValueList.contains(compareTarget)) {
                                            //範囲チェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.combo) {
                                        //コンボボックス
                                        List<String> selectValueList
                                            = ((ComboBox) cell.getBody()).getList();
                                        selectValueList.add(
                                            String.valueOf(RngConst.API_COMPARE_TARGET_NOSELECT));
                                        if (!selectValueList.contains(compareTarget)) {
                                            //範囲チェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.check) {
                                        //チェックボックス
                                        List<String> selectValueList
                                            = ((CheckBox) cell.getBody()).getList();
                                        selectValueList.add(
                                            String.valueOf(RngConst.API_COMPARE_TARGET_NOSELECT));
                                        if (!selectValueList.contains(compareTarget)) {
                                            //範囲チェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.date) {
                                        //日付入力
                                        if (!ValidateUtil.isSlashDateFormat(compareTarget)) {
                                            //日付フォーマットチェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.time) {
                                        //時間入力
                                        if (!ValidateUtil.isTimeFormat(compareTarget)) {
                                            //時間フォーマットチェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.user
                                        && paramValue == RngConst.API_COMPARE_PARAM_NOSELECT) {
                                        //ユーザ選択（未選択）
                                        int userSid = NullDefault.getInt(compareTarget, -1);
                                        CmnUsrmDao cumDao = new CmnUsrmDao(con);
                                        if (userSid < GSConstUser.USER_RESERV_SID
                                            || (userSid > GSConstUser.USER_RESERV_SID
                                                && cumDao.getUserJkbn(userSid)
                                                != GSConst.JTKBN_TOROKU)) {
                                            //範囲チェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.user
                                        && paramValue == RngConst.API_COMPARE_PARAM_POSITION) {
                                        //ユーザ選択（役職）
                                        int posSid = NullDefault.getInt(compareTarget, -1);
                                        CmnPositionDao cpsDao = new CmnPositionDao(con);
                                        CmnPositionModel cpsMdl = cpsDao.getPosInfo(posSid);
                                        if (cpsMdl == null) {
                                            //範囲チェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.group) {
                                        //グループ選択
                                        int grpSid = NullDefault.getInt(compareTarget, -1);
                                        GroupDao grpDao = new GroupDao(con);
                                        CmnGroupmModel grpMdl = grpDao.getGroup(grpSid);
                                        if (grpMdl == null) {
                                            //範囲チェック
                                            errors.add(__createErrorMsgForConditionCondition(row));
                                        }
                                    } else if (cell.getType() == EnumFormModelKbn.file) {
                                        //添付ファイル
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                }

                                //比較条件
                                if (cell.getType() == EnumFormModelKbn.radio
                                    || cell.getType() == EnumFormModelKbn.combo
                                    || (cell.getType() == EnumFormModelKbn.user
                                        && ((SimpleUserSelect) cell.getBody()).getMultiFlg()
                                        == UserGroupSelectModel.FLG_MULTI_OFF)
                                    || (cell.getType() == EnumFormModelKbn.group
                                        && ((GroupComboModel) cell.getBody()).getMultiFlg()
                                        == UserGroupSelectModel.FLG_MULTI_OFF)) {
                                    //ラジオボタン、コンボボックス
                                    //ユーザ選択（単一）、グループ選択（単一）
                                    if (compareType != RngConst.API_COMPARE_CONDITION_EQUAL
                                        && compareType != RngConst.API_COMPARE_CONDITION_NOTEQUAL) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                } else if (cell.getType() == EnumFormModelKbn.label
                                    || cell.getType() == EnumFormModelKbn.textbox
                                    || cell.getType() == EnumFormModelKbn.textarea
                                    || cell.getType() == EnumFormModelKbn.check
                                    || (cell.getType() == EnumFormModelKbn.user
                                        && ((SimpleUserSelect) cell.getBody()).getMultiFlg()
                                        == UserGroupSelectModel.FLG_MULTI_ON)
                                    || (cell.getType() == EnumFormModelKbn.group
                                        && ((GroupComboModel) cell.getBody()).getMultiFlg()
                                        == UserGroupSelectModel.FLG_MULTI_ON)) {
                                    //コメント、テキスト入力、テキスト入力（複数行）、チェックボックス
                                    //ユーザ選択（複数）、グループ選択（複数）
                                    if (compareType != RngConst.API_COMPARE_CONDITION_CONTAIN
                                        && compareType
                                            != RngConst.API_COMPARE_CONDITION_NOTCONTAIN) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                } else if (cell.getType() == EnumFormModelKbn.date
                                    || cell.getType() == EnumFormModelKbn.time
                                    || cell.getType() == EnumFormModelKbn.number
                                    || cell.getType() == EnumFormModelKbn.sum
                                    || cell.getType() == EnumFormModelKbn.calc) {
                                    //日付選択、時間選択、数値入力、自動計算（合計）、自動計算（その他）
                                    if (compareType != RngConst.API_COMPARE_CONDITION_EQUAL
                                        && compareType != RngConst.API_COMPARE_CONDITION_NOTEQUAL
                                        && compareType != RngConst.API_COMPARE_CONDITION_MORETHAN
                                        && compareType != RngConst.API_COMPARE_CONDITION_MORE
                                        && compareType != RngConst.API_COMPARE_CONDITION_LESS
                                        && compareType != RngConst.API_COMPARE_CONDITION_LESSTHAN) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                } else if (cell.getType() == EnumFormModelKbn.file) {
                                    //添付ファイル
                                    if (compareType != RngConst.API_COMPARE_CONDITION_EXISTFILE
                                        && compareType
                                            != RngConst.API_COMPARE_CONDITION_NOTEXISTFILE) {
                                        errors.add(__createErrorMsgForConditionCondition(row));
                                    }
                                }
                            }
                        }
                    }
                }
            }
            row++;
        }
        return errors;
    }

    /**
     * <br>[機  能] 実行条件のエラー文字列を作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param row 行数
     * @return エラー文字列リスト
     */
    private String __createErrorMsgForConditionCondition(int row) {

        String msgText =
            gsMsg__.getMessage("cmn.select.4",
                new String[] {gsMsg__.getMessage("rng.rng310.8")
                    + "&nbsp;(" + gsMsg__.getMessage("cmn.line",
                        new String[] {String.valueOf(row)}) + ")"});
        return msgText;
    }

    /**
     * <br>[機  能] パラメータ形式と要素の設定値が一致するかチェックする
     * <br>[解  説]
     * <br>[備  考]
     * @param paramType パラメータ形式（0:テキスト形式, 1:ファイル形式）
     * @param paramInfoMdlList パラメータ情報リスト
     * @param templateJson フォーム要素JSON
     * @param paramName パラメータ名
     * @param arrayIndex 配列要素インデックス
     * @param useKbn パラメータの使用区分
     * @return 判定結果（true:一致, false:一致しない）
     */
    public List<String> checkParamType(
        int paramType,
        List<ParamInfoModel> paramInfoMdlList,
        String templateJson,
        String paramName,
        int arrayIndex,
        int useKbn) {

        List<String> errors = new ArrayList<String>();
        boolean fileFlg = false;
        if (paramType == GSConstCommon.PARAM_TYPE_FILE) {
            fileFlg = true;
        }
        int row = 0;
        for (ParamInfoModel paramInfoMdl : paramInfoMdlList) {
            row++;
            int paramKbn = paramInfoMdl.getParamKbn();
            if (useKbn == RngConst.API_PARAM_NOT_USE && paramKbn == -1) {
                //パラメータ自体が未使用かつ、要素区分が"選択してください。"の場合はチェックしない
                continue;
            }
            if (paramKbn != RngConst.API_PARAMKBN_FORM) {
                //フォーム要素以外
                //要素区分チェック
                errors = __checkParamKbnRange(
                    errors, paramKbn, paramName, row, fileFlg, arrayIndex);

                //要素詳細チェック
                if (paramKbn == RngConst.API_PARAMKBN_RINGIINFO) {
                    int paramValue = paramInfoMdl.getParamValue();
                    if (useKbn == RngConst.API_PARAM_NOT_USE && paramValue == -1) {
                        continue;
                    }
                    if ((fileFlg
                        && paramValue != RngConst.API_PARAMVALUE_FILE
                        && paramValue != RngConst.API_PARAMVALUE_FILE_FIRST
                        && paramValue != RngConst.API_PARAMVALUE_FILE_LAST)
                        || (!fileFlg
                            && paramValue != RngConst.API_PARAMVALUE_TITLE
                            && paramValue != RngConst.API_PARAMVALUE_SINSEIID)) {
                        errors.add(__createErrorMsgForParam(paramName, row, arrayIndex));
                    }
                }
            } else if (paramKbn == RngConst.API_PARAMKBN_FORM) {
                //フォーム要素
                FormBuilder formBuilder = new FormBuilder();
                formBuilder.setFormTable(templateJson);
                List<FormCell> formList = formBuilder.getFormCellList();
                for (FormCell cell : formList) {
                    if (cell.getFormID().equals(paramInfoMdl.getParamFormId())) {
                        if ((fileFlg && cell.getType() != EnumFormModelKbn.file)
                            || (!fileFlg && cell.getType() == EnumFormModelKbn.file)) {
                            errors.add(__createErrorMsgForParam(paramName, row, arrayIndex));
                        }
                        break;
                    }
                }
            }
        }
        return errors;
    }
}