package jp.groupsession.v2.rng.biz;

import java.io.File;
import java.sql.Connection;
import java.sql.SQLException;
import java.text.CharacterIterator;
import java.text.StringCharacterIterator;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import jp.co.sjts.util.NullDefault;
import jp.co.sjts.util.io.IOToolsException;
import jp.co.sjts.util.io.ObjectFile;
import jp.co.sjts.util.json.JSONArray;
import jp.co.sjts.util.json.JSONObject;
import jp.groupsession.v2.cmn.GSConstCommon;
import jp.groupsession.v2.cmn.GSTemporaryPathUtil;
import jp.groupsession.v2.cmn.model.RequestModel;
import jp.groupsession.v2.rng.RngConst;
import jp.groupsession.v2.rng.RngValidate;
import jp.groupsession.v2.rng.dao.RngActionparamDao;
import jp.groupsession.v2.rng.dao.RngTemplateActionDao;
import jp.groupsession.v2.rng.dao.RngTemplateDao;
import jp.groupsession.v2.rng.model.AddRngActionModel;
import jp.groupsession.v2.rng.model.AddRngActionParamModel;
import jp.groupsession.v2.rng.model.ChildParamInfoModel;
import jp.groupsession.v2.rng.model.ListParamInfoModel;
import jp.groupsession.v2.rng.model.ParamInfoModel;
import jp.groupsession.v2.rng.model.RngActionparamModel;
import jp.groupsession.v2.rng.model.RngRapConditionJsonModel;
import jp.groupsession.v2.rng.model.RngRapConditionModel;
import jp.groupsession.v2.rng.model.RngRapParamJsonModel;
import jp.groupsession.v2.rng.model.RngRapParamModel;
import jp.groupsession.v2.rng.model.RngTemplateActionModel;
import jp.groupsession.v2.struts.msg.GsMessage;

/**
 * <br>[機  能] 稟議テンプレート作成時の連携API情報を取得，操作するためのビジネスロジック
 * <br>[解  説]
 * <br>[備  考] 共有テンプレート画面と関連する画面での使用を想定
 *
 * @author  JTS
 */
public class RngTemplateActionFileBiz {

    /** リクエストモデル */
    private static RequestModel reqMdl__;
    /** ディレクトリID ※本Classは共有テンプレート画面と関連する画面での使用を想定 */
    public static final String DIRID = "rng090";

    /**
     * <br>[機  能] コンストラクタ
     * <br>[解  説]
     * <br>[備  考]
     * @param reqMdl リクエストモデル
     */
    public RngTemplateActionFileBiz(RequestModel reqMdl) {
        reqMdl__ = reqMdl;
    }

    /**
     * <br>[機  能] 決裁後アクション用ファイルの保存パスを取得する
     * <br>[解  説]
     * <br>[備  考]
     * @return 保存パス
     */
    public String __getTempPath() {
        GSTemporaryPathUtil temp = GSTemporaryPathUtil.getInstance();
        String saveDir = temp.getTempPath(reqMdl__,
            RngConst.PLUGIN_ID_RINGI, DIRID,
            RngConst.RNG_FOLDER_ACTION);
        return saveDir;
    }

    /**
     * <br>[機  能] 指定されたテンプレートの決裁後アクションモデル一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param rtpSid テンプレートSID
     * @param rtpVer テンプレートバージョン
     * @param con コネクション
     * @return 決裁後アクションモデル一覧
     * @throws SQLException
     */
    public LinkedList<AddRngActionModel> getAddRngActionMdlList(
        int rtpSid, int rtpVer, Connection con) throws SQLException {

        LinkedList<AddRngActionModel> araModelList = new LinkedList<AddRngActionModel>();
        //決裁後アクション情報取得
        RngTemplateActionDao rtaDao = new RngTemplateActionDao(con);
        List<RngTemplateActionModel> rtaModelList = rtaDao.select(rtpSid, rtpVer);
        for (RngTemplateActionModel rtaModel : rtaModelList) {
            //決裁後アクション情報
            AddRngActionModel araModel = new AddRngActionModel();
            araModel.setIndex(rtaModel.getRtpaSort());
            araModel.setName(rtaModel.getRtpaName());
            araModel.setUseKbn(rtaModel.getRtpaUseKbn());
            araModel.setApiConnectSid(rtaModel.getCacSid());
            araModel.setRepeatKbn(rtaModel.getRtpaRepeatKbn());
            araModel.setRepeatType(rtaModel.getRtpaRepeatType());
            araModel.setRepeatFormId(rtaModel.getRftId());
            //アクションパラメータ情報取得
            RngActionparamDao rapDao = new RngActionparamDao(con);
            List<RngActionparamModel> rapModelList = rapDao.select(rtaModel.getRtpaSid());
            LinkedList<AddRngActionParamModel> arapModelList
                            = new LinkedList<AddRngActionParamModel>();
            for (RngActionparamModel rapModel : rapModelList) {
                AddRngActionParamModel arapModel = new AddRngActionParamModel();
                arapModel.setIndex(rapModel.getRapSort());
                arapModel.setName(rapModel.getRapName());
                arapModel.setUseKbn(rapModel.getRapUseKbn());
                arapModel.setConditionJson(
                        JSONObject.fromObject(rapModel.getRapConditionJson()));
                arapModel.setParamJson(
                        JSONObject.fromObject(rapModel.getRapParamJson()));
                arapModelList.add(arapModel);
            }
            araModel.setActionParamList(arapModelList);
            araModelList.add(araModel);
        }
        return araModelList;
    }

    /**
     * <br>[機  能] 決裁後アクション用ファイルを生成する
     * <br>[解  説]
     * <br>[備  考]
     * @param rtpSid 稟議テンプレートSID
     * @param con コネクション
     */
    public void createActionFile(
        int rtpSid,
        Connection con)
        throws SQLException, IOToolsException, JsonProcessingException {
        //決裁後アクション用ファイル生成
        GSTemporaryPathUtil temp = GSTemporaryPathUtil.getInstance();

        //既に存在する場合、処理を終了する
        File actionFile = new File(__getTempPath(), RngConst.RNG_FILE_ACTION);
        if (actionFile.exists()) {
            return;
        }

        temp.createTempDir(reqMdl__,
                RngConst.PLUGIN_ID_RINGI, DIRID,
                RngConst.RNG_FOLDER_ACTION);
        LinkedList<AddRngActionModel> araModelList = new LinkedList<>();
        //編集時、初期データをセットする
        if (rtpSid != -1) {
            RngTemplateDao rtpDao = new RngTemplateDao(con);
            int rtpVer = rtpDao.getMaxVerNo(rtpSid);
            if (rtpVer > 0) {
                araModelList = getAddRngActionMdlList(rtpSid, rtpVer, con);
            }
        }
        //ファイルに反映
        saveActionFile(araModelList);
    }

    /**
     * <br>[機  能] 決裁後アクション用ファイルを更新する
     * <br>[解  説]
     * <br>[備  考]
     * @param araModelList 決裁後アクションモデルリスト
     */
    public void saveActionFile(LinkedList<AddRngActionModel> araModelList)
            throws SQLException, IOToolsException, JsonProcessingException {
        ObjectFile objFile = new ObjectFile(__getTempPath(), RngConst.RNG_FILE_ACTION);
        JSONArray jsonArray = new JSONArray();

        //編集時、データをセットする
        for (AddRngActionModel araModel : araModelList) {
            //決裁後アクション情報
            JSONObject actionJson = new JSONObject();
            actionJson.put("index", araModel.getIndex());
            actionJson.put("name", NullDefault.getString(
                            araModel.getName(), ""));
            actionJson.put("useKbn", araModel.getUseKbn());
            actionJson.put("apiConnectSid", araModel.getApiConnectSid());
            actionJson.put("repeatKbn", araModel.getRepeatKbn());
            actionJson.put("repeatType", araModel.getRepeatType());
            actionJson.put("repeatFormId", NullDefault.getString(
                                araModel.getRepeatFormId(), ""));
            //アクションパラメータ情報
            JSONArray paramJsonArray = new JSONArray();
            for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                JSONObject paramJson = new JSONObject();
                paramJson.put("index", arapModel.getIndex());
                paramJson.put("name", arapModel.getName());
                paramJson.put("useKbn", arapModel.getUseKbn());
                paramJson.put("conditionJson", arapModel.getConditionJson());
                paramJson.put("paramJson", arapModel.getParamJson());
                paramJsonArray.add(paramJson);
            }
            actionJson.put("actionParam", paramJsonArray);
            jsonArray.add(actionJson);
        }
        objFile.save(jsonArray.toString());
    }

    /**
     * <br>[機  能] 決裁後アクションを追加する
     * <br>[解  説]
     * <br>[備  考]
     * @param addAraModel 追加する決裁後アクションモデル
     * @return 決裁後アクションIndex
     */
    public int addAction(AddRngActionModel addAraModel)
                throws SQLException, IOToolsException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();

        //Indexの最大値+1を新しい決裁後アクションに設定する
        int newIndex = 0;
        if (araModelList.size() > 0) {
            List<Integer> indexList = araModelList.stream()
                                                .map(mdl -> mdl.getIndex())
                                                .collect(Collectors.toList());
            newIndex = indexList.stream()
                        .max(Comparator.naturalOrder())
                        .get() + 1;
        }
        addAraModel.setIndex(newIndex);
        araModelList.add(addAraModel);
        //ファイルに反映
        saveActionFile(araModelList);
        return newIndex;
    }

    /**
     * <br>[機  能] 決裁後アクションを更新する
     * <br>[解  説] アクションパラメータリストは更新しない
     * <br>[備  考]
     * @param editAraModel 更新する決裁後アクションモデル
     * @param actionIndex 更新する決裁後アクション
     */
    public void updateAction(AddRngActionModel editAraModel, int actionIndex)
                throws SQLException, IOToolsException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();

        //対象の決裁後アクションを更新する
        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                araModel.setName(editAraModel.getName());
                araModel.setUseKbn(editAraModel.getUseKbn());
                araModel.setApiConnectSid(editAraModel.getApiConnectSid());
                araModel.setRepeatKbn(editAraModel.getRepeatKbn());
                araModel.setRepeatType(editAraModel.getRepeatType());
                araModel.setRepeatFormId(editAraModel.getRepeatFormId());
                araModel.setActionParamList(editAraModel.getActionParamList());
                break;
            }
        }
        //ファイルに反映
        saveActionFile(araModelList);
    }

    /**
     * <br>[機  能] 決裁後アクションを削除する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndexList 削除する決裁後アクション
     */
    public void removeAction(List<Integer> actionIndexList)
                throws SQLException, IOToolsException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        List<AddRngActionModel> removeModelList = new ArrayList<AddRngActionModel>();

        //対象の決裁後アクションを削除する
        for (AddRngActionModel araModel : araModelList) {
            if (actionIndexList.contains(araModel.getIndex())) {
                removeModelList.add(araModel);
            }
        }
        for (AddRngActionModel removeModel : removeModelList) {
            araModelList.remove(removeModel);
        }
        //ファイルに反映
        saveActionFile(araModelList);
    }

    /**
     * <br>[機  能] 決裁後アクションの並び替えを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param sortList 並び替え後の並び順
     */
    public void sortAction(ArrayList<Integer> sortList)
                    throws IOToolsException, SQLException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        LinkedList<AddRngActionModel> sortAraModelList = new LinkedList<AddRngActionModel>();
        for (int sortIndex : sortList) {
            for (AddRngActionModel araModel : araModelList) {
                if (araModel.getIndex() == sortIndex) {
                    sortAraModelList.add(araModel);
                    break;
                }
            }
        }
        //ファイルに反映
        saveActionFile(sortAraModelList);
    }

    /**
     * <br>[機  能] アクションパラメータを追加する
     * <br>[解  説]
     * <br>[備  考]
     * @param addArapModel 追加するアクションパラメータモデル
     * @param actionIndex 対象となる決裁後アクション
     * @return アクションパラメータIndex
     */
    public int addActionParam(AddRngActionParamModel addArapModel, int actionIndex)
                throws SQLException, IOToolsException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        int newIndex = 0;

        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                List<AddRngActionParamModel> arapModelList = araModel.getActionParamList();

                //Indexの最大値+1を新しい決裁後アクションに設定する
                if (arapModelList.size() > 0) {
                    List<Integer> indexList = arapModelList.stream()
                                                        .map(mdl -> mdl.getIndex())
                                                        .collect(Collectors.toList());
                    newIndex = indexList.stream()
                                .max(Comparator.naturalOrder())
                                .get() + 1;
                }
                addArapModel.setIndex(newIndex);
                arapModelList.add(addArapModel);
                break;
            }
        }
        //ファイルに反映
        saveActionFile(araModelList);
        return newIndex;
    }

    /**
     * <br>[機  能] アクションパラメータを更新する
     * <br>[解  説] 実行条件/パラメータは更新しない
     * <br>[備  考]
     * @param editArapModel 更新する決裁後アクションモデル
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 更新するアクションパラメータ
     */
    public void updateActionParam(
                AddRngActionParamModel editArapModel, int actionIndex, int paramIndex)
                throws SQLException, IOToolsException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();

        //対象のアクションパラメータを更新する
        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                List<AddRngActionParamModel> arapModelList = araModel.getActionParamList();
                for (AddRngActionParamModel arapModel : arapModelList) {
                    if (arapModel.getIndex() == paramIndex) {
                        arapModel.setIndex(editArapModel.getIndex());
                        arapModel.setName(editArapModel.getName());
                        arapModel.setUseKbn(editArapModel.getUseKbn());
                        break;
                    }
                }
                break;
            }
        }
        //ファイルに反映
        saveActionFile(araModelList);
    }

    /**
     * <br>[機  能] アクションパラメータを削除する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndexList 削除するアクションパラメータ
     */
    public void removeActionParam(int actionIndex, List<Integer> paramIndexList)
                throws SQLException, IOToolsException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        LinkedList<AddRngActionParamModel> removeModelList
                                                = new LinkedList<AddRngActionParamModel>();

        //対象のアクションパラメータを削除する
        AddRngActionModel araModel = araModelList.stream()
                .filter(m -> m.getIndex() == actionIndex)
                .findFirst()
                .orElse(null);
        if (araModel != null) {
            List<AddRngActionParamModel> arapModelList = araModel.getActionParamList();
            for (AddRngActionParamModel arapModel : arapModelList) {
                if (paramIndexList.contains(arapModel.getIndex())) {
                    removeModelList.add(arapModel);
                }
            }
            for (AddRngActionParamModel removeModel : removeModelList) {
                arapModelList.remove(removeModel);
            }
            //ファイルに反映
            saveActionFile(araModelList);
        }
    }

    /**
     * <br>[機  能] アクションパラメータの並び替えを行う
     * <br>[解  説]
     * <br>[備  考]
     * @param sortList 並び替え後の並び順
     * @param actionIndex 対象となる決裁後アクション
     */
    public void sortActionParam(ArrayList<Integer> sortList, int actionIndex)
                    throws IOToolsException, SQLException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        LinkedList<AddRngActionModel> sortAraModelList = new LinkedList<AddRngActionModel>();
        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                //アクションパラメータの並び替え
                LinkedList<AddRngActionParamModel> arapModelList
                        = new LinkedList<AddRngActionParamModel>();
                for (int sortIndex : sortList) {
                    for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                        if (arapModel.getIndex() == sortIndex) {
                            arapModelList.add(arapModel);
                            break;
                        }
                    }
                }
                araModel.setActionParamList(arapModelList);
                break;
            }
        }
        //ファイルに反映
        saveActionFile(sortAraModelList);
    }

    /**
     * <br>[機  能] 指定した決裁後アクションの実行条件を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @return 実行条件JSONモデル
     */
    public RngRapConditionJsonModel getConditionJsonModel(
        int actionIndex,
        int paramIndex) throws IOToolsException {

        String jsonStr
            = __getJson(actionIndex, paramIndex, RngConst.RNG_ACTION_JSON_CONDITION).toString();
        return __createRapConditionJsonModel(jsonStr);
    }

    /**
     * <br>[機  能] 指定した決裁後アクションのパラメータを更新する。
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @return JSONArray
     */
    public RngRapParamJsonModel getParamJsonModel(
        int actionIndex,
        int paramIndex) throws IOToolsException {

        String jsonStr
            = __getJson(actionIndex, paramIndex, RngConst.RNG_ACTION_JSON_PARAM).toString();
        return __createRapParamJsonModel(jsonStr);
    }

    /**
     * <br>[機  能] JSONを取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @param targetKbn 取得対象 0:実行条件 1:パラメータ
     * @return JSONArray
     */
    public JSONObject __getJson(int actionIndex, int paramIndex, int targetKbn)
                    throws IOToolsException {
        JSONObject ret = new JSONObject();
        List<AddRngActionParamModel> arapModelList = getActionParamModelList(actionIndex);
        for (AddRngActionParamModel arapModel : arapModelList) {
            if (arapModel.getIndex() == paramIndex) {
                if (targetKbn == RngConst.RNG_ACTION_JSON_CONDITION) {
                    ret = arapModel.getConditionJson();
                } else if (targetKbn == RngConst.RNG_ACTION_JSON_PARAM) {
                    ret = arapModel.getParamJson();
                }
                break;
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 指定した決裁後アクションの実行条件を更新する。
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @param rapConditionJsonMdl 実行条件JSONモデル
     */
    public void setConditionJsonModel(
        int actionIndex,
        int paramIndex,
        RngRapConditionJsonModel rapConditionJsonMdl)
        throws IOToolsException, SQLException, JsonProcessingException {

        JSONObject json = JSONObject.fromObject(__createConditionJson(rapConditionJsonMdl));
        __setJson(actionIndex, paramIndex, json, RngConst.RNG_ACTION_JSON_CONDITION);
    }

    /**
     * <br>[機  能] 指定した決裁後アクションのパラメータを更新する。
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @param rapParamJsonMdl パラメータ情報JSONモデル
     */
    public void setParamJsonModel(
        int actionIndex,
        int paramIndex,
        RngRapParamJsonModel rapParamJsonMdl)
        throws IOToolsException, SQLException, JsonProcessingException {

        JSONObject json = JSONObject.fromObject(__createParamJson(rapParamJsonMdl));
        __setJson(actionIndex, paramIndex, json, RngConst.RNG_ACTION_JSON_PARAM);
    }

    /**
     * <br>[機  能] JSONを更新する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @param json 更新するJSON
     * @param targetKbn 更新対象 0:実行条件 1:パラメータ
     */
    private void __setJson(int actionIndex, int paramIndex, JSONObject json, int targetKbn)
                    throws IOToolsException, SQLException, JsonProcessingException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                    if (arapModel.getIndex() == paramIndex) {
                        if (targetKbn == RngConst.RNG_ACTION_JSON_CONDITION) {
                            arapModel.setConditionJson(json);
                        } else if (targetKbn == RngConst.RNG_ACTION_JSON_PARAM) {
                            arapModel.setParamJson(json);
                        }
                        break;
                    }
                }
            }
        }
        //ファイルに反映
        saveActionFile(araModelList);
    }

    /**
     * <br>[機  能] 指定した決裁後アクションのアクションパラメータ一覧をモデル構造で取得する。
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @return JSONArray
     */
    public List<AddRngActionParamModel> getActionParamModelList(int actionIndex)
                    throws IOToolsException {
        List<AddRngActionParamModel> ret = new ArrayList<AddRngActionParamModel>();
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                ret = araModel.getActionParamList();
                break;
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 決裁後アクション用ファイルを解析し、モデル構造で取得する。
     * <br>[解  説]
     * <br>[備  考]
     * @return JSONArray
     */
    public LinkedList<AddRngActionModel> getActionModelList() throws IOToolsException {
        LinkedList<AddRngActionModel> ret = new LinkedList<AddRngActionModel>();
        //ファイル読み込み
        JSONArray jsonArray = __loadFile();
        //モデル構造に変換
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject actionJson = (JSONObject) jsonArray.get(i);
            AddRngActionModel araModel = __getActionModel(actionJson);
            ret.add(araModel);
        }
        return ret;
    }

    /**
     * <br>[機  能] 決裁後アクション用ファイルを読み込み、JSON形式で取得する
     * <br>[解  説]
     * <br>[備  考]
     * @return JSONArray
     */
    private JSONArray __loadFile() throws IOToolsException {
        ObjectFile objFile = new ObjectFile(__getTempPath(), RngConst.RNG_FILE_ACTION);
        String jsonText = String.valueOf(objFile.load());
        JSONArray jsonArray = JSONArray.fromObject(jsonText);
        return jsonArray;
    }

    /**
     * <br>[機  能] JSONから決裁後アクションモデルを生成する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionJson 対象となるJSON
     * @return JSONArray
     */
    private AddRngActionModel __getActionModel(JSONObject actionJson)
                    throws IOToolsException {
        //決裁後アクション情報
        AddRngActionModel araModel = new AddRngActionModel();
        araModel.setIndex((int) actionJson.get("index"));
        araModel.setName(
                NullDefault.getString((String) actionJson.get("name"), ""));
        araModel.setUseKbn((int) actionJson.get("useKbn"));
        araModel.setApiConnectSid((int) actionJson.get("apiConnectSid"));
        araModel.setRepeatKbn((int) actionJson.get("repeatKbn"));
        araModel.setRepeatType((int) actionJson.get("repeatType"));
        araModel.setRepeatFormId(
                NullDefault.getString((String) actionJson.get("repeatFormId"), ""));

        //アクションパラメータ情報
        LinkedList<AddRngActionParamModel> actionParamList
                        = new LinkedList<AddRngActionParamModel>();
        JSONArray actionParamArray = (JSONArray) actionJson.get("actionParam");
        for (int j = 0; j < actionParamArray.size(); j++) {
            actionParamList.add(__getActionParamModel((JSONObject) actionParamArray.get(j)));
        }
        araModel.setActionParamList(actionParamList);
        return araModel;
    }

    /**
     * <br>[機  能] JSONからアクションパラメータモデルを生成する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionParamJson 対象となるJSON
     * @return JSONArray
     */
    private AddRngActionParamModel __getActionParamModel(JSONObject actionParamJson)
                    throws IOToolsException {
        AddRngActionParamModel arapModel = new AddRngActionParamModel();
        arapModel.setIndex((int) actionParamJson.get("index"));
        arapModel.setName(
                NullDefault.getString((String) actionParamJson.get("name"), ""));
        arapModel.setUseKbn((int) actionParamJson.get("useKbn"));
        arapModel.setConditionJson((JSONObject) actionParamJson.get("conditionJson"));
        arapModel.setParamJson((JSONObject) actionParamJson.get("paramJson"));
        return arapModel;
    }

    /**
     * <br>[機  能] 実行条件JOSNを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param rapConditionJsonMdl 実行条件JSONモデル
     * @return 実行条件JSON文字列
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    private String __createConditionJson(
        RngRapConditionJsonModel rapConditionJsonMdl)
        throws JsonMappingException, JsonProcessingException {

        ObjectMapper mapper = new ObjectMapper();
        ObjectNode rapConditionJson = (ObjectNode) mapper.readTree("{}");
        rapConditionJson.put("conditionType", rapConditionJsonMdl.getConditionType());
        ArrayNode conditionList = mapper.createArrayNode();
        for (RngRapConditionModel rapConditionMdl : rapConditionJsonMdl.getConditionList()) {

            ObjectNode conditionJson = (ObjectNode) mapper.readTree("{}");
            conditionJson.put("paramKbn", rapConditionMdl.getParamKbn());
            if (rapConditionMdl.getParamFormId() != null
                && rapConditionMdl.getParamFormId().length() > 0) {
                conditionJson.put("paramFormId", rapConditionMdl.getParamFormId());
                conditionJson.put("paramFormName", rapConditionMdl.getParamFormName());
                conditionJson.put("paramFormType", rapConditionMdl.getParamFormType());
            }
            conditionJson.put("errorFlg", rapConditionMdl.isErrorFlg());
            conditionJson.put("errorTargetName", rapConditionMdl.getErrorTargetName());
            conditionJson.put("paramValue", rapConditionMdl.getParamValue());
            conditionJson.put("compareTarget", rapConditionMdl.getCompareTarget());
            conditionJson.put("compareType", rapConditionMdl.getCompareType());
            conditionList.add(conditionJson);
        }
        rapConditionJson.set("conditionList", conditionList);
        return mapper.writeValueAsString(rapConditionJson);
    }

    /**
     * <br>[機  能] パラメータ情報JOSNを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param rapParamJsonMdl パラメータ情報JSONモデル
     * @return パラメータ情報JSON文字列
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    private String __createParamJson(
        RngRapParamJsonModel rapParamJsonMdl)
        throws JsonMappingException, JsonProcessingException {

        ObjectMapper mapper = new ObjectMapper();
        ObjectNode rapParamJson = (ObjectNode) mapper.readTree("{}");

        //パスパラメータ
        ArrayNode pathParamList = mapper.createArrayNode();
        for (RngRapParamModel pathParam : rapParamJsonMdl.getPathParamList()) {
            //通常パラメータ
            pathParamList.add(__createNormalParamJson(pathParam));
        }
        rapParamJson.set("path", pathParamList);

        //クエリパラメータ
        ArrayNode queryParamList = mapper.createArrayNode();
        for (RngRapParamModel queryParam : rapParamJsonMdl.getQueryParamList()) {
            if (queryParam.getParamType().equals(RngConst.API_PARAMTYPE_LIST)) {
                //配列パラメータ
                queryParamList.add(__createListParamJson(queryParam));
            } else if (queryParam.getParamType().equals(RngConst.API_PARAMTYPE_PARAMETER)) {
                //通常パラメータ
                queryParamList.add(__createNormalParamJson(queryParam));
            }
        }
        rapParamJson.set("query", queryParamList);

        //リクエストボディパラメータ
        ArrayNode reqBodyParamList = mapper.createArrayNode();
        for (RngRapParamModel bodyParam : rapParamJsonMdl.getBodyParamList()) {

            if (bodyParam.getParamType().equals(RngConst.API_PARAMTYPE_LIST)) {
                //配列パラメータ
                reqBodyParamList.add(__createListParamJson(bodyParam));
            } else if (bodyParam.getParamType().equals(RngConst.API_PARAMTYPE_MODEL)) {
                //モデルパラメータ
                reqBodyParamList.add(__createModelParamJson(bodyParam));
            } else if (bodyParam.getParamType().equals(RngConst.API_PARAMTYPE_PARAMETER)) {
                //通常パラメータ
                reqBodyParamList.add(__createNormalParamJson(bodyParam));
            }
        }
        rapParamJson.set("body", reqBodyParamList);
        return mapper.writeValueAsString(rapParamJson);
    }

    /**
     * <br>[機  能] 通常パラメータJOSNを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param rngRapParamMdl パラメータ情報モデル
     * @return パラメータ情報JsonNode
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    private JsonNode __createNormalParamJson(
        RngRapParamModel rngRapParamMdl)
        throws JsonMappingException, JsonProcessingException {

        ObjectMapper mapper = new ObjectMapper();
        ObjectNode normalParamJson = (ObjectNode) mapper.readTree("{}");
        normalParamJson.put("name", rngRapParamMdl.getName());
        normalParamJson.put("useKbn", rngRapParamMdl.getUseKbn());
        normalParamJson.put("paramType", RngConst.API_PARAMTYPE_PARAMETER);
        normalParamJson.set("paramList", __createParamInfoJson(rngRapParamMdl.getParamInfoMap()));
        return normalParamJson;
    }

    /**
     * <br>[機  能] 配列パラメータJOSNを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param rngRapParamMdl パラメータ情報モデル
     * @return パラメータ情報JsonNode
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    private JsonNode __createListParamJson(
        RngRapParamModel rngRapParamMdl) throws JsonMappingException, JsonProcessingException {

        ObjectMapper mapper = new ObjectMapper();

        ObjectNode listParamJson = (ObjectNode) mapper.readTree("{}");
        listParamJson.put("name", rngRapParamMdl.getName());
        listParamJson.put("useKbn", rngRapParamMdl.getUseKbn());
        listParamJson.put("paramType", RngConst.API_PARAMTYPE_LIST);

        ArrayNode listParamList = mapper.createArrayNode();
        for (Entry<Integer, ListParamInfoModel> entry
            : rngRapParamMdl.getListParamInfoMap().entrySet()) {
            listParamList.add(__createParamInfoJson(entry.getValue().getParamInfoMap()));
        }
        listParamJson.set("paramList", listParamList);
        return listParamJson;
    }

    /**
     * <br>[機  能] モデルパラメータJOSNを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param rngRapParamMdl パラメータ情報モデル
     * @return パラメータ情報JsonNode
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    private JsonNode __createModelParamJson(
        RngRapParamModel rngRapParamMdl)
        throws JsonMappingException, JsonProcessingException {

        ObjectMapper mapper = new ObjectMapper();

        ObjectNode modelParamJson = (ObjectNode) mapper.readTree("{}");
        modelParamJson.put("name", rngRapParamMdl.getName());
        modelParamJson.put("useKbn", rngRapParamMdl.getUseKbn());
        modelParamJson.put("paramType", RngConst.API_PARAMTYPE_MODEL);

        ArrayNode childParamList = mapper.createArrayNode();
        for (Entry<Integer, ChildParamInfoModel> entry
            : rngRapParamMdl.getChildParamInfoMap().entrySet()) {

            ObjectNode childParamJson = (ObjectNode) mapper.readTree("{}");
            childParamJson.put("name", entry.getValue().getName());
            childParamJson.put("useKbn", entry.getValue().getUseKbn());
            if (entry.getValue().getParamType().equals(RngConst.API_PARAMTYPE_LIST)) {
                //配列パラメータ
                childParamJson.put("paramType", RngConst.API_PARAMTYPE_LIST);
                ArrayNode listParamList = mapper.createArrayNode();
                for (Entry<Integer, ListParamInfoModel> listParamEntry
                    : entry.getValue().getListParamInfoMap().entrySet()) {
                    listParamList.add(
                        __createParamInfoJson(listParamEntry.getValue().getParamInfoMap()));
                }
                childParamJson.set("paramList", listParamList);

            } else if (entry.getValue().getParamType().equals(RngConst.API_PARAMTYPE_PARAMETER)) {
                //通常パラメータ
                childParamJson.put("paramType", RngConst.API_PARAMTYPE_PARAMETER);
                childParamJson.set("paramList",
                    __createParamInfoJson(entry.getValue().getParamInfoMap()));
            }
            childParamList.add(childParamJson);
        }
        modelParamJson.set("paramList", childParamList);

        return modelParamJson;
    }

    /**
     * <br>[機  能] パラメータ情報からパラメータ情報JSONモデルを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param paramInfoMap パラメータ情報モデル
     * @return パラメータ情報JSON
     * @throws JsonProcessingException
     * @throws JsonMappingException
     */
    private ArrayNode __createParamInfoJson(
        TreeMap<Integer, ParamInfoModel> paramInfoMap)
        throws JsonMappingException, JsonProcessingException {

        ObjectMapper mapper = new ObjectMapper();
        ArrayNode ret = mapper.createArrayNode();
        for (Entry<Integer, ParamInfoModel> paramInfoEntry : paramInfoMap.entrySet()) {

            ObjectNode paramJson = (ObjectNode) mapper.readTree("{}");
            ParamInfoModel paramInfo = paramInfoEntry.getValue();
            paramJson.put("paramKbn", paramInfo.getParamKbn());
            if (paramInfo.getParamKbn() == RngConst.API_PARAMKBN_FORM) {
                paramJson.put("paramFormId", paramInfo.getParamFormId());
                paramJson.put("paramFormName", paramInfo.getParamFormName());
                paramJson.put("paramFormType", paramInfo.getParamFormType());
            }
            paramJson.put("errorFlg", paramInfo.isErrorFlg());
            paramJson.put("errorTargetName", paramInfo.getErrorTargetName());
            paramJson.put("paramValue", paramInfo.getParamValue());
            if (paramInfo.getParamKbn() == RngConst.API_PARAMKBN_MANUAL) {
                paramJson.put("paramValueManual", paramInfo.getParamValueManual());
            }
            paramJson.put("paramIndex", paramInfo.getParamIndex());
            ret.add(paramJson);
        }
        return ret;
    }

    /**
     * <br>[機  能] 実行条件JOSNから実行条件JSONモデルを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonStr 実行条件JSON文字列
     * @return 実行条件JSONモデル
     */
    private RngRapConditionJsonModel __createRapConditionJsonModel(String jsonStr) {
        RngRapConditionJsonModel rapConditionJsonMdl = new RngRapConditionJsonModel();

        ObjectMapper mapper = new ObjectMapper();
        JsonNode rapConditionJson;
        try {
            rapConditionJson = mapper.readTree(jsonStr);
        } catch (JsonProcessingException e) {
            //JSONの解析に失敗
            return null;
        }
        if (rapConditionJson == null || !rapConditionJson.has("conditionType")) {
            return null;
        }
        rapConditionJsonMdl.setConditionType(
            __getIntValue(rapConditionJson.get("conditionType"), 0));
        JsonNode conditionListJson = rapConditionJson.get("conditionList");
        List<RngRapConditionModel> rapConditionList = new ArrayList<RngRapConditionModel>();
        for (JsonNode conditionJson : conditionListJson) {
            RngRapConditionModel rapConditionMdl = new RngRapConditionModel();
            rapConditionMdl.setParamKbn(__getIntValue(conditionJson.get("paramKbn"), 0));
            if (__getIntValue(conditionJson.get("paramKbn"), 0) == RngConst.API_PARAMKBN_FORM) {
                rapConditionMdl.setParamFormId(__getStrValue(conditionJson.get("paramFormId"), ""));
                rapConditionMdl.setParamFormName(
                    __getStrValue(conditionJson.get("paramFormName"), ""));
                rapConditionMdl.setParamFormType(
                    __getIntValue(conditionJson.get("paramFormType"), 0));
            }
            rapConditionMdl.setErrorFlg(
                    __getBooleanValue(conditionJson.get("errorFlg"), false));
            rapConditionMdl.setErrorTargetName(
                __getStrValue(conditionJson.get("errorTargetName"), ""));
            rapConditionMdl.setParamValue(__getIntValue(conditionJson.get("paramValue"), 0));
            rapConditionMdl.setCompareTarget(__getStrValue(conditionJson.get("compareTarget"), ""));
            rapConditionMdl.setCompareType(__getIntValue(conditionJson.get("compareType"), 0));
            rapConditionList.add(rapConditionMdl);
        }
        rapConditionJsonMdl.setConditionList(rapConditionList);

        return rapConditionJsonMdl;
    }

    /**
     * <br>[機  能] パラメータ情報JOSNからパラメータ情報JSONモデルを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonStr パラメータ情報JSON文字列
     * @return パラメータ情報JSONモデル
     */
    private RngRapParamJsonModel __createRapParamJsonModel(String jsonStr) {
        RngRapParamJsonModel rapParamJsonMdl = new RngRapParamJsonModel();

        ObjectMapper mapper = new ObjectMapper();
        JsonNode rapParamJson;
        try {
            rapParamJson = mapper.readTree(jsonStr);
        } catch (JsonProcessingException e) {
            //JSONの解析に失敗
            return null;
        }
        if (rapParamJson == null || !rapParamJson.has("path")) {
            return null;
        }

        //パスパラメータ
        JsonNode pathParamListJson = rapParamJson.get("path");
        List<RngRapParamModel> pathParamList = new ArrayList<RngRapParamModel>();
        for (JsonNode pathParamJson : pathParamListJson) {
            pathParamList.add(__createNormalParamInfoModel(pathParamJson));
        }
        rapParamJsonMdl.setPathParamList(pathParamList);

        //クエリパラメータ
        JsonNode queryParamListJson = rapParamJson.get("query");
        List<RngRapParamModel> queryParamList = new ArrayList<RngRapParamModel>();
        for (JsonNode queryParamJson : queryParamListJson) {
            if (__getStrValue(queryParamJson.get("paramType"), "").equals(
                RngConst.API_PARAMTYPE_LIST)) {
                //配列パラメータ
                queryParamList.add(__createListParamInfoModel(queryParamJson));
            } else if (__getStrValue(queryParamJson.get("paramType"), "").equals(
                RngConst.API_PARAMTYPE_PARAMETER)) {
                //通常パラメータ
                queryParamList.add(__createNormalParamInfoModel(queryParamJson));
            }
        }
        rapParamJsonMdl.setQueryParamList(queryParamList);

        //リクエストボディパラメータ
        JsonNode bodyParamListJson = rapParamJson.get("body");
        List<RngRapParamModel> bodyParamList = new ArrayList<RngRapParamModel>();
        for (JsonNode bodyParamJson : bodyParamListJson) {
            if (bodyParamJson != null) {
                if (__getStrValue(bodyParamJson.get("paramType"), "").equals(
                    RngConst.API_PARAMTYPE_LIST)) {
                    //配列パラメータ
                    bodyParamList.add(__createListParamInfoModel(bodyParamJson));
                } else if (__getStrValue(bodyParamJson.get("paramType"), "").equals(
                    RngConst.API_PARAMTYPE_MODEL)) {
                    //モデルパラメータ
                    bodyParamList.add(__createModelParamInfoModel(bodyParamJson));
                } else if (__getStrValue(bodyParamJson.get("paramType"), "").equals(
                    RngConst.API_PARAMTYPE_PARAMETER)) {
                    //通常パラメータ
                    bodyParamList.add(__createNormalParamInfoModel(bodyParamJson));
                }
            }
        }
        rapParamJsonMdl.setBodyParamList(bodyParamList);

        return rapParamJsonMdl;
    }

    /**
     * <br>[機  能] パラメータ情報JOSNから取得した値をパラメータ情報モデルにセットする
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode パラメータ情報JsonNode
     * @param rapParamMdl パラメータ情報モデル
     */
    private void __setRapParamModel(JsonNode jsonNode, RngRapParamModel rapParamMdl) {

        rapParamMdl.setName(__getStrValue(jsonNode.get("name"), ""));
        rapParamMdl.setUseKbn(__getIntValue(jsonNode.get("useKbn"), 0));
        rapParamMdl.setParamType(__getStrValue(jsonNode.get("paramType"), ""));
    }

    /**
     * <br>[機  能] パラメータ情報JOSNから通常パラメータモデルを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode パラメータ情報JsonNode
     * @return パラメータ情報JSONモデル
     */
    private RngRapParamModel __createNormalParamInfoModel(JsonNode jsonNode) {

        RngRapParamModel rapParamMdl = new RngRapParamModel();
        __setRapParamModel(jsonNode, rapParamMdl);
        JsonNode paramListJson = jsonNode.get("paramList");
        rapParamMdl.setParamInfoMap(__createParamInfoMap(paramListJson));
        return rapParamMdl;
    }

    /**
     * <br>[機  能] パラメータ情報JOSNから配列パラメータモデルを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode パラメータ情報JsonNode
     * @return パラメータ情報JSONモデル
     */
    private RngRapParamModel __createListParamInfoModel(JsonNode jsonNode) {

        RngRapParamModel rapParamMdl = new RngRapParamModel();
        __setRapParamModel(jsonNode, rapParamMdl);
        JsonNode paramListJson = jsonNode.get("paramList");
        int listMapKey = 0;
        TreeMap<Integer, ListParamInfoModel> listParamMap
            = new TreeMap<Integer, ListParamInfoModel>();
        for (JsonNode listParamInfoJson : paramListJson) {
            ListParamInfoModel listParamInfo = new ListParamInfoModel();
            listParamInfo.setParamInfoMap(__createParamInfoMap(listParamInfoJson));
            listParamMap.put(listMapKey, listParamInfo);
            listMapKey++;
        }
        rapParamMdl.setListParamInfoMap(listParamMap);
        return rapParamMdl;
    }

    /**
     * <br>[機  能] パラメータ情報JOSNからモデルパラメータモデルを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode パラメータ情報JsonNode
     * @return パラメータ情報JSONモデル
     */
    private RngRapParamModel __createModelParamInfoModel(JsonNode jsonNode) {

        RngRapParamModel rapParamMdl = new RngRapParamModel();
        __setRapParamModel(jsonNode, rapParamMdl);
        JsonNode paramListJson = jsonNode.get("paramList");

        int childMapKey = 0;
        TreeMap<Integer, ChildParamInfoModel> childMap
            = new TreeMap<Integer, ChildParamInfoModel>();
        for (JsonNode childParamJson : paramListJson) {
            ChildParamInfoModel childMdl = new ChildParamInfoModel();
            childMdl.setName(__getStrValue(childParamJson.get("name"), ""));
            childMdl.setUseKbn(__getIntValue(childParamJson.get("useKbn"), 0));
            childMdl.setParamType(__getStrValue(childParamJson.get("paramType"), ""));

            if (childParamJson.get("paramList") != null
                && childParamJson.get("paramList").get(0) != null
                && childParamJson.get("paramList").get(0).isArray()) {
                //配列パラメータ
                JsonNode listParamJson = childParamJson.get("paramList");
                int listMapKey = 0;
                TreeMap<Integer, ListParamInfoModel> listParamMap
                    = new TreeMap<Integer, ListParamInfoModel>();
                for (JsonNode listParamInfoJson : listParamJson) {
                    ListParamInfoModel listParamInfo = new ListParamInfoModel();
                    listParamInfo.setParamInfoMap(__createParamInfoMap(listParamInfoJson));
                    listParamMap.put(listMapKey, listParamInfo);
                    listMapKey++;
                }
                childMdl.setListParamInfoMap(listParamMap);
                childMap.put(childMapKey, childMdl);
                childMapKey++;

            } else {
                //通常パラメータ
                JsonNode paramInfoListJson = childParamJson.get("paramList");
                childMdl.setParamInfoMap(__createParamInfoMap(paramInfoListJson));
                childMap.put(childMapKey, childMdl);
                childMapKey++;
            }
        }
        rapParamMdl.setChildParamInfoMap(childMap);
        return rapParamMdl;
    }

    /**
     * <br>[機  能] JOSNからパラメータ情報モデルリストを作成する
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode パラメータ情報JsonNode
     * @return パラメータ情報JSONモデル
     */
    private TreeMap<Integer, ParamInfoModel> __createParamInfoMap(JsonNode jsonNode) {

        int paramMapKey = 0;
        TreeMap<Integer, ParamInfoModel> ret
            = new TreeMap<Integer, ParamInfoModel>();
        if (jsonNode != null) {
            for (JsonNode paramInfoJson : jsonNode) {
                ParamInfoModel paramInfoMdl = new ParamInfoModel();
                paramInfoMdl.setParamKbn(__getIntValue(paramInfoJson.get("paramKbn"), 0));
                if (__getIntValue(paramInfoJson.get("paramKbn"), 0) == RngConst.API_PARAMKBN_FORM) {
                    paramInfoMdl.setParamFormId(
                        __getStrValue(paramInfoJson.get("paramFormId"), ""));
                    paramInfoMdl.setParamFormName(
                        __getStrValue(paramInfoJson.get("paramFormName"), ""));
                    paramInfoMdl.setParamFormType(
                        __getIntValue(paramInfoJson.get("paramFormType"), 0));
                }
                paramInfoMdl.setErrorFlg(
                        __getBooleanValue(paramInfoJson.get("errorFlg"), false));
                paramInfoMdl.setErrorTargetName(
                    __getStrValue(paramInfoJson.get("errorTargetName"), ""));
                paramInfoMdl.setParamValue(__getIntValue(paramInfoJson.get("paramValue"), 0));
                if (__getIntValue(paramInfoJson.get("paramKbn"), 0)
                    == RngConst.API_PARAMKBN_MANUAL) {
                    paramInfoMdl.setParamValueManual(
                        __getStrValue(paramInfoJson.get("paramValueManual"), ""));
                }
                paramInfoMdl.setParamIndex(__getIntValue(paramInfoJson.get("paramIndex"), 0));
                ret.put(paramMapKey, paramInfoMdl);
                paramMapKey++;
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] JsonNodeがNULLの場合、デフォルト値を返す
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode JsonNode
     * @param defult デフォルト値
     * @return String
     */
    private String __getStrValue(JsonNode jsonNode, String defult) {

        if (jsonNode != null) {
            return jsonNode.asText();
        } else {
            return defult;
        }
    }

    /**
     * <br>[機  能] JsonNodeがNULLの場合、デフォルト値を返す
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode JsonNode
     * @param defult デフォルト値
     * @return int
     */
    private int __getIntValue(JsonNode jsonNode, int defult) {

        if (jsonNode != null) {
            return jsonNode.asInt();
        } else {
            return defult;
        }
    }

    /**
     * <br>[機  能] JsonNodeがNULLの場合、デフォルト値を返す
     * <br>[解  説]
     * <br>[備  考]
     * @param jsonNode JsonNode
     * @param defult デフォルト値
     * @return boolean
     */
    private boolean __getBooleanValue(JsonNode jsonNode, boolean defult) {

        if (jsonNode != null) {
            return jsonNode.asBoolean();
        } else {
            return defult;
        }
    }

    /**
     * <br>[機  能] 一番最後に追加された決裁後アクションのIndexを取得する
     * <br>[解  説]
     * <br>[備  考]
     * @return 決裁後アクションIndex
     */
    public int getMaxActionIndex() throws IOToolsException {
        int ret = -1;
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        List<Integer> indexList = araModelList.stream()
                .map(mdl -> mdl.getIndex())
                .collect(Collectors.toList());
        ret = indexList.stream()
                .max(Comparator.naturalOrder())
                .get();
        return ret;
    }

    /**
     * <br>[機  能] 指定した決裁後アクション情報をモデル構造で取得する。
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @return JSONArray
     */
    public AddRngActionModel getActionModel(int actionIndex) throws IOToolsException {
        AddRngActionModel ret = null;
        //ファイル読み込み
        JSONArray jsonArray = __loadFile();
        //モデル構造に変換
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject actionJson = (JSONObject) jsonArray.get(i);
            if (actionIndex == (int) actionJson.get("index")) {
                ret = __getActionModel(actionJson);
                break;
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 指定したアクションパラメータ情報をモデル構造で取得する。
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex 対象となる決裁後アクション
     * @param paramIndex 対象となるアクションパラメータ
     * @return JSONArray
     */
    public AddRngActionParamModel getActionParamModel(int actionIndex, int paramIndex)
                    throws IOToolsException {
        AddRngActionParamModel ret = null;
        List<AddRngActionParamModel> arapModelList = getActionParamModelList(actionIndex);
        for (AddRngActionParamModel arapModel : arapModelList) {
            if (arapModel.getIndex() == paramIndex) {
                ret = arapModel;
                break;
            }
        }
        return ret;
    }
    /**
     * <br>[機  能] 申請内容の操作に伴い、使用している箇所を削除する
     * <br>[解  説] 実行条件/パラメータが対象
     * <br>[備  考]
     * @param targetFormId フォームID
     * @return 警告を表示する決裁後アクションのIndex
     */
    public List<Integer> removeFormParam(String targetFormId)
                    throws IOToolsException, JsonProcessingException, SQLException {

        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        LinkedList<AddRngActionModel> afterAraModelList
                                    = new LinkedList<AddRngActionModel>();
        List<Integer> warnIndexList = new ArrayList<Integer>();
        for (AddRngActionModel araModel : araModelList) {
            //繰り返し対象
            if (araModel.getRepeatFormId().equals(targetFormId)) {
                araModel.setRepeatFormId("");
                warnIndexList.add(araModel.getIndex());
            }
            LinkedList<AddRngActionParamModel> afterArapModelList
                                        = new LinkedList<AddRngActionParamModel>();
            for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                //実行条件
                if (arapModel.getConditionJson().toString().contains(escapeFormId(targetFormId))) {
                    arapModel = updateFormParamToCondition(
                        targetFormId, "", araModel,
                        arapModel, new String[0], -1,
                        RngConst.API_FORMACTION_FORM_DELETE);
                }
                //パラメータ
                if (arapModel.getParamJson().toString().contains(escapeFormId(targetFormId))) {
                    arapModel = updateFormParamToParameter(
                        targetFormId, "", arapModel, -1,
                        RngConst.API_FORMACTION_FORM_DELETE);
                }
                afterArapModelList.add(arapModel);
                //警告表示が有効の時、レスポンス用リストにIndexを追加する
                if ((arapModel.getWarnFlg() == RngConst.ACTIONPARAM_WARN_FLG_YES)
                        && !warnIndexList.contains(araModel.getIndex())) {
                    warnIndexList.add(araModel.getIndex());
                }
            }
            araModel.setActionParamList(afterArapModelList);
            afterAraModelList.add(araModel);
        }
        saveActionFile(afterAraModelList);
        return warnIndexList;
    }
    /**
     * <br>[機  能] 申請内容の操作に伴い、使用している箇所を更新する
     * <br>[解  説] 実行条件/パラメータが対象
     * <br>[備  考]
     * @param newFormId フォームID
     * @param targetFormId 更新前のフォームID
     * @param formValueList フォームの選択肢 (ラジオ/コンボ/チェック時のみ)
     * @param multiFlg フォームの単体/複数区分 (ユーザ選択/グループ選択時のみ)
     * @return 警告を表示する決裁後アクションのIndex
     */
    public List<Integer> editFormParam(String newFormId, String targetFormId,
                                String[] formValueList, int multiFlg)
                    throws IOToolsException, JsonProcessingException, SQLException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        LinkedList<AddRngActionModel> afterAraModelList
                                    = new LinkedList<AddRngActionModel>();
        List<Integer> warnIndexList = new ArrayList<Integer>();
        for (AddRngActionModel araModel : araModelList) {
            //繰り返し対象
            if (araModel.getRepeatFormId().equals(targetFormId)) {
                araModel.setRepeatFormId(newFormId);
            }
            LinkedList<AddRngActionParamModel> afterArapModelList
                                        = new LinkedList<AddRngActionParamModel>();
            for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                //実行条件
                if (arapModel.getConditionJson().toString().contains(escapeFormId(targetFormId))) {
                    arapModel = updateFormParamToCondition(
                        targetFormId, newFormId, araModel, arapModel,
                        formValueList, multiFlg,
                        RngConst.API_FORMACTION_FORMID_UPDATE);
                }
                //パラメータ
                if (arapModel.getParamJson().toString().contains(escapeFormId(targetFormId))) {
                    arapModel = updateFormParamToParameter(
                        targetFormId, newFormId, arapModel, multiFlg,
                        RngConst.API_FORMACTION_FORMID_UPDATE);
                }
                afterArapModelList.add(arapModel);
                //警告表示が有効の時、レスポンス用リストにIndexを追加する
                if ((arapModel.getWarnFlg() == RngConst.ACTIONPARAM_WARN_FLG_YES)
                        && !warnIndexList.contains(araModel.getIndex())) {
                    warnIndexList.add(araModel.getIndex());
                }
            }
            araModel.setActionParamList(afterArapModelList);
            afterAraModelList.add(araModel);
        }
        saveActionFile(afterAraModelList);
        return warnIndexList;
    }

    /**
     * <br>[機  能] 申請内容の操作に伴う実行条件の更新
     * <br>[解  説]
     * <br>[備  考]
     * @param targetFormId 対象フォームID
     * @param newFormId 更新後フォームID(編集時のみ使用)
     * @param araModel 決裁後アクションモデル
     * @param arapModel アクションパラメータモデル
     * @param formValueList フォームの選択肢 (ラジオ/コンボ/チェック時のみ)
     * @param multiFlg フォームの単体/複数区分 (ユーザ選択/グループ選択時のみ)
     * @param action 操作
     * @return 更新後のアクションパラメータモデル
     */
    public AddRngActionParamModel updateFormParamToCondition(String targetFormId,
            String newFormId,
            AddRngActionModel araModel,
            AddRngActionParamModel arapModel,
            String[] formValueList,
            int multiFlg,
            int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        //実行条件JSONをモデル化した状態で取得
        RngRapConditionJsonModel condJsonModel
                            = getConditionJsonModel(
                            araModel.getIndex(), arapModel.getIndex());
        //条件一覧を取得
        List<RngRapConditionModel> condModelList
                = condJsonModel.getConditionList();
        List<RngRapConditionModel> afterCondModelList
                    = new ArrayList<RngRapConditionModel>();
        for (RngRapConditionModel condModel : condModelList) {
            if (condModel.getParamKbn() == RngConst.API_PARAMKBN_FORM
                    && condModel.getParamFormId().equals(targetFormId)) {
                if (action == RngConst.API_FORMACTION_FORMID_UPDATE
                        && !condModel.isErrorFlg()) {
                    //フォームIDを更新
                    condModel.setParamFormId(newFormId);
                } else if (action == RngConst.API_FORMACTION_FORM_DELETE) {
                    //削除
                    condModel.setErrorFlg(true);
                    arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
                }
                if (formValueList != null
                        && formValueList.length > 0) {
                    //選択要素を持つフォーム要素変更時、変更によって選択肢が消えていないかチェックする
                    boolean existFlg = false;
                    for (String value : formValueList) {
                        // if (condModel.getCompareTarget())
                        if (condModel.getCompareTarget().equals(value)) {
                            existFlg = true;
                            break;
                        }
                    }
                    //選択肢の中に存在しなかった場合、エラーとする
                    if (!existFlg) {
                        condModel.setErrorFlg(true);
                        arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
                    }
                }
                if (multiFlg != -1) {
                    //単一選択/複数選択を持つフォーム要素変更時、実行条件が適切かチェックする
                    if (((condModel.getCompareType() == 1 || condModel.getCompareType() == 2)
                            && multiFlg == 1)
                            ||
                            ((condModel.getCompareType() == 3 || condModel.getCompareType() == 4)
                            && multiFlg == 0)) {
                        condModel.setErrorFlg(true);
                        arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
                    }
                }
            }
            afterCondModelList.add(condModel);
        }
        //更新が完了した条件一覧をセットする。
        condJsonModel.setConditionList(afterCondModelList);
        JSONObject json = JSONObject.fromObject(__createConditionJson(condJsonModel));
        arapModel.setConditionJson(json);
        return arapModel;
    }

    /**
     * <br>[機  能] 申請内容の操作に伴うパラメータの更新
     * <br>[解  説]
     * <br>[備  考]
     * @param targetFormId 対象フォームID
     * @param newFormId 更新後フォームID(編集時のみ使用)
     * @param arapModel アクションパラメータモデル
     * @param multiFlg フォームの単体/複数区分 (ユーザ選択/グループ選択時のみ)
     * @param action 操作
     * @return 更新後のアクションパラメータモデル
     */
    public AddRngActionParamModel updateFormParamToParameter(String targetFormId,
            String newFormId,
            AddRngActionParamModel arapModel,
            int multiFlg,
            int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        RngRapParamJsonModel paramJsonModel = __createRapParamJsonModel(
                                                    arapModel.getParamJson().toString());
        //パス
        paramJsonModel.setPathParamList(__paramUpdate(
                        targetFormId, newFormId, paramJsonModel.getPathParamList(),
                        -1, action));
        //クエリ
        paramJsonModel.setQueryParamList(__paramUpdate(
                        targetFormId, newFormId, paramJsonModel.getQueryParamList(),
                        multiFlg, action));
        //ボディ
        paramJsonModel.setBodyParamList(__paramUpdate(
                        targetFormId, newFormId, paramJsonModel.getBodyParamList(),
                        multiFlg, action));

        JSONObject json = JSONObject.fromObject(__createParamJson(paramJsonModel));
        arapModel.setParamJson(json);

        //エラーフラグが有効になっている場合、警告を表示
        if (json.toString().contains("\"errorFlg\":true")) {
            arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
        }

        return arapModel;
    }

    /**
     * <br>[機  能] パラメータの更新
     * <br>[解  説]
     * <br>[備  考]
     * @param targetFormId 対象フォームID
     * @param newFormId 更新後フォームID(編集時のみ使用)
     * @param paramModelList 該当パラメータのパラメータモデルリスト
     * @param multiFlg フォームの単体/複数区分 (ユーザ選択/グループ選択時のみ)
     * @param action 操作
     * @return 更新後のアクションパラメータモデル
     */
    public List<RngRapParamModel> __paramUpdate(String targetFormId,
                    String newFormId,
                    List<RngRapParamModel> paramModelList,
                    int multiFlg,
                    int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        for (RngRapParamModel paramModel : paramModelList) {
            //パラメータリスト
            TreeMap<Integer, ParamInfoModel> paramInfoMap = updateParamInfoMap(
                                                                targetFormId,
                                                                newFormId,
                                                                -1,
                                                                action,
                                                                paramModel.getParamInfoMap(),
                                                                false);
            paramModel.setParamInfoMap(paramInfoMap);
            //配列パラメータリスト
            int keyIndex = 0;
            TreeMap<Integer, ListParamInfoModel> listParamInfoMap = new TreeMap<>();
            for (Entry<Integer, ListParamInfoModel> entry
                        : paramModel.getListParamInfoMap().entrySet()) {
                //1つの配列内にあるパラメータリスト
                paramInfoMap = updateParamInfoMap(targetFormId, newFormId, multiFlg, action,
                                                    entry.getValue().getParamInfoMap(), true);

                ListParamInfoModel listParamInfoModel = new ListParamInfoModel();

                listParamInfoModel.setParamInfoMap(paramInfoMap);
                listParamInfoMap.put(keyIndex, listParamInfoModel);
                keyIndex++;
            }
            paramModel.setListParamInfoMap(listParamInfoMap);

            //モデルパラメータリスト
            keyIndex = 0;
            TreeMap<Integer, ChildParamInfoModel> childParamInfoMap = new TreeMap<>();
            for (Entry<Integer, ChildParamInfoModel> entry
                        : paramModel.getChildParamInfoMap().entrySet()) {
                ChildParamInfoModel childParamInfoModel = new ChildParamInfoModel();
                childParamInfoModel.setName(entry.getValue().getName());
                childParamInfoModel.setUseKbn(entry.getValue().getUseKbn());
                childParamInfoModel.setParamType(entry.getValue().getParamType());
                if (entry.getValue().getParamInfoMap().size() > 0) {
                    //通常パラメータ
                    paramInfoMap = updateParamInfoMap(targetFormId, newFormId, -1, action,
                                                    entry.getValue().getParamInfoMap(), false);
                    childParamInfoModel.setParamInfoMap(paramInfoMap);
                } else if (entry.getValue().getListParamInfoMap().size() > 0) {
                    //配列パラメータ
                    listParamInfoMap = new TreeMap<>();
                    int listKeyIndex = 0;
                    for (Entry<Integer, ListParamInfoModel> listEntry
                                 : entry.getValue().getListParamInfoMap().entrySet()) {
                        paramInfoMap = updateParamInfoMap(targetFormId, newFormId, multiFlg, action,
                                                    listEntry.getValue().getParamInfoMap(), true);
                        ListParamInfoModel listParamInfoModel
                                                = new ListParamInfoModel();
                        listParamInfoModel.setParamInfoMap(paramInfoMap);
                        listParamInfoMap.put(listKeyIndex, listParamInfoModel);
                        listKeyIndex++;
                    }
                    childParamInfoModel.setListParamInfoMap(listParamInfoMap);
                }
                childParamInfoMap.put(keyIndex, childParamInfoModel);
                keyIndex++;
            }
            paramModel.setChildParamInfoMap(childParamInfoMap);
        }
        return paramModelList;
    }

    /**
     * <br>[機  能] 申請内容の操作に伴うパラメータの更新
     * <br>[解  説]
     * <br>[備  考]
     * @param targetFormId 対象フォームID
     * @param newFormId 更新後フォームID(編集時のみ使用)
     * @param multiFlg フォームの単体/複数区分 (ユーザ選択/グループ選択時のみ)
     * @param action 操作
     * @param treeMap パラメータマップ
     * @param listFlg 配列フラグ
     * @return 更新後のアクションパラメータモデル
     */
    public TreeMap<Integer, ParamInfoModel> updateParamInfoMap(String targetFormId,
                    String newFormId,
                    int multiFlg,
                    int action,
                    TreeMap<Integer, ParamInfoModel> treeMap,
                    boolean listFlg)
                    throws IOToolsException, JsonProcessingException, SQLException {

        TreeMap<Integer, ParamInfoModel> paramInfoMap = new TreeMap<>();
        int keyIndex = 0;
        //パラメータリスト
        for (Entry<Integer, ParamInfoModel> entry
                    : treeMap.entrySet()) {
            ParamInfoModel paramInfoModel = entry.getValue();
            if (paramInfoModel.getParamKbn() == RngConst.API_PARAMKBN_FORM
                    && paramInfoModel.getParamFormId().equals(targetFormId)) {
                if (action == RngConst.API_FORMACTION_FORMID_UPDATE
                                && !paramInfoModel.isErrorFlg()) {
                    //フォームID更新
                    paramInfoModel.setParamFormId(newFormId);
                } else if (action == RngConst.API_FORMACTION_FORM_DELETE) {
                    //フォーム削除
                    paramInfoModel.setErrorFlg(true);
                } else if (action == RngConst.API_FORMACTION_INIT_OTHER
                                && !paramInfoModel.isErrorFlg()) {
                    //表要素>ボディ以外として初期化
                    paramInfoModel.setParamIndex(-1);
                } else if (action == RngConst.API_FORMACTION_INIT_BODY
                                && !paramInfoModel.isErrorFlg()) {
                    //表要素>ボディとして初期化
                    if (listFlg) {
                        paramInfoModel.setParamIndex(0);
                    } else {
                        paramInfoModel.setParamIndex(1);
                    }
                }
                if (multiFlg == 1 && treeMap.entrySet().size() > 1) {
                    //1つの配列要素内に複数選択要素を含む2つ以上の要素が存在時、エラーとする。
                    paramInfoModel.setErrorFlg(true);
                }
            }
            paramInfoMap.put(keyIndex, paramInfoModel);
            keyIndex++;
        }
        return paramInfoMap;
    }
    /**
     * <br>[機  能] 決裁後アクションの使用区分を更新する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionIndex フォームID
     * @param boolUseKbn 使用区分 true:使用 false:未使用
     */
    public void changeActionUseKbn(int actionIndex, boolean boolUseKbn)
                    throws IOToolsException, JsonProcessingException, SQLException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        for (AddRngActionModel araModel : araModelList) {
            if (araModel.getIndex() == actionIndex) {
                int useKbn = RngConst.API_PARAM_NOT_USE;
                if (boolUseKbn) {
                    useKbn = RngConst.API_PARAM_USE;
                }
                araModel.setUseKbn(useKbn);
                break;
            }
        }
        saveActionFile(araModelList);
    }
    /**
     * <br>[機  能] 申請内容の移動に伴うパラメータの初期化
     * <br>[解  説]
     * <br>[備  考]
     * @param targetFormId フォームID
     * @param tableFormId 対象を含む表要素のフォームID
     * @param action 操作
     */
    public void initBlockParamIndex(String targetFormId, String tableFormId, int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        LinkedList<AddRngActionModel> araModelList = getActionModelList();
        LinkedList<AddRngActionModel> afterAraModelList
                                    = new LinkedList<AddRngActionModel>();
        for (AddRngActionModel araModel : araModelList) {
            LinkedList<AddRngActionParamModel> afterArapModelList
                                        = new LinkedList<AddRngActionParamModel>();
            for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                if (araModel.getRepeatKbn() == RngConst.API_REPEAT_KBN_OFF
                            || (araModel.getRepeatKbn() == RngConst.API_REPEAT_KBN_ON
                                && (araModel.getRepeatFormId() == null
                                || !araModel.getRepeatFormId().equals(tableFormId)))) {
                    //パラメータ
                    if (arapModel.getParamJson().toString().contains(
                                            escapeFormId(targetFormId))) {
                        arapModel = updateFormParamToParameter(
                            targetFormId, "", arapModel, -1, action);
                    }
                }
                afterArapModelList.add(arapModel);
            }
            araModel.setActionParamList(afterArapModelList);
            afterAraModelList.add(araModel);
        }
        saveActionFile(afterAraModelList);
    }
    /**
     * <br>[機  能] 行指定パラメータの初期化
     * <br>[解  説]
     * <br>[備  考]
     * @param araModel アクションパラメータモデル
     * @param targetFormIdList フォームID
     * @param action 操作内容
     */
    public void initBlockParamIndex(AddRngActionModel araModel, List<String> targetFormIdList,
                                    int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        LinkedList<AddRngActionParamModel> afterArapModelList
                                    = new LinkedList<AddRngActionParamModel>();
        for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
            AddRngActionParamModel afterArapModel = arapModel;
            for (String targetFormId : targetFormIdList) {
                //パラメータ
                if (afterArapModel.getParamJson().toString().contains(
                                        escapeFormId(targetFormId))) {
                    afterArapModel = updateFormParamToParameter(
                        targetFormId, "",
                        afterArapModel, -1, action);
                }
            }
            afterArapModelList.add(afterArapModel);
        }
        araModel.setActionParamList(afterArapModelList);
    }

    /**
     * <br>[機  能] 警告チェックなどのチェックを行い、結果を決裁後アクション情報，アクションパラメータ情報に保存する
     * <br>[解  説]
     * <br>[備  考]
     * @param actionModelList 決裁後アクション一覧
     * @param templateFormJson テンプレートのフォームJSON文字列
     * @param con コネクション
     * @param versionErrorSid バージョンエラーになっている連携APIのSID
     * @param isReflectAction true:パラメータで警告が出た場合に決裁後アクション側に警告フラグを立てる，false:立てない
     * @throws SQLException
     */
    public void checkFormCondition(
        List<AddRngActionModel> actionModelList,
        String templateFormJson,
        Connection con,
        List<Integer> versionErrorSid,
        boolean isReflectAction) throws SQLException {

        for (AddRngActionModel araModel : actionModelList) {
            //警告チェック
            if (isRepeatFormError(araModel)) {
                araModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
            } else {
                for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
                    //アクションパラメータに問題があるかを確認
                    checkActionParam(arapModel, araModel, templateFormJson, con);
                    if (arapModel.getWarnFlg() == RngConst.ACTIONPARAM_WARN_FLG_YES) {
                        if (isReflectAction) {
                            araModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
                        }
                        //アクションパラメータに1つでも問題がある場合は処理を終了
                        break;
                    }
                }
            }
            //使用している連携APIのバージョンチェック
            if (versionErrorSid.contains(araModel.getApiConnectSid())) {
                araModel.setVersionFlg(GSConstCommon.CAC_VERSION_MISMATCH);
            }
        }
    }

    /**
     * <br>[機  能] 追加用決裁後アクションの繰り返し実行対象にエラーがあるかを確認する
     * <br>[解  説]
     * <br>[備  考]
     * @param araModel 追加用決裁後アクション情報
     * @return true:追加用決裁後アクションの繰り返し実行対象にエラーがある, false:エラーがない
     */
    public boolean isRepeatFormError(AddRngActionModel araModel) {
        return araModel.getRepeatKbn() == RngConst.API_REPEAT_KBN_ON
            && araModel.getRepeatType() == RngConst.API_REPEAT_TYPE_FORM
            && araModel.getRepeatFormId().equals("");
    }

    /**
     * <br>[機  能] 追加用アクションパラメータモデルにエラーがあるかを確認する
     * <br>[解  説]
     * <br>[備  考] エラーがある場合、追加用アクションパラメータモデルにフラグを立てる
     * @param arapModel 追加用アクションパラメータ情報
     * @param araModel 追加用決裁後アクション情報
     * @param templateFormJson テンプレート内のフォーム要素情報を保持したJSON文字列
     * @param con コネクション
     * @throws SQLException
     */
    public void checkActionParam(
        AddRngActionParamModel arapModel,
        AddRngActionModel araModel,
        String templateFormJson, Connection con) throws SQLException {

        RngValidate rngValidate = new RngValidate(new GsMessage(reqMdl__));
        boolean isError = rngValidate.isValidateParamError(
                                __createRapParamJsonModel(
                                    arapModel.getParamJson().toString()),
                                araModel,
                                templateFormJson,
                                con,
                                reqMdl__);
        //入力エラー
        if (isError) {
            arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
            return;
        }
        //削除済みの申請要素を使用
        if (arapModel.getParamJson().toString().contains(
                "\"errorFlg\":true")
            || arapModel.getConditionJson().toString().contains(
                "\"errorFlg\":true")) {
            arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
            return;
        }
        //パラメータJSONが未入力 (アクションパラメータ追加 → 値を設定せずにキャンセルで戻ると発生)
        if (arapModel.getParamJson().equals(new JSONObject())) {
            arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
            return;
        }

        //条件JSONチェック
        RngRapConditionJsonModel conditionModel
            = __createRapConditionJsonModel(arapModel.getConditionJson().toString());
        List<String> errors = rngValidate.validateCondition(
            new ArrayList<>(), conditionModel, templateFormJson, con, reqMdl__);
        if (errors != null && !errors.isEmpty()) {
            arapModel.setWarnFlg(RngConst.ACTIONPARAM_WARN_FLG_YES);
        }
    }

    /**
     * <br>[機  能] フォームIDをエスケープした文字列に変換する
     * <br>[解  説] Jacksonによる文字列のJSON化を利用して同等のエスケープを行う。
     * <br>[備  考]
     * @param formId フォームID(エスケープ前)
     * @return エスケープ後のフォームID
     */
    public String escapeFormId(String formId)
            throws JsonProcessingException {
        StringBuilder retSB = new StringBuilder();
        StringCharacterIterator stit = new StringCharacterIterator(formId);

        for (
            char c = stit.first();
            c != CharacterIterator.DONE;
            c = stit.next()
        ) {
            switch (c) {
                case '\\':
                    retSB.append("\\\\");
                    break;
                case '"':
                    retSB.append("\\\"");
                    break;
                case '/':
                    retSB.append("\\/");
                    break;
                default :
                    retSB.append(c);
                    break;
            }
        }
        return retSB.toString();
    }

    /**
     * <br>[機  能] 確認時添付パラメータの初期化
     * <br>[解  説]
     * <br>[備  考]
     * @param araModel アクションパラメータモデル
     * @param action 操作内容
     */
    public void initConfirmFileParam(AddRngActionModel araModel,
                                    int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        LinkedList<AddRngActionParamModel> afterArapModelList
                                    = new LinkedList<AddRngActionParamModel>();
        for (AddRngActionParamModel arapModel : araModel.getActionParamList()) {
            AddRngActionParamModel afterArapModel = arapModel;
            RngRapParamJsonModel paramJsonModel = __createRapParamJsonModel(
                        afterArapModel.getParamJson().toString());
            if (paramJsonModel != null && paramJsonModel.getBodyParamList() != null) {
                paramJsonModel.setBodyParamList(__paramUpdateConfirmFile(
                            paramJsonModel.getBodyParamList(), action));
                JSONObject json = JSONObject.fromObject(__createParamJson(paramJsonModel));
                afterArapModel.setParamJson(json);
            }
            afterArapModelList.add(afterArapModel);
        }
        araModel.setActionParamList(afterArapModelList);
    }
    /**
     * <br>[機  能] 確認時添付を使用している箇所を初期化する
     * <br>[解  説]
     * <br>[備  考]
     * @param paramModelList 該当パラメータのパラメータモデルリスト
     * @param action 操作
     * @return 更新後のアクションパラメータモデル
     */
    public List<RngRapParamModel> __paramUpdateConfirmFile(
                    List<RngRapParamModel> paramModelList,
                    int action)
                    throws IOToolsException, JsonProcessingException, SQLException {
        for (RngRapParamModel paramModel : paramModelList) {
            //パラメータリスト
            TreeMap<Integer, ParamInfoModel> paramInfoMap = updateParamConfirmFileMap(
                                                                action,
                                                                paramModel.getParamInfoMap(),
                                                                false);
            paramModel.setParamInfoMap(paramInfoMap);
            //配列パラメータリスト
            int keyIndex = 0;
            TreeMap<Integer, ListParamInfoModel> listParamInfoMap = new TreeMap<>();
            for (Entry<Integer, ListParamInfoModel> entry
                        : paramModel.getListParamInfoMap().entrySet()) {
                //1つの配列内にあるパラメータリスト
                paramInfoMap = updateParamConfirmFileMap(
                                    action, entry.getValue().getParamInfoMap(), true);

                ListParamInfoModel listParamInfoModel = new ListParamInfoModel();

                listParamInfoModel.setParamInfoMap(paramInfoMap);
                listParamInfoMap.put(keyIndex, listParamInfoModel);
                keyIndex++;
            }
            paramModel.setListParamInfoMap(listParamInfoMap);
        }
        return paramModelList;
    }

    /**
     * <br>[機  能] 確認時添付を使用している箇所を初期化する
     * <br>[解  説]
     * <br>[備  考]
     * @param action 操作
     * @param treeMap パラメータマップ
     * @param listFlg 配列フラグ
     * @return 更新後のアクションパラメータモデル
     */
    public TreeMap<Integer, ParamInfoModel> updateParamConfirmFileMap(int action,
                    TreeMap<Integer, ParamInfoModel> treeMap, boolean listFlg)
                    throws IOToolsException, JsonProcessingException, SQLException {

        TreeMap<Integer, ParamInfoModel> paramInfoMap = new TreeMap<>();
        int keyIndex = 0;
        //パラメータリスト
        for (Entry<Integer, ParamInfoModel> entry
                    : treeMap.entrySet()) {
            ParamInfoModel paramInfoModel = entry.getValue();
            if (paramInfoModel.getParamKbn() == RngConst.API_PARAMKBN_RINGIINFO
                        && paramInfoModel.getParamValue() >= 403
                        && paramInfoModel.getParamValue() <= 405) {
                if (action == RngConst.API_FORMACTION_INIT_REPEATFILE
                        || listFlg) {
                    paramInfoModel.setParamValue(403);
                } else if (action == RngConst.API_FORMACTION_INIT_FILE) {
                    paramInfoModel.setParamValue(404);
                }
            }
            paramInfoMap.put(keyIndex, paramInfoModel);
            keyIndex++;
        }
        return paramInfoMap;
    }
}