package jp.groupsession.v2.rng.biz;

import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.struts.util.LabelValueBean;

import jp.co.sjts.util.NullDefault;
import jp.co.sjts.util.io.IOToolsException;
import jp.groupsession.v2.cmn.GSConst;
import jp.groupsession.v2.cmn.biz.AccessUrlBiz;
import jp.groupsession.v2.cmn.biz.CommonBiz;
import jp.groupsession.v2.cmn.biz.UserBiz;
import jp.groupsession.v2.cmn.biz.UserGroupSelectBiz;
import jp.groupsession.v2.cmn.dao.GroupDao;
import jp.groupsession.v2.cmn.dao.GroupModel;
import jp.groupsession.v2.cmn.dao.UsidSelectGrpNameDao;
import jp.groupsession.v2.cmn.dao.base.CmnBelongmDao;
import jp.groupsession.v2.cmn.dao.base.CmnGroupmDao;
import jp.groupsession.v2.cmn.dao.base.CmnPositionDao;
import jp.groupsession.v2.cmn.dao.base.CmnUsrmInfDao;
import jp.groupsession.v2.cmn.formbuilder.EnumFormModelKbn;
import jp.groupsession.v2.cmn.formbuilder.FormBuilder;
import jp.groupsession.v2.cmn.formbuilder.FormCell;
import jp.groupsession.v2.cmn.formmodel.Block;
import jp.groupsession.v2.cmn.formmodel.BlockList;
import jp.groupsession.v2.cmn.model.RequestModel;
import jp.groupsession.v2.cmn.model.base.CmnGroupClassModel;
import jp.groupsession.v2.cmn.model.base.CmnGroupmModel;
import jp.groupsession.v2.cmn.model.base.CmnPositionModel;
import jp.groupsession.v2.cmn.model.base.CmnUsrmInfModel;
import jp.groupsession.v2.rng.RngConst;
import jp.groupsession.v2.rng.RtpNotfoundException;
import jp.groupsession.v2.rng.dao.RngTemplateCategoryDao;
import jp.groupsession.v2.rng.dao.RngTemplateDao;
import jp.groupsession.v2.rng.dao.RngTemplateKeiroDao;
import jp.groupsession.v2.rng.dao.RngTemplatecategoryAdmDao;
import jp.groupsession.v2.rng.dao.RngTemplatecategoryUseDao;
import jp.groupsession.v2.rng.model.AddRngActionParamModel;
import jp.groupsession.v2.rng.model.RngAconfModel;
import jp.groupsession.v2.rng.model.RngRapConditionJsonModel;
import jp.groupsession.v2.rng.model.RngRapConditionModel;
import jp.groupsession.v2.rng.model.RngTemplateCategoryModel;
import jp.groupsession.v2.rng.model.RngTemplateKeiroModel;
import jp.groupsession.v2.rng.model.RngTemplateModel;
import jp.groupsession.v2.rng.model.RngTemplatecategoryAdmModel;
import jp.groupsession.v2.rng.rng020.Rng020Keiro;
import jp.groupsession.v2.rng.rng020.Rng020KeiroBlock;
import jp.groupsession.v2.rng.rng110keiro.EnumKeiroKbn;
import jp.groupsession.v2.rng.rng110keiro.Rng110KeiroDialogForm.TargetPosSel;
import jp.groupsession.v2.rng.rng110keiro.Rng110KeiroDialogParamModel;
import jp.groupsession.v2.struts.msg.GsMessage;

/**
 * <br>[機  能] 稟議のテンプレート機能で使用するビジネスロジッククラス
 * <br>[解  説]
 * <br>[備  考]
 *
 * @author JTS
 */
public class RngTemplateBiz {

    /**
     * <br>[機  能] 指定したテンプレートSIDのテンプレート情報を返します
     * <br>[解  説]
     * <br>[備  考]
     * @param tplSid テンプレートSID
     * @param con Connection
     * @return テンプレートモデル
     * @throws SQLException SQL実行時例外
     */
    public RngTemplateModel getRtpModel(int tplSid, Connection con) throws SQLException {
        RngTemplateDao dao = new RngTemplateDao(con);
        RngTemplateModel rtModel = new RngTemplateModel();

        rtModel = dao.select(tplSid);
        return rtModel;
    }

    /**
     * <br>[機  能]経路情報からその経路にユーザが存在するかの判定を行う
     * <br>[解  説]
     * <br>[備  考]
     * @param keiro Rng110KeiroDialogParamModel
     * @param con Connection
     * @param reqMdl リクエストモデル
     * @return true:ユーザが存在する false:ユーザが存在しない
     * @throws SQLException SQLException
     */
    public boolean keiroUserCheck(Rng110KeiroDialogParamModel keiro,
            Connection con,
            RequestModel reqMdl)
            throws SQLException {

        boolean ret = false;
        ArrayList<Integer> userList = new ArrayList<Integer>();
        RngBiz rngBiz = new RngBiz(con);
        switch (keiro.getKeiroKbn()) {
            case EnumKeiroKbn.FREESET_VAL:
                ret = true;
                break;
            case EnumKeiroKbn.USERTARGET_VAL:
                UserGroupSelectBiz tUgsBiz = new UserGroupSelectBiz();
                ArrayList<String> tSids = new ArrayList<String>();
                for (String[] value : keiro.getUsrgrouptarget().getSelected().values()) {
                    for (int idx = 0; idx < value.length; idx++) {
                        tSids.add(value[idx]);
                    }
                }
                userList = tUgsBiz.getSelectedUserLabel(
                    tSids.toArray(new String[tSids.size()]), con);
                if (userList != null && userList.size() > 0) {
                    if (rngBiz.getUserList(userList).size() > 0) {
                        return true;
                    }
                }
                break;
            case EnumKeiroKbn.USERSEL_VAL:
                ArrayList<String> sSids = new ArrayList<String>();
                for (String[] value : keiro.getUsrgroupselect().getSelected().values()) {
                    for (int idx = 0; idx < value.length; idx++) {
                        sSids.add(value[idx]);
                    }
                }
                if (sSids.size() == 0) {
                    return true;
                }
                UserGroupSelectBiz sUgsBiz = new UserGroupSelectBiz();
                userList = sUgsBiz.getSelectedUserLabel(
                    sSids.toArray(new String[sSids.size()]), con);
                if (userList != null && userList.size() > 0) {
                    List<Integer> useSelList = rngBiz.getUserList(userList);
                    if (useSelList.size() > 0) {
                        if (keiro.getKeiroKbn() == EnumKeiroKbn.USERSEL_VAL
                                && useSelList.size() == 1
                                && useSelList.get(0) == reqMdl.getSmodel().getUsrsid()
                                && keiro.getOwn() == RngConst.RNG_OWNSINGI_NO) {
                            return false;
                        }
                        return true;
                    }
                }
                break;
            case EnumKeiroKbn.GROUPSEL_VAL:
                ArrayList<Integer> nGrpSids = new ArrayList<Integer>();
                for (String value : keiro.getGroupSel().getSelected()) {
                    nGrpSids.add(Integer.parseInt(value));
                }
                if (nGrpSids.size() == 0) {
                    return true;
                }
                userList = (ArrayList<Integer>) rngBiz.getUserList(nGrpSids, null);
                if (userList != null && userList.size() > 0) {
                    List<Integer> useSelList = rngBiz.getUserList(userList);
                    if (useSelList.size() > 0) {
                        if (keiro.getKeiroKbn() == EnumKeiroKbn.USERSEL_VAL
                                && useSelList.size() == 1
                                && useSelList.get(0) == reqMdl.getSmodel().getUsrsid()
                                && keiro.getOwn() == RngConst.RNG_OWNSINGI_NO) {
                            return false;
                        }
                        return true;
                    }
                }
                break;
            case EnumKeiroKbn.POSTARGET_VAL:
                for (TargetPosSel pos : keiro.getTargetposMap().values()) {
                    int grpSid = -1;
                    if (pos.getGrpSel().getSelected() != null) {
                        grpSid = Integer.parseInt(pos.getGrpSel().getSelected());
                    }
                    if (pos.getPosSel().getSelected() != null) {
                        int posSid = NullDefault.getInt(pos.getPosSel().getSelected(), 0);
                        CmnUsrmInfDao usrDao = new CmnUsrmInfDao(con);
                        ArrayList<Integer> usrSids
                            = usrDao.getBelongUsrsFromPosition(grpSid, posSid);
                        if (rngBiz.getUserList(usrSids).size() > 0) {
                            return true;
                        }
                    }
                }
                break;
            case EnumKeiroKbn.BOSSTARGET_VAL:
                ret = true;
                break;
            default:
                break;

        }
        return ret;
    }

    /**
     * <br>[機  能] 実行条件文字列を取得
     * <br>[解  説]
     * <br>[備  考]
     * @param rapConditionMdl 実行条件モデル
     * @param rngTemplateJson JSON文字列
     * @param reqMdl リクエストモデル
     * @param con コネクション
     * @return 実行条件文字列
     * @throws SQLException
     */
    public String getConditionStr(
        RngRapConditionModel rapConditionMdl,
        String rngTemplateJson,
        RequestModel reqMdl,
        Connection con) throws SQLException {

        List<RngRapConditionModel> rapConditionMdlList = new ArrayList<RngRapConditionModel>();
        rapConditionMdlList.add(rapConditionMdl);
        String ret = "";
        List<String> conditionList
            = getConditionStrList(
                rapConditionMdlList,
                rngTemplateJson,
                true, reqMdl, con);
        if (conditionList != null && conditionList.size() > 0) {
            ret = conditionList.get(0);
        }
        return ret;
    }

    /**
     * <br>[機  能] 実行条件文字列リストを取得
     * <br>[解  説]
     * <br>[備  考]
     * @param rapConditionMdlList 実行条件モデルリスト
     * @param rngTemplateJson JSON文字列
     * @param errorFlg エラーフラグ（true:不備がある実行条件を含む, false:含まない）
     * @param reqMdl リクエストモデル
     * @param con コネクション
     * @return 実行条件文字列リスト
     * @throws SQLException
     */
    public List<String> getConditionStrList(
        List<RngRapConditionModel> rapConditionMdlList,
        String rngTemplateJson,
        boolean errorFlg,
        RequestModel reqMdl,
        Connection con) throws SQLException {

        GsMessage gsMsg = new GsMessage(reqMdl);
        List<String> ret = new ArrayList<String>();
        List<RngRapConditionModel> conditionList = new ArrayList<RngRapConditionModel>();
        if (errorFlg) {
            conditionList = rapConditionMdlList;
        } else {
            conditionList
                = rapConditionMdlList.stream()
                    .filter(c -> !c.isErrorFlg())
                    .collect(Collectors.toList());
        }

        for (RngRapConditionModel rngRapConditionMdl : conditionList) {

            //条件対象
            String paramKbn = "";
            switch (rngRapConditionMdl.getParamKbn()) {
                case RngConst.API_PARAMKBN_FORM:
                    if (rngRapConditionMdl.isErrorFlg()) {
                        //実行条件に不備がある場合
                        paramKbn = rngRapConditionMdl.getParamFormName()
                            + " ${" + rngRapConditionMdl.getParamFormId() + "}";
                    } else {
                        FormBuilder formBuilder = new FormBuilder();
                        formBuilder.setFormTable(rngTemplateJson);
                        List<FormCell> cellList = formBuilder.getFormCellList();
                        for (FormCell cell : cellList) {
                            if (rngRapConditionMdl.getParamFormId().equals(cell.getFormID())) {
                                paramKbn = cell.getTitle() + " ${" + cell.getFormID() + "}";
                                break;
                            }
                        }
                    }
                    break;
                case RngConst.API_PARAMKBN_ADDUSER:
                    paramKbn = gsMsg.getMessage("rng.47");
                    break;
                case RngConst.API_PARAMKBN_ADDDATE:
                    paramKbn = gsMsg.getMessage("rng.application.date");
                    break;
                case RngConst.API_PARAMKBN_LETUSER:
                    paramKbn = gsMsg.getMessage("rng.rng330.2");
                    break;
                case RngConst.API_PARAMKBN_LETDATE:
                    paramKbn = gsMsg.getMessage("rng.rng330.3");
                    break;
                case RngConst.API_PARAMKBN_RINGIINFO:
                    paramKbn = gsMsg.getMessage("main.useddisk.rng.rndata");
                    break;
                default:
                    break;
            }

            //設定値詳細
            String paramValue = " ";
            switch (rngRapConditionMdl.getParamValue()) {
                case RngConst.API_COMPARE_PARAM_NOSELECT:
                    paramValue = "";
                    break;
                case RngConst.API_COMPARE_PARAM_POSITION:
                    paramValue += gsMsg.getMessage("cmn.post");
                    break;
                case RngConst.API_COMPARE_PARAM_TITLE:
                    paramValue += gsMsg.getMessage("cmn.title");
                    break;
                case RngConst.API_COMPARE_PARAM_SINSEIID:
                    paramValue += gsMsg.getMessage("rng.rng180.04");
                    break;
                case RngConst.API_COMPARE_PARAM_FILE:
                    paramValue += gsMsg.getMessage("rng.rng330.4");
                    break;
                default:
                    paramValue = "";
                    break;
            }

            //比較文字列
            String compareTarget = "";
            if (rngRapConditionMdl.getParamKbn() == RngConst.API_PARAMKBN_FORM) {
                if (rngRapConditionMdl.isErrorFlg()) {
                    //実行条件に不備がある場合
                    if (rngRapConditionMdl.getParamFormType()
                        == EnumFormModelKbn.user.getValue()) {
                        //ユーザ選択
                        if (rngRapConditionMdl.getErrorTargetName() != null
                            && rngRapConditionMdl.getErrorTargetName().length() > 0) {
                            //ユーザまたは役職が削除済みの場合
                            compareTarget += " [" + rngRapConditionMdl.getErrorTargetName() + "]";
                        } else {
                            compareTarget += " [" + __getCompareTargetForUser(
                                rngRapConditionMdl.getParamValue(),
                                rngRapConditionMdl.getCompareTarget(), con) + "]";
                        }
                    } else if (rngRapConditionMdl.getParamFormType()
                        == EnumFormModelKbn.group.getValue()) {
                        //グループ選択
                        if (rngRapConditionMdl.getErrorTargetName() != null
                            && rngRapConditionMdl.getErrorTargetName().length() > 0) {
                            //グループが削除済みの場合
                            compareTarget += " [" + rngRapConditionMdl.getErrorTargetName() + "]";
                        } else {
                            CmnGroupmDao cgmDao = new CmnGroupmDao(con);
                            CmnGroupmModel cgmMdl
                                = cgmDao.select(
                                    NullDefault.getInt(
                                        rngRapConditionMdl.getCompareTarget(), -1));
                            compareTarget += " [" + cgmMdl.getGrpName() + "]";
                        }
                    } else if (rngRapConditionMdl.getParamFormType()
                        != EnumFormModelKbn.file.getValue()) {
                        //ファイル以外
                        compareTarget += " [" + rngRapConditionMdl.getCompareTarget() + "]";
                    }
                } else {
                    FormBuilder formBuilder = new FormBuilder();
                    formBuilder.setFormTable(rngTemplateJson);
                    for (FormCell cell : formBuilder.getFormCellList()) {
                        if (rngRapConditionMdl.getParamFormId().equals(cell.getFormID())) {
                            if (cell.getType() == EnumFormModelKbn.user) {
                                //ユーザ選択
                                compareTarget += " [" + __getCompareTargetForUser(
                                    rngRapConditionMdl.getParamValue(),
                                    rngRapConditionMdl.getCompareTarget(), con) + "]";
                            } else if (cell.getType() == EnumFormModelKbn.group) {
                                //グループ選択
                                CmnGroupmDao cgmDao = new CmnGroupmDao(con);
                                CmnGroupmModel cgmMdl
                                    = cgmDao.select(
                                        NullDefault.getInt(
                                            rngRapConditionMdl.getCompareTarget(), -1));
                                compareTarget += " [" + cgmMdl.getGrpName() + "]";
                            } else if (cell.getType() != EnumFormModelKbn.file) {
                                //ファイル以外
                                compareTarget += " [" + rngRapConditionMdl.getCompareTarget() + "]";
                            }
                        }
                    }
                }
            } else if (rngRapConditionMdl.getParamKbn() == RngConst.API_PARAMKBN_ADDUSER
                || rngRapConditionMdl.getParamKbn() == RngConst.API_PARAMKBN_LETUSER) {
                //申請者、最終承認者
                if (rngRapConditionMdl.isErrorFlg()
                    && rngRapConditionMdl.getErrorTargetName() != null
                    && rngRapConditionMdl.getErrorTargetName().length() > 0) {
                    //ユーザまたは役職が削除済みの場合
                    compareTarget += " [" + rngRapConditionMdl.getErrorTargetName() + "]";
                } else {
                    compareTarget += " [" + __getCompareTargetForUser(
                        rngRapConditionMdl.getParamValue(),
                        rngRapConditionMdl.getCompareTarget(), con) + "]";
                }
            } else if (rngRapConditionMdl.getParamValue() != RngConst.API_COMPARE_PARAM_FILE) {
                compareTarget += " [" + rngRapConditionMdl.getCompareTarget() + "]";
            }

            //比較条件
            String compareType = " ";
            switch (rngRapConditionMdl.getCompareType()) {
                case RngConst.API_COMPARE_CONDITION_EQUAL:
                    compareType += gsMsg.getMessage("rng.rng310.57");
                    break;
                case RngConst.API_COMPARE_CONDITION_NOTEQUAL:
                    compareType += gsMsg.getMessage("rng.rng310.58");
                    break;
                case RngConst.API_COMPARE_CONDITION_CONTAIN:
                    compareType += gsMsg.getMessage("rng.rng310.60");
                    break;
                case RngConst.API_COMPARE_CONDITION_NOTCONTAIN:
                    compareType += gsMsg.getMessage("rng.rng310.61");
                    break;
                case RngConst.API_COMPARE_CONDITION_MORETHAN:
                    compareType += gsMsg.getMessage("rng.rng310.59");
                    break;
                case RngConst.API_COMPARE_CONDITION_MORE:
                    compareType += gsMsg.getMessage("rng.rng310.65");
                    break;
                case RngConst.API_COMPARE_CONDITION_LESS:
                    compareType += gsMsg.getMessage("rng.rng310.66");
                    break;
                case RngConst.API_COMPARE_CONDITION_LESSTHAN:
                    compareType += gsMsg.getMessage("rng.rng310.67");
                    break;
                case RngConst.API_COMPARE_CONDITION_EXISTFILE:
                    compareType += gsMsg.getMessage("rng.rng310.62");
                    break;
                case RngConst.API_COMPARE_CONDITION_NOTEXISTFILE:
                    compareType += gsMsg.getMessage("rng.rng310.63");
                    break;
                default:
                    compareType = "";
                    break;
            }
            String conditionStr
                = paramKbn
                + paramValue
                + " " + gsMsg.getMessage("rng.rng310.68")
                + compareTarget
                + compareType;
            ret.add(conditionStr);
        }
        return ret;
    }

    /**
     * <br>[機  能] ユーザ選択時の比較文字列を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param paramValue 設定値詳細
     * @param compareTarget 比較文字列
     * @param con コネクション
     * @return 比較文字列
     * @throws SQLException
     */
    private String __getCompareTargetForUser(
        int paramValue,
        String compareTarget,
        Connection con) throws SQLException {

        String ret = null;
        if (paramValue == RngConst.API_COMPARE_PARAM_NOSELECT) {
            //未選択
            CmnUsrmInfDao cuiDao = new CmnUsrmInfDao(con);
            CmnUsrmInfModel cuiMdl
                = cuiDao.select(NullDefault.getInt(compareTarget, -1));
            if (cuiMdl != null) {
                ret = cuiMdl.getUsiName();
            }
        } else if (paramValue == RngConst.API_COMPARE_PARAM_POSITION) {
            //役職
            CmnPositionDao cpsDao = new CmnPositionDao(con);
            CmnPositionModel cpsMdl
                = cpsDao.getPosInfo(NullDefault.getInt(compareTarget, -1));
            if (cpsMdl != null) {
                ret = cpsMdl.getPosName();
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 既存の実行条件を複写コンボを作成
     * <br>[解  説]
     * <br>[備  考]
     * @param arapMdlList アクションパラメータモデルリスト
     * @param actionIndex 決裁後アクションインデックス
     * @param actionParamIndex アクションパラメータインデックス
     * @param template FormBuilder
     * @param templateJson JSON文字列
     * @param reqMdl リクエストモデル
     * @param con コネクション
     * @return 既存の実行条件を複写コンボ
     * @throws IOToolsException
     * @throws SQLException
     */
    public List<LabelValueBean> createCopyConditionCombo(
             List<AddRngActionParamModel> arapMdlList,
             int actionIndex,
             int actionParamIndex,
             FormBuilder template,
             String templateJson,
             RequestModel reqMdl,
             Connection con) throws IOToolsException, SQLException {

        RngTemplateActionFileBiz fileBiz = new RngTemplateActionFileBiz(reqMdl);
        GsMessage gsMsg = new GsMessage(reqMdl);
        List<LabelValueBean> ret = new ArrayList<LabelValueBean>();
        ret.add(new LabelValueBean(gsMsg.getMessage("rng.rng310.64"), "-1"));
        ret.add(new LabelValueBean(gsMsg.getMessage("cmn.select.plz"), "0"));
        for (AddRngActionParamModel arapMdlForLoop : arapMdlList) {
            if (arapMdlForLoop.getIndex() == actionParamIndex) {
                continue;
            }
            //実行条件モデルを取得
            RngRapConditionJsonModel rapConditionJsonMdlForLoop
                = fileBiz.getConditionJsonModel(
                    actionIndex,
                    arapMdlForLoop.getIndex());
            if (rapConditionJsonMdlForLoop == null) {
                continue;
            }

            List<String> compareTargetList = new ArrayList<>();
            List<String> labelList = new ArrayList<>();
            int conditionIndex = 0;
            for (RngRapConditionModel mdl : rapConditionJsonMdlForLoop.getConditionList()) {
                //実行条件文字列リストを取得
                List<RngRapConditionModel> conditionMdlList = new ArrayList<>();
                conditionMdlList.add(mdl);
                List<String> conditionList = getConditionStrList(
                        conditionMdlList, templateJson, false, reqMdl, con);
                for (String conditionStr : conditionList) {
                    if (!labelList.contains(conditionStr)
                        || !compareTargetList.contains(mdl.getCompareTarget())) {
                        ret.add(
                            new LabelValueBean(
                                conditionStr,
                                String.valueOf(arapMdlForLoop.getIndex())
                                +  "." + String.valueOf(conditionIndex)));
                        labelList.add(conditionStr);
                        compareTargetList.add(mdl.getCompareTarget());
                    }
                    conditionIndex++;
                }
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 表要素のボディ内の要素リストを取得
     * <br>[解  説]
     * <br>[備  考]
     * @param templateJson JSON文字列
     * @return 表要素ボディ内の要素リスト
     * @throws IOToolsException
     * @throws SQLException
     */
    public List<FormCell> getFormCellFromBlockListBody(
        String templateJson) {

        FormBuilder formBuilder = new FormBuilder();
        formBuilder.setFormTable(templateJson);
        List<FormCell> cellList = formBuilder.getFormCellList();

        //表要素ボディ内の要素を取得
        List<FormCell> ret = new ArrayList<FormCell>();
        for (FormCell cell : cellList) {
            if (cell.getType() == EnumFormModelKbn.blocklist) {
                for (Block block : ((BlockList) cell.getBody()).getBodyList()) {
                    for (List<FormCell> bodyFormCellList : block.getFormTable()) {
                        for (FormCell bodyFormCell : bodyFormCellList) {
                            ret.add(bodyFormCell);
                        }
                    }
                }
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 指定した表要素のボディ内の要素リストを取得
     * <br>[解  説]
     * <br>[備  考]
     * @param formId フォームID
     * @param templateJson JSON文字列
     * @return 表要素ボディ内の要素リスト
     * @throws IOToolsException
     * @throws SQLException
     */
    public List<FormCell> getFormCellFromBlockListBody(
        String formId,
        String templateJson) {

        FormBuilder formBuilder = new FormBuilder();
        formBuilder.setFormTable(templateJson);
        List<FormCell> cellList = formBuilder.getFormCellList();

        //表要素ボディ内の要素を取得
        List<FormCell> ret = new ArrayList<FormCell>();
        for (FormCell cell : cellList) {
            if (cell.getFormID().equals(formId)
                && cell.getType() == EnumFormModelKbn.blocklist) {
                for (Block block : ((BlockList) cell.getBody()).getBodyList()) {
                    for (List<FormCell> bodyFormCellList : block.getFormTable()) {
                        for (FormCell bodyFormCell : bodyFormCellList) {
                            ret.add(bodyFormCell);
                        }
                    }
                }
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 指定した設定値詳細の文字列を返す
     * <br>[解  説]
     * <br>[備  考]
     * @param paramValue 設定値詳細
     * @param reqMdl リクエストモデル
     * @return 設定値詳細文字列
     */
    public String getParamValueStr(int paramValue, RequestModel reqMdl) {

        GsMessage gsMsg = new GsMessage(reqMdl);
        switch (paramValue) {
            case RngConst.API_PARAMVALUE_USER_SID:
                return gsMsg.getMessage("cmn.user.sid");
            case RngConst.API_PARAMVALUE_USER_ID:
                return gsMsg.getMessage("cmn.user.id");
            case RngConst.API_PARAMVALUE_USER_SEI:
                return gsMsg.getMessage("rng.rng090.26");
            case RngConst.API_PARAMVALUE_USER_MEI:
                return gsMsg.getMessage("rng.rng090.27");
            case RngConst.API_PARAMVALUE_USER_SEIMEI:
                return gsMsg.getMessage("rng.rng090.25");
            case RngConst.API_PARAMVALUE_USER_SEI_KANA:
                return gsMsg.getMessage("rng.rng090.29");
            case RngConst.API_PARAMVALUE_USER_MEI_KANA:
                return gsMsg.getMessage("rng.rng090.30");
            case RngConst.API_PARAMVALUE_USER_SEIMEI_KANA:
                return gsMsg.getMessage("rng.rng090.28");
            case RngConst.API_PARAMVALUE_SYAIN_NO:
                return gsMsg.getMessage("cmn.employee.staff.number");
            case RngConst.API_PARAMVALUE_BELONG_GROUP_SID:
                return gsMsg.getMessage("cmn.belong.group.sid");
            case RngConst.API_PARAMVALUE_BELONG_GROUP_ID:
                return gsMsg.getMessage("cmn.belong.group.id");
            case RngConst.API_PARAMVALUE_BELONG_GROUP_NAME:
                return gsMsg.getMessage("cmn.belong.group.name");
            case RngConst.API_PARAMVALUE_BELONG_GROUP_NAME_KANA:
                return gsMsg.getMessage("cmn.belong.group.name.kana");
            case RngConst.API_PARAMVALUE_MAILADDRESS1:
                return gsMsg.getMessage("cmn.mailaddress1");
            case RngConst.API_PARAMVALUE_MAILADDRESS2:
                return gsMsg.getMessage("cmn.mailaddress2");
            case RngConst.API_PARAMVALUE_MAILADDRESS3:
                return gsMsg.getMessage("cmn.mailaddress3");
            case RngConst.API_PARAMVALUE_GROUP_SID:
                return gsMsg.getMessage("cmn.group.sid");
            case RngConst.API_PARAMVALUE_GROUP_ID:
                return gsMsg.getMessage("cmn.group.id");
            case RngConst.API_PARAMVALUE_GROUP_NAME:
                return gsMsg.getMessage("cmn.group.name");
            case RngConst.API_PARAMVALUE_GROUP_NAME_KANA:
                return gsMsg.getMessage("cmn.group.name.kana");
            case RngConst.API_PARAMVALUE_FULLDATE_SLASH:
                return gsMsg.getMessage("cmn.format.date.time.slash");
            case RngConst.API_PARAMVALUE_FULLDATE_HYPHEN:
                return gsMsg.getMessage("ccmn.format.date.time.hyphen");
            case RngConst.API_PARAMVALUE_FULLDATE_TEXT:
                return gsMsg.getMessage("cmn.format.date.time.text");
            case RngConst.API_PARAMVALUE_DATE_SLASH:
                return gsMsg.getMessage("cmn.format.date.slash");
            case RngConst.API_PARAMVALUE_DATE_HEPHEN:
                return gsMsg.getMessage("cmn.format.date.hyphen");
            case RngConst.API_PARAMVALUE_DATE_TEXT:
                return gsMsg.getMessage("cmn.format.date.text");
            case RngConst.API_PARAMVALUE_FULLTIME_COLON:
                return gsMsg.getMessage("cmn.format.time.second.colon");
            case RngConst.API_PARAMVALUE_FULLTIME_TEXT:
                return gsMsg.getMessage("cmn.format.time.second.text");
            case RngConst.API_PARAMVALUE_HHMM_COLON:
                return gsMsg.getMessage("cmn.format.time.colon");
            case RngConst.API_PARAMVALUE_HHMM_TEXT:
                return gsMsg.getMessage("cmn.format.time.text");
            case RngConst.API_PARAMVALUE_YEAR:
                return gsMsg.getMessage("cmn.format.year");
            case RngConst.API_PARAMVALUE_MONTH:
                return gsMsg.getMessage("cmn.format.month");
            case RngConst.API_PARAMVALUE_DAY:
                return gsMsg.getMessage("cmn.format.day");
            case RngConst.API_PARAMVALUE_HOUR:
                return gsMsg.getMessage("cmn.format.time2");
            case RngConst.API_PARAMVALUE_MINUTE:
                return gsMsg.getMessage("cmn.format.minutes");
            case RngConst.API_PARAMVALUE_SECOND:
                return gsMsg.getMessage("cmn.format.second");
            default:
                return "";
        }
    }

    /**
     * <br>[機  能] 実行条件JSONモデルにフォーム情報をセットする
     * <br>[解  説]
     * <br>[備  考]
     * @param rapConditionMdlList 実行条件モデルリスト
     * @param templateJson 申請内容JSON文字列
     * @return 設定値リスト
     * @throws SQLException
     */
    public List<RngRapConditionModel> setFormInfoToCondition(
        List<RngRapConditionModel> rapConditionMdlList,
        String templateJson) throws SQLException {

        FormBuilder formBuilder = new FormBuilder();
        formBuilder.setFormTable(templateJson);
        List<FormCell> cellList = formBuilder.getFormCellList();
        for (RngRapConditionModel rapConditionMdl : rapConditionMdlList) {
            if (rapConditionMdl.getParamKbn() != RngConst.API_PARAMKBN_FORM) {
                continue;
            } else if (rapConditionMdl.getParamKbn() == RngConst.API_PARAMKBN_FORM) {
                //フォーム要素
                for (FormCell cell : cellList) {
                    if (cell.getFormID().equals(rapConditionMdl.getParamFormId())) {
                        rapConditionMdl.setParamFormName(cell.getTitle());
                        rapConditionMdl.setParamFormType(cell.getType().getValue());
                        break;
                    }
                }
            }
        }
        return rapConditionMdlList;
    }

    /**
     * <br>[機  能] 実行条件の比較文字列に紐づくグループ情報をセットする
     * <br>[解  説]
     * <br>[備  考]
     * @param rapConditionMdl RngRapConditionModel
     * @param con コネクション
     * @throws SQLException
     */
    public void setGroupInfo(
        RngRapConditionModel rapConditionMdl,
        Connection con) throws SQLException {

        GroupDao grpDao = new GroupDao(con);
        CmnGroupmModel grpMdl
            = grpDao.getDefaultGroup(
                NullDefault.getInt(rapConditionMdl.getCompareTarget(), -1));
        if (grpMdl != null) {
            rapConditionMdl.setCompareTargetGroupSid(grpMdl.getGrpSid());
        }
    }

    /**
     * <br>[機  能] 指定したカテゴリの管理者を取得する
     * <br>[解  説]
     * <br>[備  考] キー：ユーザSID, 値：管理者として設定されている共有テンプレートのカテゴリSID一覧を返す
     * @param con コネクション
     * @param categorySidList カテゴリSID一覧
     * @param removeUserList 除外するユーザ
     * @return キー：ユーザSID, 値：管理者として設定されている共有テンプレートのカテゴリSID一覧を格納したマップ
     * @throws SQLException
     */
    public Map<Integer, Set<Integer>> getUserCategoryMap(
        Connection con, List<Integer> categorySidList,
        List<Integer> removeUserList) throws SQLException {

        Map<Integer, Set<Integer>> userCategoryMap = new HashMap<>();
        if (categorySidList == null || categorySidList.isEmpty()) {
            return userCategoryMap;
        }

        categorySidList = categorySidList.stream()
            .distinct()
            .collect(Collectors.toList());
        RngTemplatecategoryAdmDao rtcaDao = new RngTemplatecategoryAdmDao(con);
        ArrayList<RngTemplatecategoryAdmModel> rtcaList = new ArrayList<>();
        for (int sid : categorySidList) {
            rtcaList.addAll(rtcaDao.select(sid));
        }

        //ユーザ指定部分を取得 (キー：ユーザSID，値：カテゴリSID一覧)
        userCategoryMap = rtcaList.stream()
            .filter(mdl -> !removeUserList.contains(mdl.getUsrSid()))
            .filter(mdl -> mdl.getUsrSid() != -1)
            .collect(Collectors.groupingBy(
                RngTemplatecategoryAdmModel::getUsrSid,
                Collectors.mapping(RngTemplatecategoryAdmModel::getRtcSid, Collectors.toSet())
            ));

        //グループ指定部分を取得
        UserBiz userBiz = new UserBiz();
        for (RngTemplatecategoryAdmModel mdl : rtcaList) {
            if (mdl.getGrpSid() == -1) {
                continue;
            }
            int categorySid = mdl.getRtcSid();
            List<CmnUsrmInfModel> belongList
                = userBiz.getBelongUserList(con, mdl.getGrpSid(), new ArrayList<>());

            List<Integer> usrSidList = belongList.stream()
                .map(CmnUsrmInfModel::getUsrSid)
                .filter(sid -> !removeUserList.contains(sid))
                .collect(Collectors.toList());

            for (int usrSid : usrSidList) {
                Set<Integer> categorySet = userCategoryMap.getOrDefault(usrSid, new HashSet<>());
                categorySet.add(categorySid);
                userCategoryMap.put(usrSid, categorySet);
            }
        }
        return userCategoryMap;
    }

    /**
     * <br>[機  能] 役職，ユーザ，グループが削除された時のアクションパラメータ使用停止通知の本文を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param reqMdl リクエスト情報
     * @param templateModelList 通知対象テンプレート一覧
     * @param isAdmin true:システム管理者，またはプラグイン管理者である, false:システム管理者，プラグイン管理者ではない
     * @return アクションパラメータ使用停止通知の本文
     * @throws URISyntaxException
     * @throws UnsupportedEncodingException
     */
    public String getTemplateSmailTuutiBody(
        RequestModel reqMdl,
        List<RngTemplateModel> templateModelList, boolean isAdmin)
            throws URISyntaxException, UnsupportedEncodingException {

        GsMessage gsMsg = new GsMessage(reqMdl);
        StringBuilder sb = new StringBuilder();
        sb.append("◎Group Session " + gsMsg.getMessage("rng.131"));
        sb.append("\r\n\r\n");
        sb.append(gsMsg.getMessage("rng.130"));
        sb.append("\r\n");
        sb.append("┏━");
        sb.append("\r\n");
        for (RngTemplateModel mdl : templateModelList) {
            sb.append("┃");
            sb.append(mdl.getRtpTitle());
            sb.append("\r\n");
        }
        sb.append("┗━");
        sb.append("\r\n\r\n");

        sb.append(gsMsg.getMessage("cmn.url") + ":");
        AccessUrlBiz urlBiz = AccessUrlBiz.getInstance();
        StringBuilder urlSb = new StringBuilder();
        urlSb.append("/" + urlBiz.getContextPath(reqMdl));
        if (templateModelList.size() == 1) {
            RngTemplateModel mdl = templateModelList.get(0);
            urlSb.append("/ringi/rng090.do?");
            urlSb.append("CMD=060title");
            urlSb.append("&rngSelectTplSid=" + mdl.getRtpSid());
            urlSb.append("&rngTplCmdMode=1");
            urlSb.append("&catSid=0");
            urlSb.append("&rngTemplateMode=1");
            if (!isAdmin) {
                urlSb.append("&rng010TransitionFlg=1");
            }
            sb.append(urlBiz.getAccessUrl(reqMdl, urlSb.toString()));
        } else {
            urlSb.append("/ringi/rng060.do?");
            urlSb.append("CMD=ringi060");
            urlSb.append("&rngTemplateMode=1");
            if (!isAdmin) {
                urlSb.append("&rng010TransitionFlg=1");
            }
            sb.append(urlBiz.getAccessUrl(reqMdl, urlSb.toString()));
        }

        return sb.toString();
    }

    /**
     *
     * <br>[機  能] テンプレートが申請目的でアクセス可能か判定します
     * <br>[解  説]
     * <br>[備  考]
     * @param rtpMdl テンプレート情報
     * @param reqMdl リクエストモデル
     * @param con コネクション
     * @return 判定結果（true:アクセス可能, fasle:アクセス不可能）
     * @throws SQLException SQL実行時例外
     */
    public boolean isUseableTemplate(
        RngTemplateModel rtpMdl,
        RequestModel reqMdl,
        Connection con) throws SQLException {

        //稟議テンプレートが存在しない場合、アクセス不可
        if (rtpMdl == null) {
            return false;
        }

        RngBiz biz = new RngBiz(con);
        RngAconfModel aconfMdl = biz.getRngAconf(con);

        //汎用稟議テンプレート
        //管理者設定 基本設定 汎用稟議テンプレートが「使用しない」に設定されている場合、アクセス不可
        if (rtpMdl.getRtpSid() <= 0
            && aconfMdl.getRarHanyoFlg() == RngConst.RAR_HANYO_FLG_NO) {
            return false;
        }

        //個人テンプレート
        if (rtpMdl.getRtpType() == RngConst.RNG_TEMPLATE_PRIVATE) {

            //管理者設定 基本設定 汎用稟議テンプレートが「使用しない」に設定されている場合、アクセス不可
            if (aconfMdl.getRarHanyoFlg() == RngConst.RAR_HANYO_FLG_NO) {
                return false;
            }

            //管理者設定 基本設定 個人テンプレートが「使用しない」に設定されている場合、アクセス不可
            if (aconfMdl.getRarTemplatePersonalFlg() == RngConst.RAR_TEMPLATE_PERSONAL_FLG_NO) {
                return false;
            }

            //セッションユーザ以外の個人テンプレートの場合、アクセス不可
            if (rtpMdl.getUsrSid() != reqMdl.getSmodel().getUsrsid()) {
                return false;
            }
        }

        //テンプレート削除確認
        if (rtpMdl.getRtpJkbn() == RngConst.JKBN_DELETE) {
            return false;
        }
        int usrSid = reqMdl.getSmodel().getUsrsid();

        // 共有テンプレートカテゴリのみ使用権限チェックを行う
        int rtcSid = rtpMdl.getRtcSid();
        RngTemplateCategoryDao rtcDao = new RngTemplateCategoryDao(con);
        RngTemplateCategoryModel rtcMdl = rtcDao.select(rtcSid);
        if (rtcMdl == null) {
            return false;
        }

        // アクセス権限設定の使用判定
        if (rtcMdl.getRtcUseLimit() != RngConst.LIMIT_USE) {
            return true; // アクセス権限設定を使用しない場合、権限チェックなし
        }

        // カテゴリ管理者権限判定
        RngTemplatecategoryAdmDao rtcAdmDao = new RngTemplatecategoryAdmDao(con);
        boolean isAdmSid = rtcAdmDao.isAdmin(rtpMdl.getRtcSid(), usrSid);
        if (isAdmSid) {
            return true; // カテゴリ管理者は権限あり
        }

        // 許可or制限のあるカテゴリSID一覧を取得
        RngTemplatecategoryUseDao rtcUseDao = new RngTemplatecategoryUseDao(con);
        boolean isUseSid = rtcUseDao.isSelectedSid(rtcSid, usrSid);
        if (rtcMdl.getRtcLimitType() == RngConst.LIMIT_TYPE_LIMIT) {
            return !isUseSid; // 制限設定＋指定ユーザ一覧にない場合、権限あり
        } else if (rtcMdl.getRtcLimitType() == RngConst.LIMIT_TYPE_ACCEPT) {
            return isUseSid;  // 許可設定＋指定ユーザ一覧にある場合、権限あり
        }
        return false;
    }

    /**
     * <br>[機  能] 稟議テンプレートが指定した経路と紐づいているか判定する
     * <br>[解  説]
     * <br>[備  考]
     * @param templateMdl 稟議テンプレートモデル
     * @param stepSid 稟議テンプレート経路SID
     * @param con コネクション
     * @return 判定結果（true:紐づいている, fasle:紐づいていない）
     * @throws SQLException SQL実行時例外
     */
    public boolean isRelatedToKeiro(
        RngTemplateModel templateMdl,
        int stepSid,
        Connection con) throws SQLException {

        //稟議テンプレートが存在しない
        if (templateMdl == null) {
            return false;
        }

        //稟議テンプレートが論理削除済み
        if (templateMdl.getRtpJkbn() == GSConst.JTKBN_DELETE) {
            return false;
        }

        //経路情報を取得
        RngTemplateKeiroDao rtkDao = new RngTemplateKeiroDao(con);
        RngTemplateKeiroModel rtkMdl = rtkDao.select(stepSid);

        //経路情報が存在しない
        if (rtkMdl == null) {
            return false;
        }

        //経路情報が論理削除済み
        if (rtkMdl.getRtkJkbn() == GSConst.JTKBN_DELETE) {
            return false;
        }

        if (templateMdl.getRctSid() > 0) {
            //稟議テンプレートが経路テンプレートと紐づく場合

            //経路情報とテンプレートが紐づいていない
            if (rtkMdl.getRctSid() != templateMdl.getRctSid()) {
                return false;
            }
        } else {
            //経路情報とテンプレートが紐づいていない
            if (rtkMdl.getRtpSid() != templateMdl.getRtpSid()) {
                return false;
            }

            //経路情報に紐づくテンプレートのバージョンが一致しない
            if (rtkMdl.getRtpVer() != templateMdl.getRtpVer()) {
                return false;
            }
        }
        return true;
    }

    /**
     * <br>[機  能] 経路情報から上長指定に選択できるグループSIDを取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param rtkMdl 稟議テンプレート経路情報モデル
     * @param usrSid セッションユーザSID
     * @param con コネクション
     * @return グループSIDリスト
     * @throws SQLException SQL実行時例外
     */
    public List<Integer> getBossTargetGroup(
        RngTemplateKeiroModel rtkMdl,
        int usrSid,
        Connection con) throws SQLException {

        List<Integer> ret = new ArrayList<Integer>();

        //経路情報を取得
        int needStep = rtkMdl.getRtkBossstepMastcnt();
        int autoStep = rtkMdl.getRtkBossstepCnt();
        int threshold = 1;
        if (rtkMdl.getRtkOutcondition() == RngConst.RNG_OUT_CONDITION_NUMBER) {
            threshold = rtkMdl.getRtkOutcondBorder();
        }

        //セッションユーザが所属するグループの階層情報を取得
        UsidSelectGrpNameDao gpDao = new UsidSelectGrpNameDao(con);
        List<CmnGroupClassModel> belongGrpClsList = gpDao.selectBelongGroupClass(usrSid);

        for (CmnGroupClassModel cgcMdl : belongGrpClsList) {

            //グループSIDとグループ管理者数の関連MAP
            Map<Integer, Integer> bossCntMap = new HashMap<Integer, Integer>();
            //上位階層グループSIDリスト（自身が所属するグループを含む）
            List<Integer> higherClassGrpList = new ArrayList<Integer>();
            //判定対象グループ
            GroupModel grpMdl = new GroupModel();
            //グループ情報をセット
            setGroupInfo(cgcMdl, grpMdl, higherClassGrpList);

            for (int grpSid : higherClassGrpList) {
                // グループ管理者ユーザを取得
                CmnBelongmDao begDao = new CmnBelongmDao(con);
                List<Integer> bossList = begDao.selectBossModel(grpSid).stream()
                                            .map(mdl -> mdl.getUsrSid())
                                            .collect(Collectors.toList());
                //プラグイン使用禁止ユーザを除外
                CommonBiz cmnBiz = new CommonBiz();
                List<Integer> cantUseSidList =
                        cmnBiz.getCantUsePluginUser(
                                con, RngConst.PLUGIN_ID_RINGI, bossList);
                bossList.removeAll(cantUseSidList);
                int bossCnt = bossList.size();
                bossCntMap.put(grpSid, bossCnt);

                //グループの階層が必須上長階層数より小さい場合、選択不可
                if (grpMdl.getClassLevel() < needStep) {
                    continue;
                }

                //グループの管理者が存在しない場合、選択不可
                if (!existsHigherClassGrpBoss(needStep, autoStep, bossCntMap, higherClassGrpList)) {
                    continue;
                }

                //グループの管理者数が承認者数に満たない場合、選択不可
                if (!isOverThreshould(threshold, autoStep, bossCntMap, higherClassGrpList)) {
                    continue;
                }
                ret.add(grpMdl.getGroupSid());
            }
        }
        return ret;
    }

    /**
     * <br>[機  能] 指定した上長階層グループに管理者が存在するか判定する
     * <br>[解  説]
     * <br>[備  考]
     * @param needStep 必須上長階層数
     * @param autoCnt 自動設定する上長階層数
     * @param bossCntMap グループSIDとグループ管理者数の関連MAP
     * @param higherClassGrpList 上位階層グループSIDリスト（自身が所属するグループを含む）
     * @return 判定結果（true:管理者が存在する, false:管理者が存在しない）
     */
    public boolean existsHigherClassGrpBoss(
        int needStep,
        int autoCnt,
        Map<Integer, Integer> bossCntMap,
        List<Integer> higherClassGrpList) {

        if (bossCntMap == null) {
            return true;
        }
        if (needStep > higherClassGrpList.size()) {
            return false;
        }
        int roop = autoCnt;
        if (autoCnt > higherClassGrpList.size()) {
            roop = higherClassGrpList.size();
        }
        for (int i = 0; i < roop; i++) {
            int grpSid = higherClassGrpList.get(i);
            if (bossCntMap.containsKey(grpSid)
                    && bossCntMap.get(grpSid) > 0) {
                return true;
            }
        }
        return false;
    }

    /**
    * <br>[機  能] 指定した上長階層グループの管理者数が必要な承認者数を満たしているか判定する
    * <br>[解  説]
    * <br>[備  考]
    * @param threshould 承認数（承認条件が「承認数」の場合は閾値、それ以外の場合は1）
    * @param autoCnt 自動設定する上長階層数
    * @param bossCntMap グループSIDとグループ管理者数の関連MAP
    * @param higherClassGrpList 上位階層グループSIDリスト（自身が所属するグループを含む）
    * @return 判定結果（true:承認者数が足りている, false:承認者数が足りていない）
    */
    public boolean isOverThreshould(
        int threshould,
        int autoCnt,
        Map<Integer, Integer> bossCntMap,
        List<Integer> higherClassGrpList) {

        int roop = autoCnt;
        if (autoCnt > higherClassGrpList.size()) {
            roop = higherClassGrpList.size();
        }
        for (int i = 0; i < roop; i++) {
            int grpSid = higherClassGrpList.get(i);
            if (bossCntMap.containsKey(grpSid)) {
                if (bossCntMap.get(grpSid) >= threshould) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * <br>[機  能] グループ階層モデルからグループモデル、上位階層グループSIDリストをセットする
     * <br>[解  説]
     * <br>[備  考]
     * @param gclMdl グループ階層モデル
     * @param grpMdl グループモデル
     * @param higherClassGrpList 上位階層グループSIDリスト
     */
    public void setGroupInfo(
        CmnGroupClassModel gclMdl,
        GroupModel grpMdl,
        List<Integer> higherClassGrpList) {

        int classNum = -1;
        if (gclMdl.getGclSid10() > 0) {
            if (classNum == -1) {
                classNum = 10;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid10());
                grpMdl.setGroupId(gclMdl.getGclId10());
                grpMdl.setGroupName(gclMdl.getGclName10());
            }
            higherClassGrpList.add(gclMdl.getGclSid10());
        }
        if (gclMdl.getGclSid9() > 0) {
            if (classNum == -1) {
                classNum = 9;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid9());
                grpMdl.setGroupId(gclMdl.getGclId9());
                grpMdl.setGroupName(gclMdl.getGclName9());
            }
            higherClassGrpList.add(gclMdl.getGclSid9());
        }
        if (gclMdl.getGclSid8() > 0) {
            if (classNum == -1) {
                classNum = 8;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid8());
                grpMdl.setGroupId(gclMdl.getGclId8());
                grpMdl.setGroupName(gclMdl.getGclName8());
            }
            higherClassGrpList.add(gclMdl.getGclSid8());
        }
        if (gclMdl.getGclSid7() > 0) {
            if (classNum == -1) {
                classNum = 7;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid7());
                grpMdl.setGroupId(gclMdl.getGclId7());
                grpMdl.setGroupName(gclMdl.getGclName7());
            }
            higherClassGrpList.add(gclMdl.getGclSid7());
        }
        if (gclMdl.getGclSid6() > 0) {
            if (classNum == -1) {
                classNum = 6;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid6());
                grpMdl.setGroupId(gclMdl.getGclId6());
                grpMdl.setGroupName(gclMdl.getGclName6());
            }
            higherClassGrpList.add(gclMdl.getGclSid6());
        }
        if (gclMdl.getGclSid5() > 0) {
            if (classNum == -1) {
                classNum = 5;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid5());
                grpMdl.setGroupId(gclMdl.getGclId5());
                grpMdl.setGroupName(gclMdl.getGclName5());
            }
            higherClassGrpList.add(gclMdl.getGclSid5());
        }
        if (gclMdl.getGclSid4() > 0) {
            if (classNum == -1) {
                classNum = 4;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid4());
                grpMdl.setGroupId(gclMdl.getGclId4());
                grpMdl.setGroupName(gclMdl.getGclName4());
            }
            higherClassGrpList.add(gclMdl.getGclSid4());
        }
        if (gclMdl.getGclSid3() > 0) {
            if (classNum == -1) {
                classNum = 3;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid3());
                grpMdl.setGroupId(gclMdl.getGclId3());
                grpMdl.setGroupName(gclMdl.getGclName3());
            }
            higherClassGrpList.add(gclMdl.getGclSid3());
        }
        if (gclMdl.getGclSid2() > 0) {
            if (classNum == -1) {
                classNum = 2;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid2());
                grpMdl.setGroupId(gclMdl.getGclId2());
                grpMdl.setGroupName(gclMdl.getGclName2());
            }
            higherClassGrpList.add(gclMdl.getGclSid2());
        }
        if (gclMdl.getGclSid1() > 0) {
            if (classNum == -1) {
                classNum = 1;
                grpMdl.setClassLevel(classNum);
                grpMdl.setGroupSid(gclMdl.getGclSid1());
                grpMdl.setGroupId(gclMdl.getGclId1());
                grpMdl.setGroupName(gclMdl.getGclName1());
            }
            higherClassGrpList.add(gclMdl.getGclSid1());
        }
    }

    /**
     * @param keiroBlockMap
     * @param kakuninKeiroBlockMap
     * @param keiroTplMap
     * @param finalKeiroTplMap
     * @param first
     * @param isLoaded
     * @throws SQLException
     * @throws RtpNotfoundException
     */
    public void initKeiroTpl(
        Map<Integer, Rng020KeiroBlock> keiroBlockMap,
        Map<Integer, Rng020KeiroBlock> kakuninKeiroBlockMap,
        Map<Integer, Rng110KeiroDialogParamModel> keiroTplMap,
        Map<Integer, Rng110KeiroDialogParamModel> finalKeiroTplMap,
        boolean first,
        boolean isLoaded) throws SQLException, RtpNotfoundException {

        __initKeiroTpl(keiroBlockMap, keiroTplMap, first, isLoaded);

        __initKeiroTpl(kakuninKeiroBlockMap, finalKeiroTplMap, first, isLoaded);

        if (keiroBlockMap.size() == 0 && first) {
            Rng110KeiroDialogParamModel pref = new Rng110KeiroDialogParamModel();
            Rng020KeiroBlock keiroBlock = keiroBlockMap.get(0);
            if (keiroBlock == null) {
                keiroBlock = new Rng020KeiroBlock();
                keiroBlockMap.put(0, keiroBlock);
            }
            Rng020Keiro keiro = keiroBlock.getKeiroSingle();
            keiroBlock.setPref(pref);
            keiro.setPref(pref);
            keiro.setKeiroKbn(EnumKeiroKbn.FREESET_VAL);
            keiroBlock.setKeiroKbn(EnumKeiroKbn.FREESET_VAL);
        }
        if (kakuninKeiroBlockMap.size() == 0  && first) {
            Rng110KeiroDialogParamModel pref = new Rng110KeiroDialogParamModel();
            Rng020KeiroBlock keiroBlock = kakuninKeiroBlockMap.get(0);
            if (keiroBlock == null) {
                keiroBlock = new Rng020KeiroBlock();
                kakuninKeiroBlockMap.put(0, keiroBlock);
            }
            keiroBlock.setKeiroKbn(EnumKeiroKbn.FREESET_VAL);
            keiroBlock.setPref(pref);
        }
    }

    /**
     *
     * <br>[機  能] 経路テンプレート情報取得 サブロジック
     * <br>[解  説] 経路と最終確認経路の共通処理を抽出
     * <br>[備  考]
     * @param keiroBlockMap 経路ブロックマップ
     * @param keiroTplMap 経路テンプレート情報
     * @param first 初期化時かどうか
     * @param isLoaded 読込みデータフラグ
     * @throws SQLException SQLException
     */
    private void  __initKeiroTpl(
        Map<Integer, Rng020KeiroBlock> keiroBlockMap,
        Map<Integer, Rng110KeiroDialogParamModel> keiroTplMap,
        boolean first, boolean isLoaded) throws SQLException {

        if (keiroTplMap == null) {
            return;
        }
        int sortNo = 0;

        boolean isLoad = (first && !isLoaded); // テンプレートからの初期データ読込み判定

        Map<Integer, List<Rng020KeiroBlock>> appBlockMap =
                new HashMap<Integer, List<Rng020KeiroBlock>>();

        //経路設定が任意経路かつ設定が共通が連続する場合は経路ブロックをまとめる
        Rng020KeiroBlock prev = null;
        Rng110KeiroDialogParamModel  prevPref = null;
        int keiroKey = 0;
        for (Entry<Integer, Rng110KeiroDialogParamModel> entry : keiroTplMap.entrySet()) {
            Rng110KeiroDialogParamModel keiroPref = entry.getValue();
            Rng020Keiro keiro = null;
            Rng020KeiroBlock keiroBlock = null;
            if (prev != null && keiroPref.getKeiroKbn() == EnumKeiroKbn.FREESET_VAL
                && prevPref != null && prevPref.equalsKeiroPrefLessDefaultSelect(keiroPref)) {
                if (!isLoad) {
                    //初回表示時以降は2行目以降の初期化は不要
                    continue;
                }
                keiroKey++;
                keiro = prev.getKeiro(String.valueOf(keiroKey));
                keiroBlock = prev;
            } else {
                keiroBlock = keiroBlockMap.get(sortNo);
                if (keiroBlock == null) {
                    keiroBlock = new Rng020KeiroBlock();
                    keiroBlockMap.put(sortNo, keiroBlock);
                }
                if (first && keiroBlock.getRtkSid() > 0
                    && keiroBlock.getRtkSid() != keiroPref.getRtkSid()) {
                    // テンプレートにある＋元となる稟議情報にない経路ブロック ⇒ テンプレートの経路ブロック情報を追加
                    prev     = null;
                    prevPref = null;

                    // 追加される経路ブロック情報を作成
                    keiroBlock = new Rng020KeiroBlock();
                    keiroBlock.setRtkSid(keiroPref.getRtkSid());
                    keiroBlock.setPref(keiroPref);
                    keiroBlock.setKeiroKbn(keiroPref.getKeiroKbn());
                    keiroBlock.setInCond(keiroPref.getInCondMap());
                    keiroKey = 0;

                    if (keiroPref.getKeiroKbn() == EnumKeiroKbn.FREESET_VAL) {
                        // 任意経路の場合、空の枠を追加
                        prev     = keiroBlock;
                        prevPref = keiroPref;
                    } else {
                        // 任意経路以外の場合、テンプレートから初期状態の経路を追加
                        keiroBlock.getKeiroSingle(); //経路情報の1行目を生成
                        keiro = keiroBlock.getKeiro(String.valueOf(keiroKey));
                        keiro.setRtkSid(keiroPref.getRtkSid());
                        keiro.setPref(keiroPref);
                        keiro.initDefault();
                    }

                    // 追加される経路ブロック情報を一時的に配列へセット
                    List<Rng020KeiroBlock> blockList = appBlockMap.get(sortNo);
                    if (blockList == null) {
                        blockList = new ArrayList<Rng020KeiroBlock>();
                    }
                    blockList.add(keiroBlock);
                    appBlockMap.put(sortNo, blockList);
                    continue;
                }

                sortNo++;
                prev = keiroBlock;
                prevPref = keiroPref;
                keiroBlock.setKeiroKbn(keiroPref.getKeiroKbn());
                keiroBlock.setInCond(keiroPref.getInCondMap());
                keiroBlock.setRtkSid(keiroPref.getRtkSid());
                keiroBlock.setPref(keiroPref);
                keiroKey = 0;
                //初回表示時以降は任意経路の初期化は不要
                if (keiroPref.getKeiroKbn() != EnumKeiroKbn.FREESET_VAL || isLoad) {
                    //経路情報の1行目を生成
                    keiroBlock.getKeiroSingle();
                }
            }

            Map<Integer, Rng020Keiro> keiroMap = keiroBlock.getKeiroMap();
            if (keiroMap.containsKey(keiroKey)) {
                keiro = keiroBlock.getKeiro(String.valueOf(keiroKey));
            }
            if (keiro == null) {
                continue;
            }
            if (keiroPref.getKeiroKbn() != EnumKeiroKbn.FREESET_VAL || isLoad) {
                keiro.setRtkSid(keiroPref.getRtkSid());
                keiro.setPref(keiroPref);
                if (isLoad) {
                    keiro.initDefault();
                }
            }
        }
        if (appBlockMap.size() <= 0) {
            return;
        }

        // 追加経路(空の経路ブロック)がある場合、ソート番号を再セット
        if (keiroBlockMap != null && keiroBlockMap.size() > 0) {
            // 既存の経路データをソート番号を再セットした配列に入れ替える
            Map<Integer, Rng020KeiroBlock> checkBlockMap = new HashMap<Integer, Rng020KeiroBlock>();
            checkBlockMap.putAll(keiroBlockMap); // 既存の経路データをコピーする
            List<Integer> checkKeys = new ArrayList<Integer>(checkBlockMap.keySet());
            Collections.sort(checkKeys);         // ソート番号順に並び替える
            keiroBlockMap.clear();               // 経路データをソート番号を再セットする為、一旦初期化
            sortNo = 0;
            for (Integer key : checkKeys) {
                List<Rng020KeiroBlock> blockList = appBlockMap.get(key);
                if (blockList != null) {
                    for (Rng020KeiroBlock block : blockList) {
                        keiroBlockMap.put(Integer.valueOf(sortNo), block);
                        sortNo++;
                    }
                }
                Rng020KeiroBlock keiroBlock = checkBlockMap.get(key);
                if (keiroBlock != null) {
                    keiroBlockMap.put(Integer.valueOf(sortNo), keiroBlock);
                    sortNo++;
                }
            }
        }
        return;
    }
}