package jp.groupsession.v2.rng.restapi.entities.files;

import java.sql.SQLException;
import java.util.List;

import jp.co.sjts.util.date.UDate;
import jp.groupsession.v2.restapi.controller.RestApiContext;
import jp.groupsession.v2.restapi.exception.RestApiPermissionException;
import jp.groupsession.v2.restapi.exception.RestApiValidateException;
import jp.groupsession.v2.restapi.msg.RestApiGsMessage;
import jp.groupsession.v2.restapi.parameter.annotation.ParamModel;
import jp.groupsession.v2.restapi.parameter.annotation.Selectable;
import jp.groupsession.v2.restapi.parameter.annotation.Validator;
import jp.groupsession.v2.rng.RngConst;
import jp.groupsession.v2.rng.RtpNotfoundException;
import jp.groupsession.v2.rng.biz.RngFormBuildBiz;
import jp.groupsession.v2.rng.biz.RngTemplateBiz;
import jp.groupsession.v2.rng.dao.RngBinDao;
import jp.groupsession.v2.rng.dao.RngDairiUserDao;
import jp.groupsession.v2.rng.dao.RngRndataDao;
import jp.groupsession.v2.rng.model.RngRndataModel;
import jp.groupsession.v2.rng.model.RngTemplateModel;
import jp.groupsession.v2.rng.restapi.RngEnumReasonCode;
import jp.groupsession.v2.rng.rng030.Rng030Biz;

/**
 *
 * <br>[機  能] 稟議 添付ファイルダウンロードAPI パラメータモデル
 * <br>[解  説]
 * <br>[備  考]
 *
 * @author JTS
 */
@ParamModel
public class RngEntitiesFilesParamModel {

    /** クエリパラメータ */
    @Selectable({"0", "1"})
    private int mode__ = 1;


    /** 稟議SID */
    private int ringiSid__ = 0;
    /** バイナリSID */
    private int binSid__ = 0;
    /**
     * @return the ringiSid
     */
    public int getRingiSid() {
        return ringiSid__;
    }
    /**
     * @param ringiSid the ringiSid to set
     */
    public void setRingiSid(int ringiSid) {
        ringiSid__ = ringiSid;
    }
    /**
     * @return the binSid
     */
    public int getBinSid() {
        return binSid__;
    }
    /**
     * @param binSid the binSid to set
     */
    public void setBinSid(int binSid) {
        binSid__ = binSid;
    }

    /**
     * <br>[機  能] 入力チェックを実行する
     * <br>[解  説]
     * <br>[備  考]
     * @param ctx RestApiコンテキスト
     * @throws SQLException SQL実行時例外
     */
    @Validator
    public void validate(RestApiContext ctx) throws SQLException {

        /***** アクセス権限チェック *****/
        //[RINGI-102] 指定した稟議を参照可能かをチェック (受信していない稟議へのアクセス)
        RngRndataDao rndataDao = new RngRndataDao(ctx.getCon());
        RngRndataModel rngDataMdl = rndataDao.select(ringiSid__);
        if (rngDataMdl == null) {
            __throwRingiException(ctx);
        }

        boolean ringiResult = false;
        if (rngDataMdl.getRngStatus() == RngConst.RNG_STATUS_DRAFT) {
            int sessionUsrSid = ctx.getRequestUserSid();
            ringiResult = (rngDataMdl.getRngAuid() == sessionUsrSid);
        } else {
            ringiResult = __checkViewRingi(ctx);
        }

        if (!ringiResult) {
            __throwRingiException(ctx);
        }

        //[RINGI-103] 指定された添付ファイルをダウンロード可能かを判定
        boolean binResult = false;
        RngBinDao binDao = new RngBinDao(ctx.getCon());
        binResult = binDao.isCheckRngTemp(ringiSid__, Long.valueOf(binSid__));

        if (!binResult) {
            throw new RestApiPermissionException(
                RngEnumReasonCode.RESOURCE_CANT_ACCESS_FILE,
                    "error.no.access.data",
                    new RestApiGsMessage()
                        .getMessage("rng.rng030.05")
            );
        }

        /***** 実行権限チェック *****/
        if (rngDataMdl.getRngStatus() == RngConst.RNG_STATUS_DRAFT) {
            //草稿の場合はテンプレートの参照権限をチェック
            __checkAccessTemplate(ctx, rngDataMdl.getRtpSid());
        }

    }

    /**
     * <br>[機  能] 指定した稟議を参照可能かチェックする
     * <br>[解  説]
     * <br>[備  考]
     * @param ctx RestApiコンテキスト
     * @return true:参照可能 false:参照不可
     * @throws SQLException
     */
    private boolean __checkViewRingi(RestApiContext ctx) throws SQLException {

        boolean canViewRingi = false;

        //ユーザが利用可能なアカウントの一覧を取得
        int sessionUsrSid = ctx.getRequestUserSid();
        Rng030Biz biz030 = new Rng030Biz(ctx.getCon(), ctx.getRequestModel());
        RngDairiUserDao dairiDao = new RngDairiUserDao(ctx.getCon());
        UDate now = new UDate();
        List<Integer> accountSidList = dairiDao.getAvailableAccountList(sessionUsrSid, now);

        //利用可能なアカウントのいずれかで稟議を参照可能かをチェック
        for (int usrSid : accountSidList) {
            if (biz030.chkViewRingi(usrSid, ringiSid__)) {
                canViewRingi = true;
                break;
            }
        }

        return canViewRingi;
    }

    /**
     * <br>[機  能] 指定した草稿で使用されているテンプレートを参照可能かチェックする
     * <br>[解  説]
     * <br>[備  考]
     * @param ctx RestApiコンテキスト
     * @param templateSid テンプレートSID
     * @throws SQLException
     */
    private void __checkAccessTemplate(RestApiContext ctx, int templateSid) throws SQLException {

        RngFormBuildBiz formBiz = new RngFormBuildBiz(ctx.getRequestModel());
        RngTemplateModel rtpMdl = null;
        try {
            rtpMdl = formBiz.getRtpModelMaxVer(ctx.getCon(), templateSid);
        } catch (RtpNotfoundException e) {
            //テンプレートが存在しない
            __throwTemplateException(ctx);
        }

        RngTemplateBiz rtpBiz = new RngTemplateBiz();
        boolean canUseTemplate = rtpBiz.isUseableTemplate(
            rtpMdl, ctx.getRequestModel(), ctx.getCon());
        if (!canUseTemplate) {
            //テンプレートの使用権限がない
            __throwTemplateException(ctx);
        }
    }

    /**
     * <br>[機  能] 稟議，草稿のアクセス権限エラーをスローする
     * <br>[解  説]
     * <br>[備  考]
     * @param ctx RestApiコンテキスト
     * @throws SQLException
     */
    private void __throwRingiException(RestApiContext ctx) {

        throw new RestApiPermissionException(
            RngEnumReasonCode.RESOURCE_CANT_ACCESS_RINGI,
                "error.no.access.data",
                new RestApiGsMessage()
                    .getMessage("main.useddisk.rng.rndata")
        );
    }

    /**
     * <br>[機  能] 草稿のテンプレートをスローする
     * <br>[解  説]
     * <br>[備  考]
     * @param ctx RestApiコンテキスト
     * @throws SQLException
     */
    private void __throwTemplateException(RestApiContext ctx) {

        throw new RestApiValidateException(
            RngEnumReasonCode.PARAM_CANT_ACCESS_TEMPLATE,
                "restapi.error.form.ringi.access.template",
                "ringiSid"
        );
    }
}
