package jp.groupsession.v2.sml.sml290;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Objects;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import jp.co.sjts.util.jdbc.JDBCUtil;
import jp.groupsession.v2.cmn.GSConst;
import jp.groupsession.v2.cmn.biz.SortChangeBiz;
import jp.groupsession.v2.cmn.model.RequestModel;
import jp.groupsession.v2.sml.dao.SmlAccountDao;
import jp.groupsession.v2.sml.dao.SmlJmeisLabelDao;
import jp.groupsession.v2.sml.dao.SmlLabelDao;
import jp.groupsession.v2.sml.dao.SmlLabelSortDao;
import jp.groupsession.v2.sml.dao.SmlSmeisLabelDao;
import jp.groupsession.v2.sml.dao.SmlWmeisLabelDao;
import jp.groupsession.v2.sml.model.LabelDataModel;
import jp.groupsession.v2.sml.model.SmlAccountModel;
import jp.groupsession.v2.sml.model.SmlLabelModel;

/**
 * <br>[機  能] ショートメール 管理者設定 ラベル管理画面のビジネスロジッククラス
 * <br>[解  説]
 * <br>[備  考]
 *
 * @author JTS
 */
public class Sml290Biz {

    /** Logging インスタンス */
    private static Log log__ = LogFactory.getLog(Sml290Biz.class);

    /**
     * <br>[機  能] ラベルの削除を行う
     * <br>[解  説]
     * <br>[備  考]
     * @param con コネクション
     * @param labelSid 削除対象のラベルSID
     * @throws SQLException SQL実行時例外
     */
    public void deleteLabel(Connection con, int labelSid) throws SQLException {

        boolean commitFlg = false;

        try {
            //ラベルを削除する
            SmlLabelDao slDao = new SmlLabelDao(con);
            slDao.delete(labelSid);

            //メール - ラベルを削除する
            SmlJmeisLabelDao jDao = new SmlJmeisLabelDao(con);
            jDao.delete(labelSid);

            SmlSmeisLabelDao sDao = new SmlSmeisLabelDao(con);
            sDao.delete(labelSid);

            SmlWmeisLabelDao wDao = new SmlWmeisLabelDao(con);
            wDao.delete(labelSid);

            //メール情報表示順を削除する
            SmlLabelSortDao mailSortDao = new SmlLabelSortDao(con);
            mailSortDao.deleteMailSortOfLabel(labelSid);

            //コミット実行
            con.commit();
            commitFlg = true;
        } catch (SQLException e) {
            log__.error("SQLException", e);
            throw e;
        } finally {
            if (!commitFlg) {
                JDBCUtil.rollback(con);
            }
        }
    }

    /**
     * <br>[機  能] 初期表示を行う
     * <br>[解  説]
     * <br>[備  考]
     * @param con コネクション
     * @param paramMdl パラメータ情報
     * @param userSid ユーザSID
     * @param reqMdl リクエスト情報
     * @throws SQLException SQL実行時例外
     */
    public void setInitData(
        Connection con, Sml290ParamModel paramMdl,
        int userSid, RequestModel reqMdl) throws SQLException {

        __setAccountName(con, paramMdl);
        List<LabelDataModel> ldMdlList = __getLabelList(con, paramMdl.getSmlAccountSid());
        __setLbnList(paramMdl, ldMdlList);
        __setSortRadio(paramMdl, ldMdlList);
    }

    /**
     * <br>[機  能] 指定されたアカウントSIDのショートメールアカウント情報を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param con コネクション
     * @param sacSid ショートメールアカウントSID
     * @throws SQLException SQL実行時例外
     * @return SmlAccountModel ショートメールアカウント情報
     */
    private SmlAccountModel __getSmlAccountModel(Connection con, int sacSid) throws SQLException {
        SmlAccountDao sacDao = new SmlAccountDao(con);
        return sacDao.select(sacSid);
    }

    /**
     * <br>[機  能] 表示用のアカウント名を設定する
     * <br>[解  説]
     * <br>[備  考]
     * @param con コネクション
     * @param paramMdl ショートメールアカウントSID
     * @throws SQLException SQL実行時例外
     */
    private void __setAccountName(
        Connection con, Sml290ParamModel paramMdl) throws SQLException {

        //アカウント名取得
        int selectSacSid = paramMdl.getSmlAccountSid();
        SmlAccountModel sacMdl = __getSmlAccountModel(con, selectSacSid);
        paramMdl.setSml290accountName(sacMdl.getSacName());
    }

    /**
     * <br>[機  能] 指定されたアカウントのラベル一覧を取得する
     * <br>[解  説]
     * <br>[備  考]
     * @param con コネクション
     * @param sacSid ショートメールアカウントSID
     * @throws SQLException SQL実行時例外
     * @return ラベル一覧
     */
    private List<LabelDataModel> __getLabelList(Connection con, int sacSid) throws SQLException {
        Sml290Dao dao = new Sml290Dao(con);
        List<LabelDataModel> labelList = dao.getLabelList(sacSid);
        return labelList;
    }

    /**
     * <br>[機  能] 表示用のアカウント名を設定する
     * <br>[解  説]
     * <br>[備  考]
     * @param paramMdl ショートメールアカウントSID
     * @param lbList ラベル一覧
     * @throws SQLException SQL実行時例外
     */
    private void __setLbnList(
        Sml290ParamModel paramMdl, List<LabelDataModel> lbList) throws SQLException {

        Sml290DataOperator ope = new Sml290DataOperator();
        ope.setLbValue(lbList);

        paramMdl.setLbnList(lbList);
    }

    /**
     * <br>[機  能] 並び順が設定されていない場合に、並び順をセットする
     * <br>[解  説]
     * <br>[備  考]
     * @param paramMdl パラメータ情報
     * @param lbList ラベル一覧
     */
    private void __setSortRadio(Sml290ParamModel paramMdl, List<LabelDataModel> lbList) {
        Sml290DataOperator ope = new Sml290DataOperator();
        ope.setSortRadio(paramMdl, lbList);
    }

    /**
     * <br>[機  能] 順序変更処理
     * <br>[解  説]
     * <br>[備  考]
     * @param con コネクション
     * @param paramMdl パラメータ情報
     * @param changeKbn 処理区分 0:順序をあげる 1:順序を下げる
     * @throws SQLException SQL実行時例外
     */
    public void updateSort(Connection con, Sml290ParamModel paramMdl, int changeKbn)
        throws SQLException {

        boolean canUpdateSort = __canSortLabel(paramMdl);
        if (!canUpdateSort) {
            return;
        }

        SortChangeBiz<SmlLabelModel> sortBiz = __getSortBiz(paramMdl, con);

        __updateSort(sortBiz, changeKbn);
    }

    /**
     * <br>[機  能] ラベルの並び順を変更可能かどうかを判定する
     * <br>[解  説]
     * <br>[備  考]
     * @param paramMdl パラメータ情報
     * @return true:並び順を変更可能,false:並び順を変更不可
     */
    private boolean __canSortLabel(Sml290ParamModel paramMdl) {

        String[] keyList = paramMdl.getSml290sortLabel();
        String selectSid = paramMdl.getSml290SortRadio();
        Sml290DataOperator ope = new Sml290DataOperator();
        boolean ret = ope.canUpdateSort(keyList, selectSid);

        return ret;
    }

    /**
     * <br>[機  能] 並び順が設定されていない場合に、並び順をセットする
     * <br>[解  説]
     * <br>[備  考]
     * @param paramMdl パラメータ情報
     * @param con コネクション
     * @return SortChangeBiz<SmlLabelModel> 並び順変更ビジネスロジック
     */
    private SortChangeBiz<SmlLabelModel> __getSortBiz(Sml290ParamModel paramMdl, Connection con) {

        int motoSid = Integer.parseInt(paramMdl.getSml290SortRadio());
        SmlLabelDao labelDao = new SmlLabelDao(con);
        SortChangeBiz<SmlLabelModel> sortBiz =
            SortChangeBiz.<SmlLabelModel> builder()
            .setFuncTargetList(() -> {
                return labelDao.getLabelList(paramMdl.getSmlAccountSid());
            })
            .setFuncIsSelected(m -> {
                return (Objects.equals(m.getSlbSid(), motoSid));
            })
            .setFuncGetOrderNo(m -> {
                return (int) m.getSlbOrder();
            })
            .setFuncExeComparater((m1, m2) -> {
                if (m1.getSlbSid() == m2.getSlbSid()) {
                    return 0;
                } else {
                    return (m1.getSlbSid() - m2.getSlbSid())
                            / Math.abs(m1.getSlbSid() - m2.getSlbSid());
                }
            })
            .setFuncUpdateSort((m, newSort) -> {
                //並び替え更新実行 ラムダ関数
                labelDao.updateOrder(m.getSlbSid(), newSort);
            })
            .build();
        return sortBiz;
    }

    /**
     * <br>[機  能] ラベルのソートを実行する
     * <br>[解  説]
     * <br>[備  考]
     * @param sortBiz 並び順変更ビジネスロジック
     * @param changeKbn ソート順変更区分
     * @throws SQLException
     */
    private void __updateSort(
        SortChangeBiz<SmlLabelModel> sortBiz, int changeKbn) throws SQLException {

        if (changeKbn == GSConst.SORT_UP) {
            sortBiz.up();
        } else if (changeKbn == GSConst.SORT_DOWN) {
            sortBiz.down();
        }
    }
}
